#include "stdafx.h"

#include <stdlib.h>

#include "matrix.h"
#include "pin.h"
#include "gate.h"
#include "metaverse.h"

// ---------------------------------------------------------------------------------------
// CGate::CGate( const char *pstrName )
//
// Construct a gate with name pstrName
CGate::CGate( const char *pstrName )
: m_pParentCircuit( NULL )
{
    m_pstrName = new char[ strlen( pstrName ) + 1 ];
    strcpy( m_pstrName, pstrName );
}

// CGate::~CGate()
//
// Destroy the gate. Free the copied string
CGate::~CGate()
{
    if( m_pstrName ){
        delete [] m_pstrName;
        m_pstrName = NULL;
    }
}

// void CGate::SetOutputBitNumbers()
//
// Copy the bit numbers from the corresponding input pins to the output pins
void CGate::SetOutputBitNumbers()
{
    if( InputPins() == OutputPins() ){
        for( int i = 0; i < InputPins(); i++ ){
            OutputPin( i )->BitNumber() = InputPin( i )->BitNumber();
        }

    }
}

// int CGate::GateNumber() const
//
// Return our gate number, i.e. our index in m_pParentCircuit->m_listGates
int CGate::GateNumber() const
{
    if( m_pParentCircuit )
        return m_pParentCircuit->FindGate( this );

    return -1;
}

// ostream& operator<<( ostream &os, const CGate &gate )
//
// Dump this gate to os for debugging purposes
ostream& operator<<( ostream &os, const CGate &gate )
{
    os << "Gate " << gate.GateNumber() << " (" << gate.Name() << ") " /*<< &gate*/ << "\n";

    for( int i = 0; i < gate.InputPins(); i++ ){
        os << "  Input " << i;

        if( gate.InputPin( i )->IsConnected() ){
            COutputPin *pOther = dynamic_cast<COutputPin*>(gate.InputPin( i )->ConnectedTo());

            os << " (bit " << gate.InputPin( i )->BitNumber() << ") "
               << " connected to " << pOther->ParentGate()->GateNumber() 
               << " pin " << pOther->ParentGate()->GetPinNumber( pOther )
               << "\n";
        }else
            os << " not connected\n";
    }

    for( i = 0; i < gate.OutputPins(); i++ ){
        os << "  Output " << i;

        if( gate.OutputPin( i )->IsConnected() ){
            CInputPin *pOther = dynamic_cast<CInputPin*>(gate.OutputPin( i )->ConnectedTo());

            os << " (bit " << gate.OutputPin( i )->BitNumber() << ") "
               << " connected to " << pOther->ParentGate()->GateNumber() 
               << " pin " << pOther->ParentGate()->GetPinNumber( pOther )
               << "\n";
        }else
            os << " not connected\n";
    }

    return os;
}

// ---------------------------------------------------------------------------------------
// CSourceGate::CSourceGate( const char *pstrName )
//
// Source gate. Data enters the system through here. Exposes one output pin
CSourceGate::CSourceGate( const char *pstrName )
 : CGate( pstrName )
 , m_pOutputPin( new COutputPin )
{ 
    m_pOutputPin->SetParent( this ); 
}

// ---------------------------------------------------------------------------------------
// CSinkGate::CSinkGate( const char *pstrName )
//
// Sink Gate. Data leaves the system through here. Exposes one input pin
CSinkGate::CSinkGate( const char *pstrName )
: CGate( pstrName )
, m_pInputPin( new CInputPin ) 
 { 
    m_pInputPin->SetParent(this); 
}

// ---------------------------------------------------------------------------------------
// CInputOutputGate::CInputOutputGate( const char *pstrName, int iPins )
//
// Input / Output gate. Exposes an equal and non-zero number of input and
// output pins
CInputOutputGate::CInputOutputGate( const char *pstrName, int iPins )
: CGate( pstrName )
, m_iPins( iPins )
{
    ASSERT( m_iPins > 0 );

    // These vectors are used as temporary workspace when simulating
    m_cvInput.SetLength( pow2( iPins ) );
    m_cvOutput.SetLength( pow2( iPins ) );

    // Create the pins
    m_pInputPin = new CInputPin[ m_iPins ];
    m_pOutputPin = new COutputPin[ m_iPins ];

    // set the parent of the pins.
    for( int i = 0; i < m_iPins; i++ ){
        m_pInputPin[ i ].SetParent( this );
        m_pOutputPin[ i ].SetParent( this );
    }

    // For each PinTemplate[i] we well eventually set an integer template which will
    // allows us to quickly find which complex numbers are to be operated on during
    // one pass of the simulation
    m_piPinTemplate = new int[ pow2( iPins ) ];
}

// CInputOutputGate::~CInputOutputGate( )
//
// Destroy the gate. Free allocated memory
CInputOutputGate::~CInputOutputGate( )
{
    if( m_pInputPin ){
        delete [] m_pInputPin;
        m_pInputPin = NULL;
    }

    if( m_pOutputPin ){
        delete [] m_pOutputPin;
        m_pOutputPin = NULL;
    }

    if( m_piPinTemplate ){
        delete [] m_piPinTemplate;
        m_piPinTemplate = NULL;
    }
}

// CComplexVector CInputOutputGate::CalcOutput( const CComplexVector *pInputVector ) const
//
// Calculate the output from this gate given an input of pInputVector. We multiply our
// gate martix by the vector in order to calculate the output.
CComplexVector CInputOutputGate::CalcOutput( const CComplexVector *pInputVector ) const
{ 
    ASSERT( pInputVector );
    return m_GateMatrix * (*pInputVector);
}

// void CInputOutputGate::CalcOutput( CComplexVector& ret, const CComplexVector *pInputVector ) const
//
// Calculate the output from this gate given an input of pInputVector, returning the
// result in ret. This saves the copying required which the above method requires
// to return the vector on the stack
void CInputOutputGate::CalcOutput( CComplexVector& ret, const CComplexVector *pInputVector ) const
{ 
    ASSERT( pInputVector );
    m_GateMatrix.Multiply( ret, *pInputVector );
}

// void CInputOutputGate::CalcOutput( CBaseComplexVector& cvAmps, int iBaseAmplitude )
//
// Calculate the output from this gate given an input of cvAmps and a base amplitude.
// We work out the input to the gate by combining the base amplitide with the 
// pre-prepared templates.
void CInputOutputGate::CalcOutput( CBaseComplexVector& cvAmps
                                   , int iBaseAmplitude
                                   , CComplexVector &cvInWorkSpace
                                   , CComplexVector &cvOutWorkSpace
                                   )
{
    for( int iInputAmp = 0; iInputAmp < pow2( m_iPins ); iInputAmp++ ){
        cvInWorkSpace[ iInputAmp ] = cvAmps[ m_piPinTemplate[iInputAmp] | iBaseAmplitude ];
    }

    m_GateMatrix.Multiply( cvOutWorkSpace, cvInWorkSpace );

    for( int iOutputAmp = 0; iOutputAmp < pow2( m_iPins ); iOutputAmp ++ ){
       cvAmps.SetElement( m_piPinTemplate[iOutputAmp] | iBaseAmplitude, cvOutWorkSpace[ iOutputAmp ] );
    }
}

// BOOL CInputOutputGate::IsReversible() const
//
// Return true if this gate is reversible.
BOOL CInputOutputGate::IsReversible() const
{
    // Reversible gate iff the gate has a unitary matrix
    // Unitariness not valid for source or sink gates
    ASSERT( InputPins() > 0 );
    ASSERT( InputPins() == OutputPins() );

    return m_GateMatrix.IsUnitary();
}

// void CInputOutputGate::PrepareForProcessing()
//
// Prepare the gate's workspaces for processing.
void CInputOutputGate::PrepareForProcessing()
{
    ASSERT( m_piPinTemplate );

    // set m_piPinTemplate
    // A gate with n input pins requires an input vector of size 2**n. We work
    // out where each element of this vector comes from by counting from 0..2**n-1 in
    // binary using the bit numbers.
    //
    // To illustrate, suppose our input pins work on bits 6 and 13. We then set
    // m_piPinMask as follows
    //
    // input vector element -   in binary - set bits of m_piPinTemplate[i]
    //          0                   00                  none
    //          1                   01                  6
    //          2                   10                  13
    //          3                   11                  6 and 13


    for( int i = 0; i < pow2( m_iPins ); i++ ){
        m_piPinTemplate[i] = 0;

        for( int iPin = 0; iPin < m_iPins; iPin++ ){
            if( i & pow2( iPin ) ) 
                m_piPinTemplate[i] |= pow2( InputPin( iPin )->BitNumber() );
        }

    }

}

// ---------------------------------------------------------------------------------------
// CStaticSource::CStaticSource( const char *pstrName, BOOL bOutput )
//
// Create the source. It will output 1 only if bOutput is true
CStaticSource::CStaticSource( const char *pstrName, BOOL bOutput )
: CSourceGate( pstrName )
{
    int iOutput = bOutput ? 1 : 0;
    int iNoOutput = 1 - iOutput;

    m_Output.SetLength( 2 );

    m_Output[ iOutput ] = 1.0;          // Probability of outputting bOutput  is 1.0
    m_Output[ iNoOutput ] = 0;          // Probability of outputting !bOutput is 0.0
}

// CComplexVector CStaticSource::CalcOutput( const CComplexVector *pInputVector ) const
//
// Calculate the output to this gate given an input of pInputVector
CComplexVector CStaticSource::CalcOutput( const CComplexVector *pInputVector ) const
{ 
    // As a source gate we always output a static value.
    return m_Output; 
}


// CRandomisedSource::CRandomisedSource( const char *pstrName )
//
// Like a static source but the value is decided randomly upon creation
CRandomisedSource::CRandomisedSource( const char *pstrName )
: CStaticSource( pstrName, rand() % 2 )
{
}

// ---------------------------------------------------------------------------------------
// CSquareRootNOT::CSquareRootNOT( const char *pstrName )
//
// Square Root NOT Gate.
CSquareRootNOT::CSquareRootNOT( const char *pstrName )
: CInputOutputGate( pstrName, 1 )
{
    // 1 input gives 2x2 matrix
    m_GateMatrix.SetSize( 2, 2 );

    // set the elements.
    m_GateMatrix[0][0] = REC_SQRT2;
    m_GateMatrix[0][1] = - REC_SQRT2;
    m_GateMatrix[1][0] = REC_SQRT2;
    m_GateMatrix[1][1] = REC_SQRT2;

    // Make sure that we haven't erred.
    ASSERT( IsReversible() );
}

// CFlipGate::CFlipGate( const char *pstrName )
//
// Performs SQRTNOT on second input line iff first line is 1
CFlipGate::CFlipGate( const char *pstrName )
: CInputOutputGate( pstrName, 2 )
{
    // 2 inputs gives 4x4 matrix
    m_GateMatrix.SetSize( 4, 4 );
    

    // set the elements.
    m_GateMatrix[0][0] = 1;
    m_GateMatrix[0][1] = 0;
    m_GateMatrix[0][2] = 0;
    m_GateMatrix[0][3] = 0;

    m_GateMatrix[1][0] = 0;
    m_GateMatrix[1][1] = 1;
    m_GateMatrix[1][2] = 0;
    m_GateMatrix[1][3] = 0;

    m_GateMatrix[2][0] = 0;
    m_GateMatrix[2][1] = 0;
    m_GateMatrix[2][2] = REC_SQRT2;
    m_GateMatrix[2][3] = - REC_SQRT2;

    m_GateMatrix[3][0] = 0;
    m_GateMatrix[3][1] = 0;
    m_GateMatrix[3][2] = REC_SQRT2;
    m_GateMatrix[3][3] = REC_SQRT2;

    // Make sure that we haven't erred.
    ASSERT( IsReversible() );
}

// CMatrixGate::CMatrixGate( const char *pstrName, int iSize, CComplexMatrix &matrix )
//
// Create a gate from a specified matrix. For the gate to be valid then it should
// be reversible.
CMatrixGate::CMatrixGate( const char *pstrName, int iSize, CComplexMatrix &matrix )
: CInputOutputGate( pstrName, iSize )
{
    // Make sure that the matrix is of a correct size.
    ASSERT( matrix.Rows() == pow2( iSize ) );
    ASSERT( matrix.Columns() == pow2( iSize ) );

    // Set the matrix
    m_GateMatrix = matrix;

    // Make sure that it is reversible.s
    ASSERT( IsReversible() );
}


