
#ifndef _GATE_H_
#define _GATE_H_

#include "pin.h"
#include "matrix.h"
#include "circuit.h"

class CCircuit;

// CGate
//
// Generic gate. Does nothing. Not pure as it provides a default
// implementation
class CGate {
    CCircuit    *m_pParentCircuit;      // Ptr to the circuit in which we are wired
    char        *m_pstrName;            // Name of this circuit

    // construction / destruction
public:
    CGate( const char *pstrName );
    virtual ~CGate();

    // Number of Input / Output pins
    virtual int InputPins() const { return 0; }
    virtual int OutputPins() const { return 0; }

    // Returning of a specific input / output pin
    virtual COutputPin *OutputPin( int n ) const { return NULL; }
    virtual CInputPin *InputPin( int n ) const { return NULL; }

    // Find the index of a pin
    virtual int GetPinNumber( const COutputPin *pOutputPin ){ return -1; }
    virtual int GetPinNumber( const CInputPin *pOutputPin ){ return -1; }

    // Called to set the output pin bit numbers 
    virtual void SetOutputBitNumbers();

    // Called to calculate the output of the gate given certain inputs
    virtual CComplexVector CalcOutput( const CComplexVector *pInputVector ) const { 
        return CComplexVector(); 
    }
    virtual void CalcOutput( CComplexVector& ret, const CComplexVector *pInputVector ) const { 
        ret.Reset(); 
    }
    virtual void CalcOutput( CBaseComplexVector& cvAmps
                           , int iBaseAmplitude
                           , CComplexVector &cvInWorkSpace
                           , CComplexVector &cvOutWorkSpace
                           ){ }
    virtual void CalcOutput( CBaseComplexVector& cvAmps, int iBaseAmplitude ){ }

    // Prepares the gate for any processing which it may be required to perform
    virtual void PrepareForProcessing(){ }

    // Set / return the parent circuit
    CCircuit *ParentCircuit(){ return m_pParentCircuit; }
    void SetParentCircuit( CCircuit *pParentCircuit ){ 
        ASSERT( !m_pParentCircuit );
        m_pParentCircuit = pParentCircuit; 
    }

    // returns the name
    const char *Name() const { return m_pstrName; }

    // returns our index in the parent circuit. Mainly used for debugging
    int GateNumber() const;

    // output stream
    friend ostream& operator<<( ostream &os, const CGate &gate );
};

// CSourceGate
//
// Data enters the system through here. We have zero input pins and one
// output pin
class CSourceGate : public CGate {
    COutputPin *m_pOutputPin;

public:
    CSourceGate( const char *pstrName );
    virtual ~CSourceGate( ){ if( m_pOutputPin ) delete m_pOutputPin; }

    virtual int OutputPins() const { return 1; }
    virtual COutputPin *OutputPin( int n ) const { ASSERT( n == 0 ); return m_pOutputPin; }

    virtual int GetPinNumber( const COutputPin *pOutputPin ){
        return pOutputPin == m_pOutputPin ? 0 : -1;
    }
};

// CSinkGate
//
// Data exits the system through here. Has one input pins and no outputs
class CSinkGate : public CGate {
    CInputPin *m_pInputPin;

public:
    CSinkGate( const char *pstrName );
    virtual ~CSinkGate( ){ if( m_pInputPin ) delete m_pInputPin; }

    virtual int InputPins() const { return 1; }
    virtual CInputPin *InputPin( int n ) const { ASSERT( n == 0 ); return m_pInputPin; }

    virtual int GetPinNumber( const CInputPin *pInputPin ){
        return pInputPin == m_pInputPin ? 0 : -1;
    }
};

// CGate
//
// Generic input / output gate
class CInputOutputGate : public CGate {
protected:
    COutputPin      *m_pOutputPin;      // Array of output pins
    CInputPin       *m_pInputPin;       // Array of input pins
    const int       m_iPins;            // Number of input / output pins

    CComplexMatrix  m_GateMatrix;       // Matrix for this gate
    CComplexVector  m_cvInput;          // Temporary space for storing input vectors
    CComplexVector  m_cvOutput;         // Temporary space for storing output vectors
    int             *m_piPinTemplate;   // Bit template for each pin

public:
    CInputOutputGate( const char *pstrName, int iPins );
    ~CInputOutputGate();

    virtual int InputPins() const { return m_iPins; }
    virtual int OutputPins() const { return m_iPins; }

    virtual COutputPin *OutputPin( int n ) const { 
        ASSERT( n>=0 && n<m_iPins && m_pOutputPin ); 
        return &m_pOutputPin[n]; 
    }
    virtual CInputPin *InputPin( int n ) const { 
        ASSERT( n>=0 && n<m_iPins && m_pInputPin );
        return &m_pInputPin[n]; 
    }

    virtual void PrepareForProcessing();

    virtual int GetPinNumber( const COutputPin *pOutputPin ){
        for( int i = 0; i < m_iPins; i++ )
            if( pOutputPin == &m_pOutputPin[i] )
                return i;
        return -1;
    }

    virtual int GetPinNumber( const CInputPin *pInputPin ){
        for( int i = 0; i < m_iPins; i++ )
            if( pInputPin == &m_pInputPin[i] )
                return i;
        return -1;
    }

    // Is this gate reversible?
    BOOL IsReversible() const;

    virtual CComplexVector CalcOutput( const CComplexVector *pInputVector ) const;
    virtual void CalcOutput( CComplexVector& ret, const CComplexVector *pInputVector ) const;

    virtual void CalcOutput( CBaseComplexVector& cvAmps
                           , int iBaseAmplitude
                           , CComplexVector &cvInWorkSpace
                           , CComplexVector &cvOutWorkSpace
                           );

    virtual void CalcOutput( CBaseComplexVector& cvAmps, int iBaseAmplitude )
    { CalcOutput( cvAmps, iBaseAmplitude, m_cvInput, m_cvOutput ); }
};

// -- Non generic gates
// .. Sources

// CStaticSource
//
// Value of output bit is specified in construction
class CStaticSource : public CSourceGate {
    CComplexVector m_Output;
public:
    CStaticSource( const char *pstrName, BOOL bOutput );

    virtual CComplexVector CalcOutput( const CComplexVector *pInputVector ) const;
};

// CStaticSource
//
// Value of output bit is randomly decided in construction
class CRandomisedSource : public CStaticSource {
public:
    CRandomisedSource( const char *pstrName );
};

// CSquareRootNOT
//
// A gate which performs the 'Square Root of NOT' function. (two
// in series will invert a bit)
class CSquareRootNOT : public CInputOutputGate{
public:
    CSquareRootNOT( const char *pstrName );
};

// CSquareRootNOT
//
// A gate which performs the 'Square Root of NOT' function on its
// second input bit iff the first input bit is 1
class CFlipGate : public CInputOutputGate{
public:
    CFlipGate( const char *pstrName );
};

// CMatrixGate
//
// Generic gate whose matrix and size is specified in construction.
class CMatrixGate : public CInputOutputGate {
public:
    CMatrixGate( const char *pstrName, int iSize, CComplexMatrix &matrix );
};


#endif