// quantumDoc.cpp : implementation of the CQuantumDoc class
//

#include "stdafx.h"
#include <process.h>
#include "quantum.h"

#include "qexception.h"
#include "qlist.h"
#include "quantumDoc.h"
#include "matrix.h"
#include "metaverse.h"
#include "circuit.h"
#include "SimProgress.h"
#include "HaskellEvaluationDialog.h"
#include "FactorisationParams.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#ifdef _DEBUG
ostream *g_os;
#endif

/////////////////////////////////////////////////////////////////////////////
// CQuantumDoc

IMPLEMENT_DYNCREATE(CQuantumDoc, CDocument)

BEGIN_MESSAGE_MAP(CQuantumDoc, CDocument)
	//{{AFX_MSG_MAP(CQuantumDoc)
		// NOTE - the ClassWizard will add and remove mapping macros here.
		//    DO NOT EDIT what you see in these blocks of generated code!
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////

// qostrstream::qostrstrean
//
// Constucts an object identical to a strstream but with a minimum size on 
// the output buffer. This stops the system from having to reallocate the
// output buffer too often. ( the minimum size also acts as the size to
// grow the buffer by when it needs to be extended ).
qostrstream::qostrstream()
: ostream(new strstreambuf( 65536 ))
{
    delbuf(1);
}


/////////////////////////////////////////////////////////////////////////////

// CQuantumDoc::CQuantumDoc
CQuantumDoc::CQuantumDoc()
: m_pMetaverse( NULL )
, m_pCircuit( NULL )
, m_postrstream( NULL )
, m_pBuffer( NULL )
, m_SimulationFinished( FALSE, FALSE )
, m_iCalculationPriority( THREAD_PRIORITY_BELOW_NORMAL )
, m_pSimProgress( NULL )
, m_bCanRunSimulation( FALSE )
, m_bSimulateSimply( TRUE )
, m_iMaxThreads( 4 )
, m_bSparseVectors( FALSE )
{
    // Allocate an output stream
    m_postrstream = new qostrstream( );

    // Create the metaverse!
    m_pMetaverse = new CMetaverse( *m_postrstream );

    // Set a global (yuck) variable which holds points to our output stream
    // when in debug mode. This allows us to output debugging text to the
    // output window via the DBG_OUT macro.
#ifdef _DEBUG
    g_os = m_postrstream;
#endif
}


// CQuantumDoc::~CQuantumDoc
//
CQuantumDoc::~CQuantumDoc()
{
    // Big crunch time at the end of the metaverse!
    if( m_pMetaverse ) delete m_pMetaverse;

    if( m_pCircuit ) delete m_pCircuit;
    if( m_pBuffer ) delete [] m_pBuffer;
    if( m_postrstream ) delete m_postrstream;

#ifdef _DEBUG
    g_os = NULL;
#endif

}

// CQuantumDoc::OnNewDocument
//
// Reinitialise the document.

BOOL CQuantumDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

    // Clear the output window
	((CEditView*)m_viewList.GetHead())->SetWindowText(NULL);

    // Create a new output stream, metaverse and circuit
    if( m_postrstream ) delete m_postrstream;
    if( m_pMetaverse ) delete m_pMetaverse;
    if( m_pCircuit ) delete m_pCircuit;

    m_postrstream = new qostrstream( );
    m_pMetaverse = new CMetaverse( *m_postrstream );

    m_bCanRunSimulation = FALSE;

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CQuantumDoc serialization


// CQuantumDoc::Serialize(CArchive& ar)
//
// Called by the MFC framework to load or store the document to ar.
// We don't save the simulation data, just the output text
void CQuantumDoc::Serialize(CArchive& ar)
{
    if( ar.IsStoring() ){
	    // CEditView contains an edit control which handles all serialization
	    ((CEditView*)m_viewList.GetHead())->SerializeRaw(ar);
    } else {
        // Loading. Filename will be stored. We will perform the actual 
        // loading when the simulation starts
        m_bCanRunSimulation = TRUE;
    }
}


/////////////////////////////////////////////////////////////////////////////
// CQuantumDoc diagnostics

#ifdef _DEBUG
void CQuantumDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CQuantumDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////

// CQuantumDoc::GetOutputString()
//
// Called by the view of this document to get the output text from the simulation.
// We only return the last 32k output.
CString CQuantumDoc::GetOutputString()
{
    char *str = m_postrstream->str();
    int iLen = m_postrstream->pcount();
    int iStart, iEnd;

    if( iLen <= 32000 ){
        iStart = 0;
        iEnd =iLen;
    } else {
        iStart = iLen - 32001;
        iEnd = 32000;
    }

    CString cs( str + iStart, iEnd );
    m_postrstream->rdbuf()->freeze( 0 );
    
    return cs;
}


/////////////////////////////////////////////////////////////////////////////
// CQuantumDoc::QuantumGo
//
// Called by the view to start a simulation.
void CQuantumDoc::QuantumGo( WorkerOptions workerOptions, int iData, int iTests, int iTestAlgorithms )
{
    try {
        Simdata sd;

        // Set the default options for a simulation
        sd.m_pDoc               = this;
        sd.m_iCircuitInputs     = CMetaverse::InputFromDefault;
        sd.m_iCircuitOutputs    = CMetaverse::OutputNothing;
        sd.m_iOutputFormat      = CMetaverse::OutputBinary;  
        sd.m_bOutputToFile      = FALSE;
        sd.m_iData              = iData;
        sd.m_iTests             = iTests;
        sd.m_iTestAlgorithms    = iTestAlgorithms;
        sd.m_WorkerOptions      = workerOptions;
        sd.m_bStoreInitialAmps  = (workerOptions == RunContinuousTests);
        sd.m_bSimulateSimply    = m_bSimulateSimply;
        sd.m_iMaxThreads        = m_iMaxThreads;
        sd.m_bSparseVectors     = m_bSparseVectors;
        sd.m_iThreadPriority    = m_iCalculationPriority;

        // Running a loaded circuit?
        if( workerOptions == RunLoadedCircuit ){
            CString strPath = GetPathName();    // Filename
            CString strCircuit;                 // Default circuit name is ""
            ifstream ifs;                       // Input stream to process

            // loading stuff. Haskell file??
            if( strPath.Right(3) == CString(_T(".hs")) ){
                CHaskellEvaluationDialog dlg;   // Get the params for Haskell & the sim

                if( IDOK != dlg.DoModal() )
                    return;

                // Get the circuit name
                strCircuit = dlg.m_strHaskellEval;

                // store the simulation parameters
                sd.m_strInputFile   = dlg.m_strInputFile;
	            sd.m_strOutputFile  = dlg.m_strOutputFile;
	            sd.m_bOutputToFile  = dlg.m_bOutputToFile;
	            sd.m_iBitsFrom      = dlg.m_iBitsFrom;
	            sd.m_iBitsTo        = dlg.m_iBitsTo;
	            sd.m_iCircuitInputs = dlg.m_iCircuitInputs;
	            sd.m_iCircuitOutputs= dlg.m_iCircuitOutputs;
	            sd.m_iOutputFormat  = dlg.m_iOutputFormat;
	            sd.m_iBitsToOutput  = dlg.m_iBitsToOutput;

                // Run Haskell, set ifs to the start of the circuit dump
                if( !GetHaskellOutput( strPath, strCircuit, ifs ))
                    return;

            } else
                // Not a Haskell file. Assume a circuit dump
                ifs.open( strPath );

            // Attempt to load the circuit from the stream ifs
            if( !LoadFrom( strPath, strCircuit, ifs ) )
                return;

            // Are we going to try and perform a quantum factorisation?
            // If so verify that the circuit is OK and get the number
            // to factorise
            if( sd.m_iCircuitInputs == CMetaverse::InputForQFactorisation ){
                int iRegisterASize = 2 * m_pCircuit->NumberOfInputBits() / 3;
                int iRegisterBSize = m_pCircuit->NumberOfInputBits() - iRegisterASize;

                if( iRegisterBSize < 2 ) 
                    QTHROW( "Circuit needs to have an input size of at least 6 bits" );

                int iAPow2 = pow2( iRegisterASize );
                int iHigherLimit = (int) floor( sqrt( iAPow2 ) );
                int iLowerLimit = (int) ceil( sqrt( iAPow2 / 2) );

                CFactorisationParams fpdlg( iLowerLimit, iHigherLimit );
                if( fpdlg.DoModal() != IDOK ) 
                    QTHROW( "Simulation aborted!" );

                if( !(fpdlg.m_iNumberToFactorise & 1) )
                    QTHROW( "Number to factorise is divisible by 2!" );

                sd.m_iNumberToFactorise = fpdlg.m_iNumberToFactorise;
            }


        }

        // Need access to the progress bar dialog box. Gain the mutex. The mutex will 
        // passed to the dalog and released after it has finished constructing 
        // and displaying itself. It will then gain the lock again before closing,
        // and we will finally release the lock when we have finished.
        CSingleLock l( &m_ProgressMutex, TRUE );

        {
            // Create the progress dialog box
            CSimProgress dlg( NULL, l );

            // Set the pointers to the mutex and the dialog box
            sd.m_pSimProgress       = &dlg;
            sd.m_pProgressMutex     = &m_ProgressMutex;

            // Make sure that the stop event isn't still signalled from a 
            // previous simulation
            m_pMetaverse->ResetStopEvent();

            // Start the thread which will perform the simulation. It will start 
            // executing the method CQuantumDoc::SimThreadStart 
            AfxBeginThread( CQuantumDoc::SimThreadStart, &sd, m_iCalculationPriority );

            // Dialog box must release lock in OnInitDialog and gain it again
            // before closing
            dlg.DoModal();

            // Dialog box is going away. Make sure that nothing uses it.
            sd.m_pSimProgress = NULL;
            m_pMetaverse->StopSimulation();
        }

        // Unlock the mutex
        l.Unlock();

        // Wait until we have been signalled that the simulation thread has finished.
        // That way we know that it is safe to destroy the Simdata object sd.
        CSingleLock l2( &m_SimulationFinished, TRUE );

        // BUG: What if _this_ thread crashes _after_ starting the other thread??? The
        // program will then hang or crash. Pretty unlikely though...
    } catch ( CQException qex ){
        // Quantum exception
        AfxMessageBox( qex.GetErrorMessage(), MB_ICONEXCLAMATION | MB_OK );

    } catch ( char *str ) {
        // General string exception
        AfxMessageBox( str, MB_ICONEXCLAMATION | MB_OK );

    } catch ( CException *ex ){
        // MFC exception
        TCHAR str[ 1024 ];
        ex->GetErrorMessage( str, 1024 );

        AfxMessageBox( str, MB_ICONEXCLAMATION | MB_OK );
        ex->Delete();
    } catch ( exception ex ){
        // ANSI draft C++ exception
        AfxMessageBox( ex.what(), MB_ICONEXCLAMATION | MB_OK );

    } catch ( ... ){
        // Unknown exception. Don't throw the error as there is nothing above us to catch it.
        AfxMessageBox( IDS_UNKNOWN_SIM_ERROR, MB_ICONEXCLAMATION | MB_OK );
    }

}

// CQuantumDoc::SimThreadStart( LPVOID lpVoid )
//
// This is where the main simulation thread runs from. lpVoid points to an
// object of type Sindata which holds the parameters for the simulation.
UINT CQuantumDoc::SimThreadStart( LPVOID lpVoid )
{
    Simdata *psd = (Simdata*) lpVoid;

    ASSERT( psd );

    // Each thread gets its own random number generator seed. We need to make sure 
    // that the random number generator for this thread is randomly seeded.
    srand( (unsigned)time( NULL ) );

    // Don't allow the thread to crash!

    // Make sure that we get exceptions on out of memory
    if( !AfxGetNewHandler() ) AfxSetNewHandler( AfxNewHandler );
    try {
        // Run tests?
        if( psd->m_WorkerOptions == RunContinuousTests )
            CQuantumDoc::QuantumSimpleRandomTests( psd );
        else{
            // Nope, running a single circuit.
            HWND hInfo = NULL;

            // Set the info string in the progress dialog box to ""
            {   CSingleLock l( psd->m_pProgressMutex, TRUE );

                if( psd->m_pSimProgress )
                    hInfo = psd->m_pSimProgress->m_Info.m_hWnd;
            }

            ::SetWindowText( hInfo, "" );

            // Run the single circuit
            CQuantumDoc::RunSimulation( psd );
        }

    } catch ( CQException qex ){
        // Quantum exception.
        AfxMessageBox( qex.GetErrorMessage(), MB_ICONEXCLAMATION | MB_OK );

    } catch ( char *str ) {
        // General string exception
        AfxMessageBox( str, MB_ICONEXCLAMATION | MB_OK );

    } catch ( CException *ex ){
        // MFC exception
        TCHAR str[ 1024 ];
        ex->GetErrorMessage( str, 1024 );

        AfxMessageBox( str, MB_ICONEXCLAMATION | MB_OK );
        ex->Delete();

    } catch ( exception ex ){
        // ANSI draft C++ exception
        AfxMessageBox( ex.what(), MB_ICONEXCLAMATION | MB_OK );

    } catch ( ... ){
        // Unkown exception. Don't throw the error on as we are at the top 
        // level of this thread.
        AfxMessageBox( IDS_UNKNOWN_SIM_ERROR, MB_ICONEXCLAMATION | MB_OK );
    }

    // Close the progress dialog box
    {   CSingleLock l( psd->m_pProgressMutex, TRUE );

        // Need to post a WM_CLOSE message. Other methods (DestroyWindow et al)
        // aren't really valid for externally closing a dialog box
        if( psd->m_pSimProgress )
            ::PostMessage( psd->m_pSimProgress->m_hWnd, WM_CLOSE, 0, 0 );
    }

    // Signal that the simulation thread is about to go away...
    psd->m_pDoc->m_SimulationFinished.SetEvent();

    return 0;
}


// CQuantumDoc::RunSimulation( Simdata *psd )
//
// Runs a single simulation with the parameters specified in Simdata* psd.
// Also updates the information in the progress dialog box with the name, size, etc
// of the circuit that is currently being simulated
UINT CQuantumDoc::RunSimulation( Simdata *psd )
{
    ASSERT( psd );
    ASSERT( psd->m_pDoc );
    ASSERT( psd->m_pDoc->m_pMetaverse );
    CQuantumDoc *pDoc = psd->m_pDoc;

    
    // Set the pointers to the input output character arrays. Having seperate 
    // pointers mean that we achieve independance from MFC
    psd->m_szInputFile  = (const char *) psd->m_strInputFile;
    psd->m_szOutputFile = (const char *) psd->m_strOutputFile;

    // Handles of the text message windows in the progress dialog box
    HWND hTest = NULL;
    HWND hTestSize = NULL;
    HWND hCircuit = NULL;
    HWND hAlgorithm = NULL;

    // Find the handles
    {   CSingleLock l( psd->m_pProgressMutex, TRUE );

        if( psd->m_pSimProgress ){
            hTest = psd->m_pSimProgress->m_Test.m_hWnd;
            hCircuit = psd->m_pSimProgress->m_Circuit.m_hWnd;
            hTestSize = psd->m_pSimProgress->m_TestSize.m_hWnd;
            hAlgorithm = psd->m_pSimProgress->m_Algorithm.m_hWnd;
        }
    }

    // Set the name
    if( hTest )
        ::SetWindowText( hTest, pDoc->m_pCircuit->GetName() );

    // Set the circuit name
    if( hCircuit )
        ::SetWindowText( hCircuit, pDoc->m_pCircuit->GetCircuit() );
    
    // Set the algorithm name
    if( hAlgorithm ){
        CString str;
        CString strVectors;

        strVectors.LoadString( psd->m_bSparseVectors ? IDS_SPARSE : IDS_STATIC );

        if( psd->m_bSimulateSimply )
            str.FormatMessage( IDS_SIMPLE_FORMAT, strVectors );
        else
            str.FormatMessage( IDS_COMPLEX_FORMAT, psd->m_iMaxThreads, strVectors );
       
        ::SetWindowText( hAlgorithm, str );
    }
    
    // Set the circuit size
    if( hTestSize ){
        CString str;

        str.FormatMessage( IDS_SIZE_FORMAT
                         , pDoc->m_pCircuit->NumberOfInputBits()
                         , pDoc->m_pCircuit->NumberOfGates()
                         );
        ::SetWindowText( hTestSize, str );
    }

    // Start the simulation
    pDoc->m_pMetaverse->StartSimulation( pDoc->m_pCircuit
                                       , CQuantumDoc::SimCallback
                                       , (void*) psd 
                                       , *((SimParams*) psd)
                                       );
    return 0;
}

// CQuantumDoc::QuantumSimpleRandomTests( Simdata *psd )
//
// Performs continuous random simple tests and checks the results.
void CQuantumDoc::QuantumSimpleRandomTests( Simdata *psd )
{
    ASSERT( psd );
    ASSERT( psd->m_pDoc );
    ASSERT( psd->m_pDoc->m_pMetaverse );

    CMetaverse *pMetaverse = psd->m_pDoc->m_pMetaverse;

    for( int iTests = 0; ; ++iTests ){
        // Get a random size for the circuit in the range 2..psd->m_iData
        int iSize = 2 + (rand() % (psd->m_iData-2) );
        int iCircuit;

        // Make sure that we have some tests to perform and at least
        // one algorithm to use. Each set bit in m_iTest denotes a test
        // that we are allowed to perform. I.e. bit 0 set means that we 
        // can perform test 0. A similar system operates for m_iTestAlgorithms
        ASSERT( psd->m_iTestAlgorithms );
        ASSERT( psd->m_iTests );

        // Decide which algorithm to use.
        if( psd->m_iTestAlgorithms == 1 )
            // Use only the simple algorithm when here
            psd->m_bSimulateSimply = TRUE;
        else if( psd->m_iTestAlgorithms == 2 )
            // Use only the complicated algorithm
            psd->m_bSimulateSimply = FALSE;
        else
            // decide randomly
            psd->m_bSimulateSimply = (BOOL) (rand() % 2);

        // Randomly decide a circuit
        do{
            iCircuit = rand() % CCircuit::NumberOfCircuitTests();
        } while( !( psd->m_iTests & pow2( iCircuit ) ));

        // Random number of threads
        psd->m_iMaxThreads = 1 + (rand() % 8);
        psd->m_bSparseVectors = (BOOL) (rand() % 2);

        // Load the specified circuit.
        psd->m_pDoc->LoadTestCircuit( iCircuit, iSize );

        // Find the handle of the information text window in the progress 
        // dialog box.
        HWND hInfo = NULL;

        {   CSingleLock l( psd->m_pProgressMutex, TRUE );

            if( psd->m_pSimProgress )
                hInfo = psd->m_pSimProgress->m_Info.m_hWnd;
        }

        // Set the informatin test string (no. of tests peroformed so far)
        if( hInfo ){
            CString str;

            str.FormatMessage( IDS_INFO_FORMAT
                             , iTests
                             );
            ::SetWindowText( hInfo, str );
        }

        // Simulate the loaded circuit
        CQuantumDoc::RunSimulation( psd );

        // Did the user hit the cancel button?
        if( pMetaverse->SimulationStopped() )
            break;

        // No, determine if the circuit output the correct results. Get the
        // input and output amplitudes
        CBaseComplexVector *pcomplexInitial = pMetaverse->m_pcomplexInitialAmplitudes;
        CBaseComplexVector *pcomplexOutput = pMetaverse->m_pcomplexOutputAmplitudes;

        // Must be of the same length & greater than 0
        ASSERT( pcomplexInitial->Length() > 0 );
        ASSERT( pcomplexInitial->Length() == pcomplexOutput->Length() );

        int iVectorSize = pcomplexInitial->Length();

        // All of our test circuits set one input and expect one output. This
        // corresponds to exactly one amplitude in each of the input and output vectors
        // to having a probability of 1.0, with the rest having a probability of 0.0
        // Find out which input / output was set.
        int iInitial = -1, iOutput = -1;

        double dSum = 0.0;

        for( int i = 0; i < iVectorSize; i++ ){
            double dInitial = (*pcomplexInitial)[i].MagSquared();
            double dOutput  = (*pcomplexOutput)[i].MagSquared();

            // See if the input or output vectors has a prob. of 1.0
            if( FLOAT_EQ( dInitial, 1 ) ) iInitial = i;
            if( FLOAT_EQ( dOutput, 1 ) )  iOutput = i;

            dSum += dInitial + dOutput;
        }

        // We assume that testing will be done in debugging mode so it is 
        // sufficient to ASSERT that the tests were correct.

        // Make sure that one input and one output was set.
        ASSERT( iInitial != -1 && iOutput != -1 );

        // Make sure that the input and output probabilities sum to 2
        ASSERT( fabs( dSum - 2 ) < 0.001 );

        // Make sure that the output is what we expected for this input
        ASSERT( CCircuit::SimpleTestCorrect( iCircuit, iSize, iInitial, iOutput ));
    }

}


// CQuantumDoc::LoadTestCircuit( int iCircuit, int iSize )
//
// Load a test circuit. Finds out what we should call this circuit by
// looking in the string table. 
void CQuantumDoc::LoadTestCircuit( int iCircuit, int iSize )
{
    ASSERT( m_pMetaverse );
    CString strName, strCircuit;

    // kill any existing circuit.
    if( m_pCircuit )
        delete m_pCircuit;

    // strName = "<<Standard test circuit>>" or equivalent
    strName.LoadString( IDS_STANDARD_TEST_CIRCUIT );

    // Get the circuit name
    static UINT nIDName[] = { IDS_TEST0, IDS_TEST1, IDS_TEST2, 
                              IDS_TEST3, IDS_TEST4, IDS_TEST5, 
                              IDS_TEST6 };

    strCircuit.LoadString( nIDName[ iCircuit ] );

    // Go and create the circuit
    m_pCircuit = CCircuit::CreateTestCircuit( iCircuit, strName, strCircuit, iSize, *m_postrstream );
}


// CQuantumDoc::LoadFrom
//
// Load a circuit from ifs, naming it strPath & strCircuit
BOOL CQuantumDoc::LoadFrom( CString &strPath, CString &strCircuit, ifstream &ifs )
{
    if( m_pCircuit ) delete m_pCircuit;

    m_pCircuit = new CCircuit( strPath, strCircuit );
    m_pCircuit->LoadFrom( ifs, (*m_postrstream) );

    return TRUE;
}

// CQuantumDoc::GetHaskellOutput( CString strPath, CString strHaskellEval, ifstream &ifs )
//
// Called to run the Haskell interpreter and return a stream which points to
// the start of the circuit data which Haskell outputs
BOOL CQuantumDoc::GetHaskellOutput( CString strPath, CString strHaskellEval, ifstream &ifs )
{
    CQuantumApp *pApp   = dynamic_cast<CQuantumApp *>(AfxGetApp());
    CString strCommand  = pApp->m_strHugsExecutable;
    CString strLib      = pApp->m_strHugsLibraries;
    CString strTempDir  = pApp->m_strTempDir;

    if( strTempDir.GetLength() > 0 && strTempDir[strTempDir.GetLength()-1]!='\\' )
        strTempDir += "\\";

    CString strInput = strTempDir + "QuantumInput";
    CString strOutput = strTempDir + "QuantumOutput";

    CString strPrompt = "QUANTUM_PROMPT ";

    CString strOptions = "-P" + strLib;
    CString strPromptOptions = CString("-p\"") + strPrompt + "\" ";

    // Store the input to Haskell in a file.
    {
        ofstream ofs( strInput );
        if( !ofs ){
            AfxMessageBox( IDS_ERROR_RUNNING_HASKELL );
            return FALSE;
        }
        ofs << strHaskellEval << endl;
        ofs << ":q" << endl;
    }

    // build the command line
    CString strCommandLine = strCommand 
                             + CString(" ") + strOptions 
                             + CString(" ") + strPromptOptions
                             + CString(" ") + strPath
                             + CString(" < ") + strInput 
                             + CString(" > ") + strOutput 
                             ;

    // Run Haskell
    if( -1 == system( strCommandLine ) ){
        ASSERT( FALSE );
        AfxMessageBox( IDS_ERROR_RUNNING_HASKELL );
        return FALSE;
    }
                           
    // Try to open Hakell's output
    ifs.open( strOutput );
    ifs.unsetf( ios::skipws );

    if( !ifs ){
        AfxMessageBox(IDS_UNEXPECTED_HASKELL_LOAD_ERROR);
        return FALSE;
    } 

    // Seek the ifs stream to the marker 'strPrompt' which tells us when 
    // the circuit dump is about to start
    int iCompare = 0;

    for(;;){
        char c;
        ifs >> c;

        if( ifs.eof() || !ifs )
            break;

        if( c == strPrompt[ iCompare ] ){
            iCompare++;
            if( iCompare == strPrompt.GetLength() )
                break;
        }
        else
            iCompare = 0;
    }

    // Shouldn't be at the end of file yet...
    if( ifs.eof() || !ifs ){
        AfxMessageBox(IDS_UNEXPECTED_HASKELL_LOAD_ERROR);
        return FALSE;
    }

    return TRUE;
}

// CQuantumDoc::SimCallback(int iProgress, LPVOID lpVoid)
//
// Called by a simulation to inform us of progress during the simulation.
void CQuantumDoc::SimCallback(int iProgress, LPVOID lpVoid, CBaseComplexVector *pcvAmps)
{
    CQuantumDoc::Simdata *psd = (CQuantumDoc::Simdata*) lpVoid;

    // Update the progress bar
    {   CSingleLock l( psd->m_pProgressMutex, TRUE );

        if( psd->m_pSimProgress )
            psd->m_pSimProgress->SetPos( iProgress, 1000, pcvAmps );
    }
}
