#include "stdafx.h"

#include "token.h"

// CToken::CToken()
//
// Create a Token with no type
CToken::CToken()
: m_TokenType( TOKEN_NULL )
{
}

// CToken::CToken( istream &is )
//
// Read a token with the type being given by the data in the input stream is
CToken::CToken( istream &is )
: m_TokenType( TOKEN_NULL )
{
    ReadToken( is );
}

// CToken::TokenType CToken::SkipBlanks( istream &is )
//
// Skip blank characters in the stream is. We consider a character to be
// blank if it is not printable, i.e. a white-space, carriage return, linefeed, etc.
CToken::TokenType CToken::SkipBlanks( istream &is )
{
    int iChar;

    // Check for validity.
    if( is.eof() ) return TOKEN_EOF;
    if( !is ) return TOKEN_ERROR;


    do{ // until we get a printable character...

        // Eat the white spaces
        is.eatwhite();

        // We don't expect to have an error!
        if( !is ) return TOKEN_ERROR;

        // Peek at the next char
        iChar = is.peek();

        // Check for end of files and erros
        if( iChar == EOF ) return TOKEN_EOF;
        if( !is ) return TOKEN_ERROR;

    } while( !isprint( iChar ) );

    return TOKEN_NULL;
}

// CToken::TokenType CToken::ReadTokenString( istream &is, char *szToken, int iMaxTokenSize )
//
// Read string from is up to next [](), or whitespace
CToken::TokenType CToken::ReadTokenString( istream &is, char *szToken, int iMaxTokenSize )
{
    static const char *szEndOfString = "[](),:";
    int iTokenLength= 0;

    // check for EOF
    if( is.peek() == EOF ) return TOKEN_EOF;

    for( ;; ){
        int iChar = is.peek();

        if( !is ) return TOKEN_ERROR;

        // String is ended if we are at the end of the file, the character isn't
        // printable, it is a whitespace or it is one of [](),:
        if( iChar == EOF || !isprint( iChar ) 
         || isspace( iChar ) || strchr( szEndOfString, iChar ) 
          ){
            // Terminate the string and return. Attach no type to the token
            szToken[ iTokenLength ] = '\0';
            return TOKEN_NULL;
        } else
            // Else store the character.
            is >> szToken[ iTokenLength++ ];

        // Make sure that we aren't going to exceed the maximum string length.
        if( iTokenLength == iMaxTokenSize - 1 ){
            szToken[ iTokenLength ] = '\0';
            return TOKEN_TOO_BIG;
        }
    }

    ASSERT( !"Should never get here!" );
    return TOKEN_ERROR;
}

// CToken CToken::ReadNumber( istream &is )
//
// Read a number from is. The number may be an integer or a complex number in the
// form real :+ imag
CToken CToken::ReadNumber( istream &is )
{
    CToken tokenReturn;
    const int iMaxTokenSize = 100;
    char szToken[ iMaxTokenSize + 1];

    // Read the string on the input
    TokenType tokenType = ReadTokenString( is, szToken, iMaxTokenSize );

    // Of a determined type? (e.g. an error token)
    if( tokenType != TOKEN_NULL ){
        tokenReturn.m_TokenType = tokenType;
        return tokenReturn;
    }

    // Integer? I.e. pure digits '0'..'9'
    BOOL bInteger = TRUE;
    for( char *s = szToken; *s; s++ ){
        if( !isdigit( (int) (unsigned char) *s ) ){
            bInteger = FALSE;
            break;
        }
    }

    char *pEnd = NULL;

    // Possible integer if true
    if( bInteger ){
        long lValue = strtol( szToken, &pEnd, 10 );
        // Check for correct conversion.
        if(( pEnd != strlen( szToken ) + szToken) || errno == ERANGE )
            tokenReturn.m_TokenType = TOKEN_ERROR_BAD_INT;
        else {
            tokenReturn.m_TokenType = TOKEN_INT;
            tokenReturn.m_iValue = lValue;
        }

        return tokenReturn;
    }

    // Otherwise complex number when here. Attempt to convert the real part.
    double dValue = strtod( szToken, &pEnd );
    if(( pEnd != strlen( szToken ) + szToken) || errno == ERANGE ){
        // Bad conversion when here
        tokenReturn.m_TokenType = TOKEN_ERROR_BAD_FLOAT;
        return tokenReturn;
    } else 
        tokenReturn.m_dReal = dValue;

    // Expect :+
    if( ReadToken( is ).m_TokenType != TOKEN_COLON_PLUS ){
        tokenReturn.m_TokenType = TOKEN_EXPECTED_COMPLEX;
        return tokenReturn;
    }

    // Skip any blanks after the :+
    tokenType = SkipBlanks( is );

    // Read the next string
    if( tokenType == TOKEN_NULL )
        tokenType = ReadTokenString( is, szToken, iMaxTokenSize );

    // Check for error returns
    if( tokenType != TOKEN_NULL ){
        tokenReturn.m_TokenType = tokenType;
        return tokenReturn;
    }

    // Attempt to convert the imaginary part.
    dValue = strtod( szToken, &pEnd );
    if(( pEnd != strlen( szToken ) + szToken) || errno == ERANGE ){
        // Bad conversion when here.
        tokenReturn.m_TokenType = TOKEN_ERROR_BAD_FLOAT;
        return tokenReturn;
    } else 
        tokenReturn.m_dImag = dValue;

    // Finally set the token type and return
    tokenReturn.m_TokenType = TOKEN_COMPLEX;

    return tokenReturn;
}

// CToken CToken::ReadToken( istream &is )
//
// Read a token from is
CToken CToken::ReadToken( istream &is )
{
    CToken tokenReturn;
    tokenReturn.m_TokenType = SkipBlanks( is );

    if( tokenReturn.m_TokenType != TOKEN_NULL )
        return tokenReturn;

    // Get the first character
    char c = '\0';
    is >> c;

    switch( c ){
    case '(':
        tokenReturn.m_TokenType = TOKEN_OPEN_BRACKET;
        break;

    case ')':
        tokenReturn.m_TokenType = TOKEN_CLOSE_BRACKET;
        break;

    case '[':
        tokenReturn.m_TokenType = TOKEN_OPEN_SQUARE_BRACKET;
        break;

    case ']':
        tokenReturn.m_TokenType = TOKEN_CLOSE_SQUARE_BRACKET;
        break;

    case ',':
        tokenReturn.m_TokenType = TOKEN_COMMA;
        break;

    case ':':
        if( is.peek() == '+' ){
            tokenReturn.m_TokenType = TOKEN_COLON_PLUS;
            is >> c;
        } else
            tokenReturn.m_TokenType = TOKEN_ERROR_COLON_NO_PLUS;
        break;

    default:
        // None of the above. We assume that it is a number (strings are not 
        // valid input to the simulator)
        is.putback( c );
        tokenReturn = ReadNumber( is );
    }

    return tokenReturn;
}

// istream& operator>>( istream &is, CToken &token )
//
// Read from is into token
istream& operator>>( istream &is, CToken &token )
{
    token = CToken::ReadToken( is );

    return is;
}

// const char *CToken::GetTokenString( TokenType tokenType )
//
// return a the humanised name of a token
const char *CToken::GetTokenString( TokenType tokenType )
{
    static struct {
        TokenType m_TokenType;
        const char *m_szString;
    } stringList[] = {
        {TOKEN_NULL,                 "NULL"},
        {TOKEN_OPEN_BRACKET,         "("},
        {TOKEN_CLOSE_BRACKET,        ")"},
        {TOKEN_COMMA,                ","},
        {TOKEN_OPEN_SQUARE_BRACKET,  "["},
        {TOKEN_CLOSE_SQUARE_BRACKET, "]"},
        {TOKEN_COLON_PLUS,           ":+"},
        {TOKEN_TOO_BIG,              "ERROR - Token too big"},
        {TOKEN_ERROR,                "Unexpected error encountered"},
        {TOKEN_EOF,                  "End Of File"},
        {TOKEN_EXPECTED_COMPLEX,     "ERROR - Expected a complex number"},
        {TOKEN_ERROR_BAD_FLOAT,      "ERROR - Bad floating point number"},
        {TOKEN_ERROR_BAD_INT,        "ERROR - Bad integer number"},
        {TOKEN_ERROR_BAD_INT,        "ERROR - expected a '+' after the ':'"},
        {TOKEN_INT,                  "integer"},
        {TOKEN_COMPLEX,              "complex"}
    };

    for( int i =0; i < sizeof stringList / sizeof stringList[0]; i++ ){
        if( tokenType == stringList[i].m_TokenType ){
            return stringList[i].m_szString;
        }
    }
    
    ASSERT( FALSE );
    return "Token: AAARRRRGGGHHHH!!! Unknown Token";
    
}

// ostream &operator<<( ostream &os, CToken &token )
//
// Output a string, mainly for reporting errors.
ostream &operator<<( ostream &os, CToken &token )
{
    os << "Token: " << CToken::GetTokenString( token.m_TokenType );

    if( token.m_TokenType == CToken::TOKEN_INT ){
        os << " " << token.m_iValue;
    } else if( token.m_TokenType == CToken::TOKEN_COMPLEX ) {
        os << ", real = " << token.m_dReal << " imag = " << token.m_dImag;
    }
    
    
    return (os << endl);
}
