/*
 * Created on 28-Oct-2004
 */
package kenya.sole.ui.editor;


import java.io.File;
import java.util.Date;
import java.util.List;

import kenya.FileLoaderSaver;
import kenya.sole.ui.core.EditingWindow;
import kenya.sole.ui.editor.kenya.KenyaKeyListener;
import kenya.sole.ui.editor.kenya.KenyaPaintListener;
import kenya.sole.ui.editor.kenya.KenyaTableSelectionChangedListener;
import kenya.sole.ui.editor.kenya.KenyaTextChangedListener;
import kenya.sole.ui.editor.util.DefaultHighlightingManager;
import kenya.sole.ui.editor.util.HighlightingFactory;
import kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import kenya.sourceCodeInformation.interfaces.ISourceCodeInformation;
import kenya.ui.IButtonManager;
import kenya.ui.errors.HumanErrorLabelProvider;
import kenya.ui.errors.JavacErrorLabelProvider;
import kenya.ui.errors.KenyaErrorContentProvider;
import kenya.ui.indentation.IndentationStrategy;
import kenya.ui.syntaxhighlighting.KenyaLineStyler;
import mediator.ICheckedCode;
import mediator.subscription.ICheckedCodeReceiver;

import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;

/**
 * @author Thomas Timbul
 */
public class KenyaPart extends EditorPart implements ICheckedCodeReceiver {
	
	protected volatile TableViewer kenyaHumanErrorTable;
	protected volatile TableViewer kenyaJavacErrorTable;
	
	protected IndentationStrategy indentationStrategy;
	protected KenyaErrorContentProvider kenyaTableContents;
	
	private final Date lastKeyEvent = new Date();
	
	private volatile boolean reparse = false;
	
	private volatile FileLoaderSaver kenyaFileLoaderSaver = null;
	
	private final Runnable runLoadFromKenyaFile = new Runnable() {
		
		public void run() {
			loadFromKenyaFile();
		}
	};
	
	
	
	public KenyaPart(
				Composite parent,
				int style,
				IStatusLineManager statusLine,
				IButtonManager buttonManager
			) {
		super(parent, style, statusLine, buttonManager);
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#init()
	 */
	public void init() {
		super.init();
		
		setHighlightingManager(
				new DefaultHighlightingManager(
						this,
            HighlightingFactory.createErrorHighlighter(this),
						HighlightingFactory.createWarningHighlighter(this),
						HighlightingFactory.createLinkedLocationHighlighter(this)
				));
		
		indentationStrategy = new IndentationStrategy();
		kenyaTableContents = new KenyaErrorContentProvider();
		
		TabFolder kenyaTabs = new TabFolder(this, SWT.NULL);
		
		TabItem humanErrorsTab = new TabItem(kenyaTabs, SWT.NULL);
		humanErrorsTab.setText("Kenya errors");
		
		TabItem javacErrorsTab = new TabItem(kenyaTabs, SWT.NULL);
		javacErrorsTab.setText("Java errors");
		
		kenyaHumanErrorTable = new TableViewer(kenyaTabs, SWT.BORDER
				| SWT.SINGLE | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL);
		kenyaHumanErrorTable.setUseHashlookup(true);
		kenyaHumanErrorTable.setContentProvider(kenyaTableContents);
		kenyaHumanErrorTable.setLabelProvider(new HumanErrorLabelProvider(
				buttonManager,
				editorText.getLineDelimiter()
		));
		ISelectionChangedListener tableSelectionChanged
				= new KenyaTableSelectionChangedListener(this);
		
		kenyaHumanErrorTable.addSelectionChangedListener(tableSelectionChanged);
		kenyaHumanErrorTable.getTable().setLinesVisible(false);
		kenyaHumanErrorTable.getTable().setHeaderVisible(false);
		humanErrorsTab.setControl(kenyaHumanErrorTable.getControl());

		kenyaJavacErrorTable = new TableViewer(kenyaTabs, SWT.BORDER
				| SWT.SINGLE | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL);
		kenyaJavacErrorTable.setUseHashlookup(true);
		kenyaJavacErrorTable.setContentProvider(kenyaTableContents);
		kenyaJavacErrorTable.setLabelProvider(new JavacErrorLabelProvider(
    		buttonManager,
				editorText.getLineDelimiter()
		));
		kenyaJavacErrorTable.addSelectionChangedListener(tableSelectionChanged);
		kenyaJavacErrorTable.getTable().setLinesVisible(false);
		kenyaJavacErrorTable.getTable().setHeaderVisible(false);
		javacErrorsTab.setControl(kenyaJavacErrorTable.getControl());
		
		this.setWeights(new int[] {3, 1} );
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#buildEditorContent(org.eclipse.swt.custom.SashForm)
	 */
	protected void buildEditorContent(SashForm parent) {
		super.buildEditorContent(parent);
		setLineStyler(new KenyaLineStyler());
		
		editorText.addVerifyKeyListener(new KenyaKeyListener(this));
		editorText.addPaintListener(new KenyaPaintListener(this));
		editorText.getContent().addTextChangeListener(
				new KenyaTextChangedListener(this));
		
		if (highlightingManager != null) {
			editorText.addMouseListener(highlightingManager
					.createKenyaMouseListener(this));
			editorText.addLineBackgroundListener(highlightingManager
					.createKenyaLineBackgroundListener(this));
		}
		
	}
	
	/* (non-Javadoc)
	 * @see kenya.gui.editor.EditorPart#setFont(org.eclipse.swt.graphics.Font)
	 */
	public void setFont(Font font) {
		super.setFont(font);
		if (kenyaHumanErrorTable != null) {
			kenyaHumanErrorTable.getControl().setFont(font);
			kenyaJavacErrorTable.getControl().setFont(font);
		}
	}
	
	public synchronized void reparse(boolean value) {
		reparse = value;
	}
	
	public synchronized boolean reparse() {
		return reparse;
	}
	
	public IndentationStrategy getIndentationStrategy() {
		return indentationStrategy;
	}
	
	public void selectInKenyaErrorTable(final ISourceCodeInformation error) {
		if (!isVisible())
			return;
		if (Thread.currentThread()
				.equals(this.getDisplay().getThread())) {
			synchronized (this) {
				if (kenyaJavacErrorTable == null
						|| kenyaHumanErrorTable == null)
					return;
				if (error == null && !this.isDisposed()) {
					kenyaJavacErrorTable.getTable().deselectAll();
					kenyaHumanErrorTable.getTable().deselectAll();
				} else {
					int index = kenyaTableContents.getIndex(error);
					if (index == -1) {
						kenyaJavacErrorTable.getTable().deselectAll();
						kenyaHumanErrorTable.getTable().deselectAll();
					} else {
						kenyaJavacErrorTable.getTable().setSelection(index);
						kenyaJavacErrorTable.getTable().showSelection();
						kenyaHumanErrorTable.getTable().setSelection(index);
						kenyaHumanErrorTable.getTable().showSelection();
					}
				}
			}
		} else {
			if (!isVisible())
				return;
			this.getDisplay().syncExec(new Runnable() {

				public void run() {
					selectInKenyaErrorTable(error);
				}
			});
		}
	}
	
	public synchronized void resetTimer() {
		lastKeyEvent.setTime(System.currentTimeMillis());
	}
	
	public synchronized Date getLastKeyEvent() {
		return lastKeyEvent;
	}
	
	public void setCheckedCode(final ICheckedCode checker) {
		getHighlightingManager().clear();
		if (!this.isDisposed() && Thread.currentThread()
				.equals(this.getDisplay().getThread())) {
			synchronized (this) {
				kenyaHumanErrorTable.setInput(checker);
				kenyaJavacErrorTable.setInput(checker);
				List warnings = checker.getInfos();
				
				if (checker.isErroredCode()) {
					List errors = checker.getErrors();
					for (int idx = 0; idx < errors.size(); idx++) {
						ISourceCodeError error = (ISourceCodeError) errors
								.get(idx);
						getHighlightingManager().addSourceCodeError(error);
					}
					if (errors.size() == 1)
						safeSetStatus("1 error found.");
					else
						safeSetStatus("" + errors.size() + " errors found.");
				} else {
					if (warnings.size() == 0)
						safeSetStatus("No errors or warnings found.");
					else if (warnings.size() == 1)
						safeSetStatus("1 warning found; no errors found.");
					else
						safeSetStatus("" + warnings.size()
								+ " warnings found; no errors found.");
				}
	
				for (int idx = 0; idx < warnings.size(); idx++) {
					ISourceCodeInformation warning = (ISourceCodeInformation) warnings
							.get(idx);
					getHighlightingManager().addSourceCodeWarning(warning);
				}
				//reparse the block comments in case they have changed...
				((KenyaLineStyler)getLineStyler())
					.parseBlockComments(getEditorTextWidget().getContent());
				getEditorTextWidget().redraw();
			}
		} else {
			if (!isVisible())
				return;
			this.getDisplay().syncExec(new Runnable() {
	
				public void run() {
					setCheckedCode(checker);
				}
			});
		}
	
	}
	
	public synchronized void setKenyaFile(FileLoaderSaver fls) {
		kenyaFileLoaderSaver = fls;
	}
	
	
	public void loadFromKenyaFile() {
		if (editorText != null && getKenyaFile() != null) {
			if (Thread.currentThread().equals(
						editorText.getDisplay().getThread()))
					{
				synchronized (this) {
					FileLoaderSaver fls = getKenyaFile();
					//ModificationEvent e = new ModificationEvent(this, fls, true);
					//fireEditorPreModified(e);
					String contents = fls.loadFile(fls.getPath());
					editorText.setText(contents);
					
					this.getIndentationStrategy().parseText(editorText);
					editorText.setCaretOffset(0);
					editorText.setSelection(0);
					editorText.setFocus();
					this.reparse(true);
					this.setModified(false);
//					fireEditorPostModified(e);
				}
			} else {
				editorText.getDisplay().syncExec(runLoadFromKenyaFile);
			}
		}
	}
	
	public synchronized FileLoaderSaver getKenyaFile() {
		return kenyaFileLoaderSaver;
	}

	public void saveKenyaAs(final boolean saveJava) {
		if (Thread.currentThread()
				.equals(this.getDisplay().getThread())) {
			synchronized (this) {
				Shell shell = new Shell();
				FileDialog dialog = new FileDialog(shell, SWT.SAVE);
				dialog.setFilterExtensions(new String[] { "*.k" });
				dialog.setText("Save Kenya file");
				
				String lastDir = EditingWindow.lastDir;
				
				if (lastDir == null) {
					if (getKenyaFile() != null) {
						lastDir = getKenyaFile().getDir();
						if (lastDir.length() == 0)
							lastDir = null;
					}
				}
				if (lastDir != null)
					dialog.setFilterPath(lastDir);

				if (dialog.open() != null) {
					lastDir = dialog.getFilterPath();

					File file = new File(dialog.getFilterPath(), dialog
							.getFileName());
					if (getKenyaFile() == null) {
						FileLoaderSaver fls = new FileLoaderSaver();
						fls.setPath(file.getAbsolutePath());
						setKenyaFile(fls);
					} else {
						FileLoaderSaver fls = getKenyaFile();
						fls.setPath(file.getAbsolutePath());
					}
					saveKenya(saveJava);
				}
			}
		} else {
			this.getDisplay().syncExec(new Runnable() {
				public void run() {

					saveKenyaAs(saveJava);
				}
			});
		}
	}
	
	public void saveKenya(final boolean saveJava) {
		System.err.println("save disabled!");
//		if (Thread.currentThread()
//				.equals(this.getDisplay().getThread())) {
//			synchronized (this) {
//				FileLoaderSaver fls = getKenyaFile();
//				if (fls == null || fls.getPath() == null) {
//					saveKenyaAs(saveJava);
//					return;
//				} else {
//					fls.saveFile(getEditorTextWidget().getText());
//					setModified(false);
//					safeSetStatus("Kenya code saved to " + fls.getPath());
//				}
//				if (saveJava) {
//					Shell shell = new Shell();
//					try {
//						ICheckedCode checkedCode = Mediator
//								.check(new StringReader(getEditorTextWidget()
//										.getText()));
//						getHighlightingManager().clear();
//						setKenyaCheckedCode(checkedCode);
//						if (!checkedCode.isErroredCode()) {
//							( (JavaPart)getEditor(JAVATABINDEX) ).setJavaCode(checkedCode.translate());
//							MessageBox box = new MessageBox(shell, SWT.YES | SWT.NO);
//							box.setMessage("Would you like to save the Java translation of your Kenya?");
//							box.setText("Save Java Translation");
//							if (box.open() == SWT.YES) {
//								saveJava();
//							} else {
//								safeSetStatus("Java Translation not saved.");
//							}
//						} else {
//							( (JavaPart)getEditor(JAVATABINDEX) ).setJavaCode(null);
//							MessageBox box = new MessageBox(shell, SWT.OK);
//							box
//									.setMessage("Kenya code contains errors, unable to save Java translation.");
//							box.setText("Save Java Translation");
//							box.open();
//						}
//					} catch (IOException e1) {
//						System.err.println("Error whilst parsing code.");
//					}
//
//				}
//			}
//		} else {
//			this.getDisplay().syncExec(new Runnable() {
//				public void run() {
//
//					saveKenya(saveJava);
//				}
//			});
//		}
	}
	
	
}
