/* *******************************************************************************
 *   kenya                                                             *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/
/*
 * Created on 25-Aug-2004
 */
package kenya.sole.ui.editor.interpreter;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;

import kenya.interpreter.util.InterpreterLastPointReachedCatcher;
import kenya.sole.ui.editor.InterpreterPart;
import kenya.sole.ui.util.StdOutWriter;
import kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import mediator.stackMachine.IStackMachineInformationProvider;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Text;
import org.wellquite.kenya.stackMachine.IPointListener;
import org.wellquite.kenya.stackMachine.StackMachine;
import org.wellquite.kenya.stackMachine.misc.AbstractJob;
import org.wellquite.kenya.stackMachine.misc.JobDispatch;
import org.wellquite.kenya.stackMachine.types.ArrayTypeFactory;
import org.wellquite.kenya.stackMachine.types.StringTypeFactory;
import org.wellquite.kenya.stackMachine.types.interfaces.IAtomicClosure;
import org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedClass;
import org.wellquite.kenya.stackMachine.types.interfaces.IStringType;
import org.wellquite.kenya.stackMachine.types.interfaces.IType;

/**
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class InterpreterInterpretSelectionListener extends SelectionAdapter {

	private final Button interpret;

	private final InterpreterPart interpreter;

	private final StyledText stdOut;

	private final Text stdIn;

	private final Button eof;

	private final Button terminate;

	private final Runnable enableStuff;

	private final Runnable disableStuff;

	private SelectionListener eofListener = null;

	private KeyListener keyListener = null;

	private SelectionListener terminateListener = null;

	public InterpreterInterpretSelectionListener(InterpreterPart intp) {

		interpreter = intp;
		stdOut = intp.getInterpreterStdOutText();
		stdIn = intp.getInterpreterStdInText();
		interpret = intp.getInterpreterInterpretButton();
		eof = intp.getInterpreterEOFButton();
		terminate = intp.getInterpreterTerminateButton();

		enableStuff = new Runnable() {

			public void run() {
				interpret.setEnabled(false);
				stdIn.setEnabled(true);
				stdIn.setFocus();
				stdIn.setText("");
				terminate.setEnabled(true);
				terminate.setText("Terminate");
				eof.setEnabled(true);
				stdOut.setText("");
			}
		};

		disableStuff = new Runnable() {

			public void run() {
				interpret.setEnabled(true);
				stdIn.setEnabled(false);
				terminate.setText("Terminate");
				terminate.setEnabled(false);
				eof.setEnabled(false);
				interpreter.getInterpreterContinueButton().setEnabled(false);
				StackMachine sm = interpreter.getStackMachine();
				if (sm != null) {
					sm.resume();
					interpreter.setStackMachine(null);
				}
			}
		};
	}

	public synchronized void widgetSelected(SelectionEvent e) {
		interpret.setEnabled(false);

		final StackMachine sm = new StackMachine();

		final IStackMachineInformationProvider ismip = interpreter
				.getBaseStackMachine();

		if (ismip == null)
			return;

		try {
			PipedInputStream pis = new PipedInputStream();
			PipedOutputStream pos = new PipedOutputStream(pis);

			final PrintStream outStream = new PrintStream(pos, true);
			sm.setOut(outStream);

			BufferedReader outReader = new BufferedReader(
					new InputStreamReader(pis), 16);
			AbstractJob stdOutReader = new StdOutWriter(outReader, stdOut,
					SWT.COLOR_BLUE);

			PipedInputStream pisErr = new PipedInputStream();
			PipedOutputStream posErr = new PipedOutputStream(pisErr);

			final PrintStream outStreamErr = new PrintStream(posErr, true);
			sm.setErr(outStreamErr);

			BufferedReader outErrReader = new BufferedReader(
					new InputStreamReader(pisErr), 16);
			AbstractJob stdErrReader = new StdOutWriter(outErrReader, stdOut,
					SWT.COLOR_RED);

			PipedOutputStream qout = new PipedOutputStream();
			PipedInputStream qin = new PipedInputStream(qout);
			final InputStream inStream = new BufferedInputStream(qin);
			sm.setIn(inStream);

			final PrintWriter inWriter = new PrintWriter(
					new OutputStreamWriter(qout), true);

			if (keyListener != null)
				stdIn.removeKeyListener(keyListener);
			keyListener = new KeyAdapter() {

				public void keyReleased(KeyEvent e) {
					if (e.character == SWT.CR) {
						e.doit = false;
						String text = stdIn.getText() + "\n";
						stdIn.setText("");

						stdOut.append(text);
						StyleRange sr = new StyleRange();
						sr.start = stdOut.getCharCount() - text.length();
						sr.length = text.length();
						sr.foreground = stdOut.getDisplay().getSystemColor(
								SWT.COLOR_DARK_GREEN);
						stdOut.setStyleRange(sr);
						stdOut.setCaretOffset(stdOut.getCharCount());
						stdOut.setSelection(stdOut.getCharCount());
						stdOut.showSelection();

						inWriter.write(text);
						inWriter.flush();
					}
				}
			};
			stdIn.addKeyListener(keyListener);

			if (eofListener != null)
				eof.removeSelectionListener(eofListener);

			eofListener = new SelectionAdapter() {

				public void widgetSelected(SelectionEvent e) {
					inWriter.close();
					stdIn.setEnabled(false);
					eof.setEnabled(false);
				}
			};
			eof.addSelectionListener(eofListener);

			if (terminateListener != null)
				terminate.removeSelectionListener(terminateListener);

			terminateListener = new InterpreterTerminateButtonSelectionListener(
					interpreter, inWriter);
			terminate.addSelectionListener(terminateListener);

			AbstractJob launch = new AbstractJob() {

				public void execute() {
					try {
						interpreter.getDisplay().syncExec(enableStuff);
						IInterpretedClass launchClass = ismip
								.getEntryPointClass();
						String launchMethod = ismip.getEntryPoint();
						IPointListener bcm = new BlueCarrotMover(interpreter);
						sm.addPositionReachedListener(bcm);

						InterpreterLastPointReachedCatcher ilprc = new InterpreterLastPointReachedCatcher(
								sm);
						sm.addPositionReachedListener(ilprc);

						if (interpreter.getInterpreterStepMode()) {
							sm.setStepMode(true);
						} else {
							sm.setStepMode(false);
						}

						interpreter.setStackMachine(sm);
						try {
							if (ismip.entryPointTakesArguments()) {
								String[] argsString = interpreter
										.getInterpreterArgs();
								IType[] argsIType = new IType[argsString.length];
								for (int idx = 0; idx < argsString.length; idx++) {
									argsIType[idx] = StringTypeFactory
											.createStringType(argsString[idx]);
								}
								sm.push(ArrayTypeFactory
										.createArrayType(argsIType, IStringType.TYPE_NAME));
							}
							interpreter.safeSetStatus("Debugger starting.");
							sm.invokeMethod(launchClass, launchMethod);
							interpreter.positionBlueCarrot(0);
							interpreter.safeSetStatus("Debugger completed.");
						} catch (Throwable t) {
							sm.shutdown();
							IAtomicClosure lastClosure = ilprc.getLastClosure();
							ISourceCodeLocation loc = (ISourceCodeLocation) ilprc
									.getLastPointData();
							if (loc == null || lastClosure == null) {
								outStreamErr.println(t);
							} else {
								outStreamErr.println(t + " at " + lastClosure
										+ ", line " + loc.getLineNumber()
										+ ", column " + loc.getColumnNumber());
								interpreter.positionBlueCarrot(loc);
							}
						} finally {
							outStream.close();
							outStreamErr.close();
							inWriter.close();

							sm.removePositionReachedListener(bcm);

							sm.shutdown();
							interpreter.setStackMachine(null);

							interpreter.getDisplay().syncExec(
									disableStuff);
						}
					} catch (RuntimeException e) {
						System.err.println("SM encountered an error:\n");
						e.printStackTrace(System.err);
					}
				}
			};

			JobDispatch.enqueueJob(stdOutReader);
			JobDispatch.enqueueJob(stdErrReader);
			JobDispatch.enqueueJob(launch);
		} catch (IOException e3) {
			e3.printStackTrace();
		}
	}

}