/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004
 */
package kenya.sole.ui.core;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import kenya.FileLoaderSaver;
import kenya.errors.KenyaInternalError;
import kenya.sole.ui.action.AboutAction;
import kenya.sole.ui.action.FormatAction;
import kenya.sole.ui.action.ISwitchActionTarget;
import kenya.sole.ui.action.SwitchAction;
import kenya.sole.ui.action.UniqueCheckedManager;
import kenya.sole.ui.action.file.CloseAction;
import kenya.sole.ui.action.file.ExitAction;
import kenya.sole.ui.action.file.JavaSaveAction;
import kenya.sole.ui.action.file.JavaSaveAsAction;
import kenya.sole.ui.action.file.KenyaSaveAction;
import kenya.sole.ui.action.file.KenyaSaveAsAction;
import kenya.sole.ui.action.file.NewAction;
import kenya.sole.ui.action.file.OpenAction;
import kenya.sole.ui.action.file.OpenInNewAction;
import kenya.sole.ui.action.file.SavePromptAction;
import kenya.sole.ui.action.fontchange.EditorBigFontAction;
import kenya.sole.ui.action.fontchange.EditorFixedFontAction;
import kenya.sole.ui.action.fontchange.EditorHugeFontAction;
import kenya.sole.ui.action.fontchange.EditorMediumFontAction;
import kenya.sole.ui.action.fontchange.EditorProportionalFontAction;
import kenya.sole.ui.action.fontchange.EditorSmallFontAction;
import kenya.sole.ui.action.fontchange.FontChangeAction;
import kenya.sole.ui.action.fontchange.IOBigFontAction;
import kenya.sole.ui.action.fontchange.IOFixedFontAction;
import kenya.sole.ui.action.fontchange.IOHugeFontAction;
import kenya.sole.ui.action.fontchange.IOMediumFontAction;
import kenya.sole.ui.action.fontchange.IOProportionalFontAction;
import kenya.sole.ui.action.fontchange.IOSmallFontAction;
import kenya.sole.ui.editor.EditorPart;
import kenya.sole.ui.editor.EditorPartsFontTarget;
import kenya.sole.ui.editor.InterpreterPart;
import kenya.sole.ui.editor.JavaPart;
import kenya.sole.ui.editor.KenyaMediatorInput;
import kenya.sole.ui.editor.KenyaPart;
import kenya.sole.ui.editor.kenya.KenyaHelperThread;
import kenya.sole.ui.event.IModificationListener;
import kenya.sole.ui.event.ModificationEvent;
import kenya.sole.ui.event.TabSelectionAdapter;
import kenya.sole.ui.icons.ButtonManager;
import kenya.sole.ui.util.IHighlightingManager;
import kenya.ui.IButtonManager;
import mediator.IJavaCode;
import mediator.subscription.ISubscriber;
import mediator.subscription.ISubscriptionData;
import mediator.subscription.MediationService;
import mediator.subscription.MediationSubscription;
import mediator.subscription.MediationTerms;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.window.ApplicationWindow;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.wellquite.kenya.stackMachine.misc.JobDispatch;

/**
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class EditingWindow extends ApplicationWindow
                           implements ISubscriber, IModificationListener
		{
	
	public static final int KENYATABINDEX = 0;
	public static final int JAVATABINDEX = 1;
	public static final int INTERPRETERTABINDEX = 2;
	
	private final Runnable runIsVisible = new Runnable() {

		public void run() {
			isVisible();
		}
	};
	
	private final Runnable runCloseWindow = new Runnable() {

		public void run() {
			closeWindow();
		}
	};
	
	private final Object secondaryLock = new Object();
	
	private volatile IButtonManager buttonManager = null;
	
	private volatile TabFolder topTabs = null;
	
	private volatile SelectionListener tabSelectionListener = null;
	
	private volatile boolean isVisible = false;
	
	private volatile IAction javaSaveAction = null;
	
	private volatile IAction javaSaveAsAction = null;
	
	private volatile IHighlightingManager highlights = null;
	
	public static volatile String lastDir = null;
	
	private static final String file_menu_id = "file_menu";
	private static final String edit_menu_id = "edit_menu";
	
	/**
	 * the ID for the mediationservice subscription
	 */
	private String subscriptionID;
	
	public EditingWindow() {
		super(null);
		addStatusLine();
		addMenuBar();
	}
	
	// only the GUI thread will call this
	protected synchronized Control createContents(Composite parent) {
		this.getShell().setText("Kenya IDE");
		
		buttonManager = new ButtonManager();

		topTabs = new TabFolder(parent, SWT.NULL);
		
		final TabItem kenyaTab = new TabItem(topTabs, SWT.NULL);
		kenyaTab.setText("&Kenya");

		final TabItem javaTab = new TabItem(topTabs, SWT.NULL);
		javaTab.setText("&Java");
		
		final TabItem interpreterTab = new TabItem(topTabs, SWT.NULL);
		interpreterTab.setText("&Debugger");
		
		int style = SWT.VERTICAL | SWT.BORDER;
		
		final KenyaPart kenyaTopControl
			= new KenyaPart(
					topTabs,
					style,
					this.getStatusLineManager(),
					buttonManager
			);
		kenyaTopControl.init();
		kenyaTab.setControl(kenyaTopControl);
		kenyaTopControl.addModificationListener(this);
		
		final JavaPart javaTopControl
			= new JavaPart(
					topTabs,
					style,
					this.getStatusLineManager(),
					buttonManager,
					getNextClassName()
			);
		javaTopControl.init();
		
		javaTab.setControl(javaTopControl);
		
		final InterpreterPart interpreterTopControl
			= new InterpreterPart(
					topTabs,
					style,
					this.getStatusLineManager(),
					buttonManager
			);
		interpreterTopControl.init();
		interpreterTab.setControl(interpreterTopControl);
		
		final StyledText kenyaText
				= kenyaTopControl.getEditorTextWidget();
		interpreterTopControl.getEditorTextWidget().addKeyListener(
				new KeyListener() {
					
					public void keyPressed(KeyEvent e) {
					}
					
					public void keyReleased(KeyEvent e) {
						if (e.character >= ' ') {
							int caretOffset = kenyaText.getCaretOffset();
							topTabs.setSelection(KENYATABINDEX);
							kenyaText.setSelection(caretOffset);
							kenyaText.showSelection();
							kenyaText.insert("" + e.character);
							kenyaText.setCaretOffset(caretOffset + 1);
							kenyaText.setFocus();
						}
					}
				}
			);
		
		tabSelectionListener = new TabSelectionAdapter(
					kenyaTopControl,
					javaTopControl,
					interpreterTopControl
				);
		topTabs.addSelectionListener(tabSelectionListener);
		
		FontData fontdata = javaTopControl.getFont().getFontData()[0];
		fontdata.setName("courier");
		fontdata.setHeight(12);
		Font ioFont = new Font(topTabs.getDisplay(), fontdata);
		
		javaTopControl.setSecondaryFont(ioFont);
		interpreterTopControl.setSecondaryFont(ioFont);
		
		fontdata = javaTopControl.getFont().getFontData()[0];
		fontdata.setName("arial");
		fontdata.setHeight(12);
		Font editorFont = new Font(topTabs.getDisplay(), fontdata);
		
		kenyaTopControl.setFont(editorFont);
		javaTopControl.setFont(editorFont);
		interpreterTopControl.setFont(editorFont);
		
		parent.setSize(600, 800);
		
		//Create a subscription to the mediation service
		
		MediationTerms terms = new MediationTerms(
					new KenyaMediatorInput(kenyaTopControl),
					kenyaTopControl,
					javaTopControl
				);
		
		MediationSubscription msub = new MediationSubscription(this, terms);
		subscriptionID = MediationService.getInstance().addSubscription(msub);
		
		JobDispatch.enqueueJob(
    		new KenyaHelperThread(
    				(KenyaPart)getEditor(EditingWindow.KENYATABINDEX),
						(JavaPart)getEditor(EditingWindow.JAVATABINDEX),
						subscriptionID
				)
			);
		
		//after we have setup the tabs its safe to create the menu bar and so forth
		this.setupMenuManager();
		
		return topTabs;
	}
	
	/* (non-Javadoc)
	 * @see mediator.subscription.ISubscriber#subscriptionUpdate(mediator.subscription.ISubscriptionData)
	 */
	public void subscriptionUpdate(final ISubscriptionData data) {
		try {
			if(data==null || data.getContents()==null) {
				throw new RuntimeException("mediator subscription terms violation:" +
						"\nservice update is null");
			}
			Object[] d = data.getContents();
			
			switch(d.length) {
				case 1:
					//according to the terms, the only item is an Exception
					//this case means that the Mediator itself threw an exception
					Exception e = (Exception)d[0];
					e.printStackTrace();
					throw e;
				case 2:
					//this is the normal and desirable case. The contents consist of
					// the ICheckedCode and the IJavaCode that were sent to the
					// corresponding receivers
					IJavaCode javaCode = (IJavaCode)d[1];
					if (javaCode==null) {
						javaSaveAction.setEnabled(false);
						javaSaveAsAction.setEnabled(false);
						if(getTopTabs().isDisposed()) {
							return;
						} else if(Thread.currentThread().equals(
									getTopTabs().getDisplay().getThread())
								) {
							
							if (getTopTabs().getSelectionIndex() != KENYATABINDEX) {
								switchToTab(KENYATABINDEX);
							}
						} else {
							getTopTabs().getDisplay().syncExec(new Runnable() {
								public void run() {
									switchToTab(KENYATABINDEX);
								}
							});
						}
					} else {
						javaSaveAction.setEnabled(true);
						javaSaveAsAction.setEnabled(true);
					}
					safeRedraw();
					break;
				default:
					throw new RuntimeException("mediator subscription terms violation:" +
					"\nservice update contents are invalid");
			}
		} catch(Exception e) {
			throw new KenyaInternalError("" +
					"The Kenya<->Java Mediator has reached an error state." +
					"\nThe details are as follows:" +
					"\n"+e.getMessage());
		}
	}
	
	public void safeRedraw() {
		if (Thread.currentThread()
				.equals(getTopTabs().getDisplay().getThread())) {
			synchronized (this) {
				getTopTabs().redraw();
			}
		} else {
			if (!isVisible())
				return;
			getTopTabs().getDisplay().syncExec(
					new Runnable() {
						public void run() {
							safeRedraw();
						}
					}
				);
		}
	}
	
	protected void handleShellCloseEvent() {
		closeWindow();
	}
	
	private static int counter = 0;
	private static synchronized String getNextClassName() {
    counter++;
    return "Class" + counter;
  }
	
	protected synchronized void setupMenuManager() {
		
		/*
		 * this method is used to build the menu rather than createMenuManager
		 * as there are explicit dependencies between the Actions
		 * and the contents being existent (not null). This method is called
		 * from the createContents method and thus will only be called by
		 * the GUI thread.
		 */
		
		MenuManager bar_menu = this.getMenuBarManager();

		MenuManager file_menu = new MenuManager("&File");
		bar_menu.add(file_menu);
		
		KenyaPart k = (KenyaPart)getEditor(KENYATABINDEX);
		file_menu.add(new NewAction(this));
		file_menu.add(new OpenAction(k));
		file_menu.add(new OpenInNewAction(k));
		file_menu.add(new KenyaSaveAction(k));
		file_menu.add(new KenyaSaveAsAction(k));
		
		JavaPart j = (JavaPart)getEditor(JAVATABINDEX);
		
		javaSaveAction = new JavaSaveAction(j);
		javaSaveAsAction = new JavaSaveAsAction(j);
		
		javaSaveAction.setEnabled(false);
		javaSaveAsAction.setEnabled(false);
		file_menu.add(javaSaveAction);
		file_menu.add(javaSaveAsAction);
		
		file_menu.add(new CloseAction(this));
		file_menu.add(new ExitAction(this));
		
		MenuManager edit_menu = new MenuManager("&Edit");
		bar_menu.add(edit_menu);
		edit_menu.add(new FormatAction(k));
		
		MenuManager view_menu = new MenuManager("&View");
		bar_menu.add(view_menu);
		
		MenuManager editing_fonts_menu = new MenuManager("&Editor Fonts");
		view_menu.add(editing_fonts_menu);
		
		MenuManager editing_fonts_size_menu = new MenuManager("Size");
		
		editing_fonts_menu.add(editing_fonts_size_menu);
		UniqueCheckedManager ucm = new UniqueCheckedManager();
		
		//for font changing actions, use fontTarget
		EditorPart[] elist = new EditorPart[]{
				k,
				j,
				getEditor(INTERPRETERTABINDEX)
			};
		
		EditorPartsFontTarget fontTarget = new EditorPartsFontTarget(elist);
		
		FontChangeAction fontAction = new EditorSmallFontAction(fontTarget);
		ucm.addAction(fontAction);
		editing_fonts_size_menu.add(fontAction);
		
		fontAction = new EditorMediumFontAction(fontTarget);
		ucm.addAction(fontAction);
		editing_fonts_size_menu.add(fontAction);
		
		fontAction = new EditorBigFontAction(fontTarget);
		ucm.addAction(fontAction);
		editing_fonts_size_menu.add(fontAction);
		
		fontAction = new EditorHugeFontAction(fontTarget);
		ucm.addAction(fontAction);
		editing_fonts_size_menu.add(fontAction);
		
		MenuManager editing_fonts_type_menu = new MenuManager("Type");
		editing_fonts_menu.add(editing_fonts_type_menu);
		ucm = new UniqueCheckedManager();
		
		fontAction = new EditorProportionalFontAction(fontTarget);
		ucm.addAction(fontAction);
		editing_fonts_type_menu.add(fontAction);
		
		fontAction = new EditorFixedFontAction(fontTarget);
		ucm.addAction(fontAction);
		editing_fonts_type_menu.add(fontAction);
		
		MenuManager java_io_fonts_menu = new MenuManager("&I/O Fonts");
		view_menu.add(java_io_fonts_menu);
		
		MenuManager java_io_fonts_size_menu = new MenuManager("Size");
		java_io_fonts_menu.add(java_io_fonts_size_menu);
		ucm = new UniqueCheckedManager();
		
		fontAction = new IOSmallFontAction(fontTarget);
		ucm.addAction(fontAction);
		java_io_fonts_size_menu.add(fontAction);
		
		fontAction = new IOMediumFontAction(fontTarget);
		ucm.addAction(fontAction);
		java_io_fonts_size_menu.add(fontAction);
		
		fontAction = new IOBigFontAction(fontTarget);
		ucm.addAction(fontAction);
		java_io_fonts_size_menu.add(fontAction);
		
		fontAction = new IOHugeFontAction(fontTarget);
		ucm.addAction(fontAction);
		java_io_fonts_size_menu.add(fontAction);

		MenuManager java_io_fonts_type_menu = new MenuManager("Type");
		java_io_fonts_menu.add(java_io_fonts_type_menu);
		ucm = new UniqueCheckedManager();

		fontAction = new IOProportionalFontAction(fontTarget);
		ucm.addAction(fontAction);
		java_io_fonts_type_menu.add(fontAction);

		fontAction = new IOFixedFontAction(fontTarget);
		ucm.addAction(fontAction);
		java_io_fonts_type_menu.add(fontAction);
		
		ISwitchActionTarget starget = new EditingWindowSwitchTarget(this);
		SwitchAction sa;
		{
			sa = new SwitchAction(starget, KENYATABINDEX);
			sa.setText("Switch to &Kenya@Ctrl+K");
      sa.setToolTipText("View Kenya code");
      view_menu.add(sa);
		}
		{
			sa = new SwitchAction(starget, JAVATABINDEX);
			sa.setText("Switch to &Java@Ctrl+J");
      sa.setToolTipText("View Java translation");
      view_menu.add(sa);
		}
		{
			sa = new SwitchAction(starget, INTERPRETERTABINDEX);
			sa.setText("Switch to &Debugger@Ctrl+D");
			sa.setToolTipText("View Debugger");
			view_menu.add(sa);
		}
		
		MenuManager help_menu = new MenuManager("&Help");
		bar_menu.add(help_menu);
		help_menu.add(new AboutAction());
		
		//need this line to have these changes reflect on the menu
		bar_menu.updateAll(true);
	}
	
	private void setWindowVisible(boolean isVisible) {
		synchronized (secondaryLock) {
			this.isVisible = isVisible;
		}
	}
	
	private boolean getWindowVisible() {
		synchronized (secondaryLock) {
			return isVisible;
		}
	}
	
	public boolean isVisible() {
		TabFolder topTabs = getTopTabs();
		if (topTabs == null || topTabs.isDisposed() || KenyaUI.isExiting())
			return false;
		
		if (Thread.currentThread().equals(topTabs.getDisplay().getThread())) {
			synchronized (this) {
				setWindowVisible(topTabs.isVisible());
			}
		} else {
			topTabs.getDisplay().syncExec(runIsVisible);
		}
		return getWindowVisible();
	}
	
	public synchronized TabFolder getTopTabs() {
		return topTabs;
	}
	
	public void closeWindow() {
		if (Thread.currentThread()
				.equals(getTopTabs().getDisplay().getThread())) {
			synchronized (this) {
				if (unloadKenyaFile()) {
					close();
					//unsubscribe from translation
					MediationService.getInstance().cancelSubscription(subscriptionID);
				}
			}
		} else {
			if (!isVisible())
				return;
			getTopTabs().getDisplay().syncExec(runCloseWindow);
		}
	}
	
	public boolean unloadKenyaFile() {
		if (getEditor(KENYATABINDEX).isModified()) {
			SavePromptAction spa = new SavePromptAction((KenyaPart)getEditor(KENYATABINDEX));
			spa.run();
			if (spa.isCancelled())
				return false;
		}
		return true;
	}
	
	public synchronized IButtonManager getButtonManager() {
		return buttonManager;
	}
	
	public synchronized SelectionListener getTabSelectionListener() {
		return tabSelectionListener;
	}
	
	public synchronized EditorPart getEditor(int idx) {
		return ((EditorPart)topTabs.getItem(idx).getControl());
	}
	
	public synchronized void switchToTab(int tab) {
		if (tab == KENYATABINDEX || tab == JAVATABINDEX
				|| tab == INTERPRETERTABINDEX) {
			if (topTabs != null && getTopTabs().getSelectionIndex() != tab) {
				getTopTabs().setSelection(tab);
				Event e = new Event();
				e.widget = topTabs;
				e.item = getEditor(tab);
				tabSelectionListener.widgetSelected(new SelectionEvent(e));
			}
		}
	}
	
	public static String[] parseIntoArgs(String input) {
		Pattern p = Pattern.compile("[ ]*([^ ]+)[ ]*");
		Matcher m = p.matcher(input);
		m.reset();
		List args = new ArrayList();
		while (m.find()) {
			args.add(m.group(1));
		}
		return (String[]) args.toArray(new String[args.size()]);
	}
	
	
	public void safeSetStatus(final String message) {
		if (Thread.currentThread()
				.equals(getTopTabs().getDisplay().getThread())) {
			synchronized (this) {
				setStatus(message);
			}
		} else {
			if (!isVisible())
				return;
			getTopTabs().getDisplay().syncExec(new Runnable() {
	
				public void run() {
					safeSetStatus(message);
				}
			});
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.gui.editor.event.IModificationListener#objectModified(kenya.gui.editor.event.ModificationEvent)
	 */
	public void objectModified(ModificationEvent e) {
		if(e.source!=null && e.source.equals(getEditor(KENYATABINDEX))) {
			KenyaPart kenya = (KenyaPart)e.source;
			boolean modified = e.modified;
			FileLoaderSaver kenyaFileLoaderSaver = kenya.getKenyaFile();;
			if (!kenya.isModified() && modified) {
				if (kenyaFileLoaderSaver != null
						&& kenyaFileLoaderSaver.getPath() != null
						&& !kenyaFileLoaderSaver.getPath().equals("")) {
					getShell().setText(
							"Kenya IDE: " + kenyaFileLoaderSaver.getPath()
									+ " (modified)");
				} else {
					getShell().setText("Kenya IDE (modified)");
				}
			} else if (kenya.isModified() && !modified) {
				if (kenyaFileLoaderSaver != null
						&& kenyaFileLoaderSaver.getPath() != null
						&& !kenyaFileLoaderSaver.getPath().equals("")) {
					getShell().setText(
							"Kenya IDE: " + kenyaFileLoaderSaver.getPath());
				} else {
					getShell().setText("Kenya IDE");
				}
			}
		} else {
			//ignore
		}
	}
	
	/*
	 *  (non-Javadoc)
	 * @see kenya.gui.editor.event.IModificationListener#objectAboutToBeModified(kenya.gui.editor.event.ModificationEvent)
	 */
	public void objectAboutToBeModified(ModificationEvent e) {
		if(e.source instanceof KenyaPart) {
			KenyaPart k = (KenyaPart)e.source;
			if(k!=null && k.equals(getEditor(KENYATABINDEX))) {
				if(k.getKenyaFile()!= null && k.getKenyaFile().equals(e.details)) {
					//switchToTab(KENYATABINDEX);
				}
			}
		}
		
	}
	
}