/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on Aug 10, 2004
 *
 */
package kenya.sole.ui.editor.java;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintWriter;

import kenya.sole.ui.core.KenyaUI;
import kenya.sole.ui.editor.JavaPart;
import kenya.sole.ui.util.StdOutWriter;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Text;
import org.wellquite.kenya.stackMachine.misc.AbstractJob;
import org.wellquite.kenya.stackMachine.misc.IJob;
import org.wellquite.kenya.stackMachine.misc.JobDispatch;

/**
 * @author Matthew Sackman
 * @version 1
 */
public class JavaExecuteButtonSelectionListener extends SelectionAdapter {

    private static final String outputDir = System.getProperty("user.home")
    + File.separatorChar + ".kenyaTmp";

    private static final IJob deleteOnExitJob = new AbstractJob() {

        private void delDir(File dir) {
            if (dir.isFile()) {
                dir.delete();
            } else {
                File [] children = dir.listFiles();
                for (int idx = 0; idx < children.length; idx++) {
                    delDir(children[idx]);
                }
                dir.delete();
            }
        }
        
        public void execute() {
            File output = new File(outputDir);
            File [] children = output.listFiles();
            for (int idx = 0; idx < children.length; idx++) {
                delDir(children[idx]);
            }
        }
    };
    
    private static final int STDOUTCOLOR = SWT.COLOR_BLUE;

    private static final int STDERRCOLOR = SWT.COLOR_RED;

    private final JavaPart java;

    private final StyledText stdOutText;

    private final Text stdInText;

    private final Button executeJavaButton;

    private final Button eofButton;

    private final Button terminateButton;

    private final String delimiter;

    private KeyListener keyListener = null;

    private SelectionListener eofListener = null;

    private final Runnable enableStuff;

    private final Runnable disableStuff;

    static {
        File output = new File(outputDir);
        if (!output.exists()) {
            if (!output.mkdir()) {
                System.err.println("Unable to create dir " + outputDir);
                KenyaUI.exit();
            }
        } else if (!output.isDirectory()) {
            System.err.println(outputDir + " should be a dir. Please remove "
                    + outputDir + " and restart Kenya.");
        }

        KenyaUI.addExitListener(deleteOnExitJob);
    }

    public JavaExecuteButtonSelectionListener(JavaPart java) {
        this.java = java;
        stdOutText = java.getJavaStdOutText();
        stdInText = java.getJavaStdInText();
        executeJavaButton = java.getJavaExecuteButton();
        eofButton = java.getJavaEOFButton();
        terminateButton = java.getJavaTerminateButton();
        delimiter = stdOutText.getLineDelimiter();

        enableStuff = new Runnable() {

            public void run() {
                eofButton.setEnabled(true);
                terminateButton.setEnabled(true);
                terminateButton.setText("Terminate");
                stdInText.setEnabled(true);
                stdInText.setFocus();
            }
        };

        disableStuff = new Runnable() {

            public void run() {
                terminateButton.setEnabled(false);
                terminateButton.setText("Terminate");
                eofButton.setEnabled(false);
                stdInText.setText("");
                stdInText.setEnabled(false);
                executeJavaButton.setText("E&xecute Java");
                executeJavaButton.setSelection(false);
                executeJavaButton.setEnabled(true);
            }
        };

    }

    public void widgetSelected(SelectionEvent e) {
        executeJavaButton.setEnabled(false);
        terminateButton.setEnabled(false);
        if (java.getJavaCode() == null) {
            executeJavaButton.setText("E&xecute Java");
        } else {
            executeJavaButton.setText("Executing...");
            try {
                final PipedOutputStream stdInPOS = new PipedOutputStream();
                final BufferedReader stdInReader = new BufferedReader(
                        new InputStreamReader(new PipedInputStream(stdInPOS)));
                final PrintWriter stdInWriter = new PrintWriter(stdInPOS);

                final PipedInputStream stdOutPIS = new PipedInputStream();
                final BufferedWriter stdOutWriter = new BufferedWriter(
                        new OutputStreamWriter(new PipedOutputStream(stdOutPIS)));

                final PipedInputStream stdErrPIS = new PipedInputStream();
                final BufferedWriter stdErrWriter = new BufferedWriter(
                        new OutputStreamWriter(new PipedOutputStream(stdErrPIS)));

                if (keyListener != null)
                        stdInText.removeKeyListener(keyListener);
                keyListener = new KeyAdapter() {

                    public void keyReleased(KeyEvent e) {
                        if (e.character == SWT.CR) {
                            e.doit = false;
                            String text = stdInText.getText() + delimiter;
                            stdInText.setText("");

                            stdOutText.append(text);
                            StyleRange sr = new StyleRange();
                            sr.start = stdOutText.getCharCount()
                                    - text.length();
                            sr.length = text.length();
                            sr.foreground = stdOutText.getDisplay()
                                    .getSystemColor(SWT.COLOR_DARK_GREEN);
                            stdOutText.setStyleRange(sr);
                            stdOutText
                                    .setCaretOffset(stdOutText.getCharCount());
                            stdOutText.setSelection(stdOutText.getCharCount());
                            stdOutText.showSelection();

                            stdInWriter.write(text);
                            stdInWriter.flush();
                        }
                    }
                };
                stdInText.addKeyListener(keyListener);

                if (eofListener != null)
                        eofButton.removeSelectionListener(eofListener);

                eofListener = new SelectionAdapter() {

                    public void widgetSelected(SelectionEvent e) {
                        stdInWriter.close();
                        stdInText.setEnabled(false);
                        eofButton.setEnabled(false);
                    }
                };
                eofButton.addSelectionListener(eofListener);

                final AbstractJob stdOutRunner = new StdOutWriter(
                        new InputStreamReader(stdOutPIS), stdOutText,
                        STDOUTCOLOR);

                final AbstractJob stdErrRunner = new StdOutWriter(
                        new InputStreamReader(stdErrPIS), stdOutText,
                        STDERRCOLOR);

                final AbstractJob launchJava = new AbstractJob() {

                    public void execute() {
                        try {
                            executeJavaButton.getDisplay()
                                    .syncExec(enableStuff);
                            java.safeSetStatus("Java execution starting.");
                            java.getJavaCode().execute(outputDir,
                                    java.getJavaClassName(), stdInReader,
                                    new BufferedWriter(stdInWriter),
                                    stdOutWriter, stdErrWriter,
                                    java.getJavaArgs());
                            stdOutWriter.close();
                            stdErrWriter.close();
                            stdInWriter.close();
                            java.setJavaRunning(false);
                            executeJavaButton.getDisplay().syncExec(
                                    disableStuff);
                            File javaFile = new File(outputDir, java
                                    .getJavaClassName()
                                    + ".java");
                            if (javaFile.isFile()) {
                                javaFile.delete();
                            }
                            java.safeSetStatus("Execution of Java complete.");
                        } catch (IOException e1) {
                            e1.printStackTrace();
                        }
                    }
                };

                stdOutText.setText("");
                stdOutText.setStyleRanges(new StyleRange[0]);

                java.setJavaRunning(true);

                JobDispatch.enqueueJob(stdOutRunner);
                JobDispatch.enqueueJob(stdErrRunner);
                JobDispatch.enqueueJob(launchJava);
            } catch (IOException e2) {
                e2.printStackTrace();
            }

        }
    }

}