/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/
/*
 * Created on 23-Jul-2004
 */
package kenya.sole.ui.util;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import kenya.sole.ui.editor.EditorPart;

import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;

/**
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public abstract class AbstractHighlighter implements ICustomPaintListener {

    private final Color seaColor;

    private final Color backgroundColor;

    private final EditorPart editor;

    private final Set rows = new HashSet();

    private static final ISeaDrawer seaDrawer = new DefaultSeaDrawer();

    private final Set highlights = new HashSet();

    protected AbstractHighlighter(EditorPart editor, Color seaColor,
            Color background) {
        this.editor = editor;
        this.seaColor = seaColor;
        this.backgroundColor = background;
    }

    /*
     * (non-Javadoc)
     * 
     * @see kenya.gui.editor.ICustomPaintListener#addHighlight(int, int)
     */
    public synchronized void addHighlight(int offset, int length, int lineNumber) {
        highlights.add(new Point(offset, length));
        rows.add(new Integer(lineNumber));
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.swt.events.PaintListener#paintControl(org.eclipse.swt.events.PaintEvent)
     */
    public void paintControl(PaintEvent e) {
        GC gc = e.gc;
        if (!editor.isVisible()) return;

        StyledText text = editor.getEditorTextWidget();

        Set highlightsCopy = null;

        synchronized (this) {
            highlightsCopy = new HashSet(highlights);
        }

        if (highlightsCopy != null) {
            Iterator it = highlightsCopy.iterator();
            while (it.hasNext()) {
                Point point = (Point) it.next();

                int startChar = point.x;
                if (startChar >= text.getCharCount())
                        startChar = text.getCharCount() - 1;
                Point start = text.getLocationAtOffset(startChar);

                int endChar = point.x + point.y;
                if (endChar >= text.getCharCount())
                        endChar = text.getCharCount() - 1;
                Point end = text.getLocationAtOffset(endChar);

                int baseline = start.y + text.getLineHeight() - 1;
                start.y = baseline;
                end.y = baseline;

                if (e.x < end.x && e.x + e.width >= start.x && e.y < baseline
                        && e.y + e.height >= baseline) {
                    seaDrawer.drawLine(gc, start.x, baseline, end.x, baseline,
                            seaColor);
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see kenya.gui.editor.ICustomPaintListener#clearHighlights()
     */
    public synchronized void clearHighlights() {
        highlights.clear();
        rows.clear();
    }

    /*
     * (non-Javadoc)
     * 
     * @see kenya.gui.editor.ICustomPaintListener#getBackgroundColor()
     */
    public Color getBackgroundColor() {
        return backgroundColor;
    }

    /*
     * (non-Javadoc)
     * 
     * @see kenya.gui.editor.ICustomPaintListener#isHighlightOnLine(int)
     */
    public synchronized boolean isHighlightOnLine(int line) {
        if (rows.contains(new Integer(line))) return true;
        return false;
    }
}