<?xml version="1.0"?>

<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:java="http://xml.apache.org/xslt/java"
                xmlns:bpws="http://schemas.xmlsoap.org/ws/2002/07/business-process/"
                xmlns:pu="class:com.ibm.cs.bpws.parser.ParserUtils"
                version="1.0">

<xsl:output method="xml" indent="yes"/>

<xsl:param name="process"/>

<!-- ====================================================================== -->
<!-- utility functions -->

<!-- parse correlations -->
<xsl:template name="parse-correlations">
  <xsl:param name="child"/>
  <xsl:param name="correlations"/>
  <!-- handle correlations, if asked to do so. Note that the pattern 
       attribute below is present in the invoke case.
        <correlations>?
           <correlation set="ncname" initiation="yes|no" 
                        pattern="in|out|out-in">+
        </correlations>
  -->
  <xsl:variable name="cs"
                select="java:com.ibm.cs.bpws.model.CorrelationsImpl.new ()"/>
  <xsl:if test="java:setCorrelations ($child, $cs)"/>
  <xsl:for-each select="$correlations/bpws:correlation">
    <xsl:variable name="c" 
                 select="java:com.ibm.cs.bpws.model.CorrelationImpl.new ()"/>
    <xsl:if test="@set">
      <xsl:if test="java:setSet ($c, @set)"/>
    </xsl:if>
    <xsl:if test="@initiation">
      <xsl:choose>
        <xsl:when test="@initiation='yes'">
          <xsl:if test="java:setInitiation ($c, true ())"/>
        </xsl:when>
        <xsl:when test="@initiation='no'">
          <xsl:if test="java:setInitiation ($c, false ())"/>
        </xsl:when>
      </xsl:choose>
    </xsl:if>
    <xsl:if test="@pattern">
      <xsl:if test="java:setPattern ($c, @pattern)"/>
    </xsl:if>
    <xsl:if test="java:addCorrelation ($cs, $c)"/>
  </xsl:for-each>
</xsl:template>

<!-- do standard stuff for every activity -->
<xsl:template name="parse-standard-stuff">
  <xsl:param name="parent"/>
  <xsl:param name="child"/>
  <xsl:param name="current-element"/>
  <xsl:param name="parseCorrelations" select="false()"/>

  <!-- insert child into parent -->
  <xsl:choose>
    <xsl:when test="pu:isParentFlowOrSequence ($parent)">
      <xsl:if test="java:addActivity ($parent, $child)"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:if test="java:setActivity ($parent, $child)"/>
    </xsl:otherwise>
  </xsl:choose>

  <!-- set child's name if given -->
  <xsl:if test="$current-element/@name">
    <xsl:if test="java:setName ($child, $current-element/@name)"/>
  </xsl:if>
 
  <!-- set the suppress join fault bit if given -->
  <xsl:if test="$current-element/@suppressJoinFailure">
    <xsl:choose>
      <xsl:when test="$current-element/@suppressJoinFailure='yes'">
        <xsl:if test="java:setSuppressJoinFailure ($child, true ())"/>
      </xsl:when>
      <xsl:when test="$current-element/@suppressJoinFailure='no'">
        <xsl:if test="java:setSuppressJoinFailure ($child, false ())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unrecognized value '</xsl:text>
          <xsl:value-of select="$current-element/@suppressJoinFailure"/>
          <xsl:text>' for "suppressJoinFailure" attribute of &lt;process&gt;.
          </xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>

  <!-- set join condition if set -->
  <xsl:if test="$current-element/@joinCondition"> 
    <xsl:if test="java:setJoinCondition ($child,
                                         $current-element/@joinCondition)"/>
  </xsl:if>

  <!-- deal with links that are sourced here -->
  <xsl:for-each select="bpws:source">
    <xsl:variable name="source" 
                  select="java:com.ibm.cs.bpws.model.SourceImpl.new ()"/>
    <xsl:variable name="link" 
                  select="java:com.ibm.cs.bpws.model.LinkImpl.new ()"/>
    <xsl:if test="java:setLink ($source, $link)"/>
    <xsl:if test="java:addSource ($child, $source)"/>
    <xsl:if test="@linkName">
      <xsl:if test="java:setName ($link, @linkName)"/>
    </xsl:if>
    <xsl:if test="@transitionCondition">
      <xsl:if test="java:setTransitionCondition ($source,
                                                 @transitionCondition)"/>
    </xsl:if>
  </xsl:for-each>

  <!-- deal with links that are targetted here -->
  <xsl:for-each select="bpws:target">
    <xsl:variable name="target" 
                  select="java:com.ibm.cs.bpws.model.TargetImpl.new ()"/>
    <xsl:variable name="link" 
                  select="java:com.ibm.cs.bpws.model.LinkImpl.new ()"/>
    <xsl:if test="java:setLink ($target, $link)"/>
    <xsl:if test="java:addTarget ($child, $target)"/>
    <xsl:if test="@linkName">
      <xsl:if test="java:setName ($link, @linkName)"/>
    </xsl:if>
  </xsl:for-each>

  <!-- handle correlations, if asked to do so. -->
  <xsl:if test="$parseCorrelations and bpws:correlations">
    <xsl:call-template name="parse-correlations">
      <xsl:with-param name="child" select="$child"/>
      <xsl:with-param name="correlations" select="bpws:correlations"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>
  
<!-- ====================================================================== -->
<!-- parse <containers> -->

<xsl:template match="bpws:containers">
  <xsl:variable name="containers" 
                select="java:com.ibm.cs.bpws.model.ContainersImpl.new ()"/>
  <xsl:if test="java:setContainers ($process, $containers)"/>
  <xsl:for-each select="bpws:container">
    <xsl:variable name="container" 
                  select="java:com.ibm.cs.bpws.model.ContainerImpl.new ()"/>
    <xsl:if test="@name">
      <xsl:if test="java:setName ($container, @name)"/>
    </xsl:if>
    <xsl:if test="@messageType">
      <xsl:if test="java:setMessageType ($container, 
                                         pu:getQNameFromString (., 
                                                              @messageType))"/>
    </xsl:if>
    <xsl:if test="java:addContainer ($containers, $container)"/>
  </xsl:for-each>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <partners> -->

<xsl:template match="bpws:partners">
  <xsl:variable name="partners" 
                select="java:com.ibm.cs.bpws.model.PartnersImpl.new ()"/>
  <xsl:if test="java:setPartners ($process, $partners)"/>
  <xsl:for-each select="bpws:partner">
    <xsl:variable name="partner" 
                  select="java:com.ibm.cs.bpws.model.PartnerImpl.new ()"/>
    <xsl:if test="@name">
      <xsl:if test="java:setName ($partner, @name)"/>
    </xsl:if>
    <xsl:if test="@serviceLinkType">
      <xsl:if test="java:setServiceLinkType ($partner, 
                                             pu:getQNameFromString (., 
                                                          @serviceLinkType))"/>
    </xsl:if>
    <xsl:if test="@myRole">
      <xsl:if test="java:setMyRole ($partner, @myRole)"/>
    </xsl:if>
    <xsl:if test="@partnerRole">
      <xsl:if test="java:setPartnerRole ($partner, @partnerRole)"/>
    </xsl:if>
    <xsl:if test="java:addPartner ($partners, $partner)"/>
  </xsl:for-each>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse:
       <correlationSets>?
         <correlationSet name="ncname" properties="qname-list"/>+
       </correlationSets>
-->

<xsl:template match="bpws:correlationSets">
  <xsl:variable name="css" 
               select="java:com.ibm.cs.bpws.model.CorrelationSetsImpl.new ()"/>
  <xsl:if test="java:setCorrelationSets ($process, $css)"/>
  <xsl:for-each select="bpws:correlationSet">
    <xsl:variable name="cs" 
                select="java:com.ibm.cs.bpws.model.CorrelationSetImpl.new ()"/>
    <xsl:if test="@name">
      <xsl:if test="java:setName ($cs, @name)"/>
    </xsl:if>
    <xsl:if test="@properties">
      <xsl:if test="java:setProperties ($cs,
                                        pu:getPropertyQNameList (.,
                                                               @properties))"/>
    </xsl:if>
    <xsl:if test="java:addCorrelationSet ($css, $cs)"/>
  </xsl:for-each>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse:
  <faultHandlers>?
    <catch faultName="qname"? faultContainer="ncname"?>*
      activity | compensate
    </catch>
    <catchAll>?
      activity | compensate
    </catchAll>
  </faultHandlers>
-->

<xsl:template match="bpws:faultHandlers">
  <xsl:param name="parent"/>
  <xsl:variable name="fh" 
               select="java:com.ibm.cs.bpws.model.FaultHandlersImpl.new ()"/>
  <xsl:if test="java:setFaultHandlers ($parent, $fh)"/>
  <xsl:for-each select="bpws:*">
    <xsl:choose>
      <xsl:when test="local-name () = 'catch'">
        <xsl:variable name="catch"
                    select="java:com.ibm.cs.bpws.model.CatchFaultImpl.new ()"/>
        <xsl:if test="java:addCatchFault ($fh, $catch)"/>
        <xsl:if test="@faultName">
          <xsl:if test="java:setFaultName ($catch,
                                           pu:getQNameFromString (.,
                                                                @faultName))"/>
        </xsl:if>
        <xsl:if test="@faultContainer">
          <xsl:if test="java:setFaultContainer ($catch, 
                                                pu:getContainer ($process,
                                                           @faultContainer))"/>
        </xsl:if>
        <!-- parse my activity -->
        <xsl:apply-templates select="bpws:receive |
                                     bpws:reply |
                                     bpws:invoke |
                                     bpws:throw |
                                     bpws:wait |
                                     bpws:empty |
                                     bpws:sequence |
                                     bpws:switch |
                                     bpws:while |
                                     bpws:pick |
                                     bpws:flow |
                                     bpws:scope | 
                                     bpws:terminate |
                                     bpws:assign | 
                                     bpws:compensate">
          <xsl:with-param name="parent" select="$catch"/>
        </xsl:apply-templates>
      </xsl:when>

      <xsl:when test="local-name () = 'catchAll'">
        <xsl:variable name="catchAll"
                      select="java:com.ibm.cs.bpws.model.CatchAllImpl.new ()"/>
        <xsl:if test="java:setCatchAll ($fh, $catchAll)"/>
        <!-- parse my activity -->
        <xsl:apply-templates select="bpws:receive |
                                     bpws:reply |
                                     bpws:invoke |
                                     bpws:throw |
                                     bpws:wait |
                                     bpws:empty |
                                     bpws:sequence |
                                     bpws:switch |
                                     bpws:while |
                                     bpws:pick |
                                     bpws:flow |
                                     bpws:scope | 
                                     bpws:terminate |
                                     bpws:assign | 
                                     bpws:compensate">
          <xsl:with-param name="parent" select="$catchAll"/>
        </xsl:apply-templates>
      </xsl:when>

      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unexpected element '</xsl:text>
          <xsl:value-of select="local-name ()"/>
          <xsl:text>' found in &lt;faultHandlers&gt;</xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse:
  <compensationHandler>?
    activity | compensate
  </compensationHandler>
-->

<xsl:template match="bpws:compensationHandler"
              name="parse-compensation-handler">
  <xsl:param name="parent"/>
  <xsl:variable name="ch" 
           select="java:com.ibm.cs.bpws.model.CompensationHandlerImpl.new ()"/>
  <xsl:if test="java:setCompensationHandler ($parent, $ch)"/>
  <xsl:apply-templates select="bpws:receive |
                               bpws:reply |
                               bpws:invoke |
                               bpws:throw |
                               bpws:wait |
                               bpws:empty |
                               bpws:sequence |
                               bpws:switch |
                               bpws:while |
                               bpws:pick |
                               bpws:flow |
                               bpws:scope |
                               bpws:terminate |
                               bpws:assign |
                               bpws:compensate">
    <xsl:with-param name="parent" select="$ch"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse:
      <receive partner="ncname" portType="qname" operation="ncname"
               container="ncname"
               createInstance="yes|no"?
               standard-attributes>
        standard-elements
        <correlations>?
           <correlation set="ncname" initiation="yes|no">+
        </correlations>
      </receive>
-->
<xsl:template match="bpws:receive">
  <xsl:param name="parent"/>
  <xsl:variable name="recv" 
                select="java:com.ibm.cs.bpws.model.ReceiveImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$recv"/>
    <xsl:with-param name="current-element" select="."/>
    <xsl:with-param name="parseCorrelations" select="true()"/>
  </xsl:call-template>

  <!-- set the various properties -->
  <xsl:if test="@partner">
    <xsl:if test="java:setPartner ($recv,
                                   pu:getPartner ($process, @partner))"/>
  </xsl:if>
  <xsl:if test="@portType">
    <xsl:if test="java:setPortType ($recv,
                                    pu:getQNameFromString (., @portType))"/> 
  </xsl:if>
  <xsl:if test="@operation">
    <xsl:if test="java:setOperation ($recv, @operation)"/>
  </xsl:if>
  <xsl:if test="@container">
     <xsl:if test="java:setContainer ($recv, 
                                     pu:getContainer ($process, @container))"/>
  </xsl:if>

  <xsl:if test="@createInstance">
    <xsl:choose>
      <xsl:when test="@createInstance='yes'">
        <xsl:if test="java:setCreateInstance ($recv, true ())"/>
      </xsl:when>
      <xsl:when test="@createInstance='no'">
        <xsl:if test="java:setCreateInstance ($recv, false ())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unrecognized value '</xsl:text>
          <xsl:value-of select="@createInstance"/>
          <xsl:text>' for "createInstance" attribute of &lt;receive&gt;.
          </xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <reply> -->

<xsl:template match="bpws:reply">
  <xsl:param name="parent"/>
  <xsl:variable name="reply" 
                select="java:com.ibm.cs.bpws.model.ReplyImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$reply"/>
    <xsl:with-param name="current-element" select="."/>
    <xsl:with-param name="parseCorrelations" select="true()"/>
  </xsl:call-template>

  <!-- set the various properties -->
  <xsl:if test="@partner">
    <xsl:if test="java:setPartner ($reply, pu:getPartner ($process, 
                                                          @partner))"/>
  </xsl:if>
  <xsl:if test="@portType">
    <xsl:if test="java:setPortType ($reply, 
                                    pu:getQNameFromString (., @portType))"/>
  </xsl:if>
  <xsl:if test="@operation">
    <xsl:if test="java:setOperation ($reply, @operation)"/>
  </xsl:if>
  <xsl:if test="@container">
    <xsl:if test="java:setContainer ($reply, 
                                     pu:getContainer ($process, @container))"/>
  </xsl:if>
  <xsl:if test="@faultName">
    <xsl:if test="java:setFaultName ($reply,
                                     pu:getQNameFromString (., @faultName))"/>
  </xsl:if>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <invoke> -->

<xsl:template match="bpws:invoke">
  <xsl:param name="parent"/>
  <xsl:variable name="invoke"
                select="java:com.ibm.cs.bpws.model.InvokeImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$invoke"/>
    <xsl:with-param name="current-element" select="."/>
    <xsl:with-param name="parseCorrelations" select="true()"/>
  </xsl:call-template>

  <!-- set the various properties -->
  <xsl:if test="@partner">
    <xsl:if test="java:setPartner ($invoke, pu:getPartner ($process, 
                                                           @partner))"/>
  </xsl:if>
  <xsl:if test="@portType">
    <xsl:if test="java:setPortType ($invoke,
                                    pu:getQNameFromString (., @portType))"/>
  </xsl:if>
  <xsl:if test="@operation">
    <xsl:if test="java:setOperation ($invoke, @operation)"/>
  </xsl:if>
  <xsl:if test="@inputContainer">
    <xsl:if test="java:setInputContainer ($invoke,
                                          pu:getContainer ($process,
                                                           @inputContainer))"/>
  </xsl:if>
  <xsl:if test="@outputContainer">
    <xsl:if test="java:setOutputContainer ($invoke,
                                           pu:getContainer ($process,
                                                          @outputContainer))"/>
  </xsl:if>

  <!-- parse any nested fault handlers and compensation handler -->
  <xsl:for-each select="bpws:catch | bpws:compensationHandler">
    <xsl:choose>
      <xsl:when test="local-name () = 'catch'">
        <xsl:variable name="catch"
                    select="java:com.ibm.cs.bpws.model.CatchFaultImpl.new ()"/>
        <xsl:if test="java:addCatchFault ($invoke, $catch)"/>
        <xsl:if test="@faultName">
          <xsl:if test="java:setFaultName ($catch,
                                           pu:getQNameFromString (.,
                                                                @faultName))"/>
        </xsl:if>
        <xsl:if test="@faultContainer">
          <xsl:if test="java:setFaultContainer ($catch, 
                                                pu:getContainer ($process,
                                                           @faultContainer))"/>
        </xsl:if>
        <!-- parse my activity -->
        <xsl:apply-templates select="bpws:receive |
                                     bpws:reply |
                                     bpws:invoke |
                                     bpws:throw |
                                     bpws:wait |
                                     bpws:empty |
                                     bpws:sequence |
                                     bpws:switch |
                                     bpws:while |
                                     bpws:pick |
                                     bpws:flow |
                                     bpws:scope | 
                                     bpws:terminate |
                                     bpws:assign | 
                                     bpws:compensate">
          <xsl:with-param name="parent" select="$catch"/>
        </xsl:apply-templates>
      </xsl:when>

      <xsl:when test="local-name () = 'catchAll'">
        <xsl:variable name="catchAll"
                      select="java:com.ibm.cs.bpws.model.CatchAllImpl.new ()"/>
        <xsl:if test="java:setCatchAll ($invoke, $catchAll)"/>
        <!-- parse my activity -->
        <xsl:apply-templates select="bpws:receive |
                                     bpws:reply |
                                     bpws:invoke |
                                     bpws:throw |
                                     bpws:wait |
                                     bpws:empty |
                                     bpws:sequence |
                                     bpws:switch |
                                     bpws:while |
                                     bpws:pick |
                                     bpws:flow |
                                     bpws:scope | 
                                     bpws:terminate |
                                     bpws:assign | 
                                     bpws:compensate">
          <xsl:with-param name="parent" select="$catchAll"/>
        </xsl:apply-templates>
      </xsl:when>

      <xsl:when test="local-name () = 'compensationHandler'">
        <xsl:call-template name="parse-compensation-handler">
          <xsl:with-param name="parent" select="$invoke"/>
        </xsl:call-template>
      </xsl:when>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <throw> -->

<xsl:template match="bpws:throw">
  <xsl:param name="parent"/>
  <xsl:variable name="throw"
                select="java:com.ibm.cs.bpws.model.ThrowImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$throw"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <!-- set the various properties -->
  <xsl:if test="@faultName">
    <xsl:if test="java:setFaultName ($throw,
                                     pu:getQNameFromString (., @faultName))"/>
  </xsl:if>
  <xsl:if test="@faultContainer">
    <xsl:if test="java:setFaultContainer ($throw,
                                          pu:getContainer ($process,
                                                          @faultContainer))"/>
  </xsl:if>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <wait> -->

<xsl:template match="bpws:wait">
  <xsl:param name="parent"/>
  <xsl:variable name="wait" 
                select="java:com.ibm.cs.bpws.model.WaitImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$wait"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <!-- set the various properties -->
  <xsl:if test="@for">
    <xsl:if test="java:setForExpr ($wait, @for)"/>
  </xsl:if>
  <xsl:if test="@until">
    <xsl:if test="java:setUntil ($wait, @until)"/>
  </xsl:if>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <empty> -->

<xsl:template match="bpws:empty">
  <xsl:param name="parent"/>
  <xsl:variable name="empty" 
                select="java:com.ibm.cs.bpws.model.EmptyImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$empty"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <terminate> -->

<xsl:template match="bpws:terminate">
  <xsl:param name="parent"/>
  <xsl:variable name="terminate" 
                select="java:com.ibm.cs.bpws.model.TerminateImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$terminate"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <compensate> -->

<xsl:template match="bpws:compensate">
  <xsl:param name="parent"/>
  <xsl:variable name="compensate" 
                select="java:com.ibm.cs.bpws.model.CompensateImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$compensate"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <xsl:if test="@scope">
    <!-- THE FOLLOWING IS WRONG!!! Need to find the scope .. -->
    <xsl:variable name="scope"
                  select="java:com.ibm.cs.bpws.model.ScopeImpl.new ()"/>
    <xsl:if test="java:setName ($scope, @scope)"/>
    <xsl:if test="java:setScope ($compensate, $scope)"/>
  </xsl:if>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <assign> -->

<xsl:template name="parse-from">
  <xsl:param name="from-elem"/>
  <xsl:param name="from"/>
  <xsl:variable name="fe" select="$from-elem"/>

  <xsl:choose>
    <xsl:when test="$fe/@container">
      <xsl:if test="java:setContainer ($from, 
                                       pu:getContainer ($process,
                                                        $fe/@container))"/>
      <xsl:choose>
        <xsl:when test="$fe/@part"> <!-- container/part/query -->
          <xsl:if test="java:setPart ($from, $fe/@part)"/>
          <xsl:if test="$fe/@query">
            <xsl:if test="java:setQuery ($from, $fe/@query)"/>
          </xsl:if>
        </xsl:when>
        <xsl:when test="$fe/@property"> <!-- container/property -->
          <xsl:if test="java:setProperty ($from, 
                                          pu:getQNameFromString ($fe, 
                                                            $fe/@property))"/>
        </xsl:when>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$fe/@partner">
      <xsl:if test="java:setPart ($from,
                                  pu:getPartner ($process, $fe/@partner))"/>
      <xsl:if test="$fe/@serviceReference">
        <xsl:if test="java:setServiceReference ($from,
                                                $fe/@serviceReference)"/>
      </xsl:if>
    </xsl:when>
    <xsl:when test="$fe/@expression">
      <xsl:if test="java:setExpression ($from, $fe/@expression)"/>
    </xsl:when>
    <xsl:otherwise>
      <!-- literal value case. Note that the setting is done directly
           in Java because Xalan auto converts an Element to a NodeSet;
           meaning can't make the call from here. -->
      <xsl:if test="pu:setLiteralValue ($from, $fe/*[1])"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="parse-to">
  <xsl:param name="to-elem"/>
  <xsl:param name="to"/>
  <xsl:variable name="te" select="$to-elem"/>
  <xsl:choose>
    <xsl:when test="$te/@container">
      <xsl:if test="java:setContainer ($to,
                                       pu:getContainer ($process,  
                                                       $te/@container))"/>
      <xsl:choose>
        <xsl:when test="$te/@part"> <!-- container/part/query -->
          <xsl:if test="java:setPart ($to, $te/@part)"/>
          <xsl:if test="$te/@query">
            <xsl:if test="java:setQuery ($to, $te/@query)"/>
          </xsl:if>
        </xsl:when>
        <xsl:when test="$te/@property"> <!-- container/property -->
          <xsl:if test="java:setProperty ($to, 
                                          pu:getQNameFromString ($te, 
                                                        $te/@property))"/>
        </xsl:when>
      </xsl:choose>
    </xsl:when>
    <xsl:when test="$te/@partner">
      <xsl:if test="java:setPart ($to, 
                                  pu:getPartner ($process, 
                                                 $te/@partner))"/>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<xsl:template name="parse-copy">
  <xsl:param name="assign"/>

  <xsl:variable name="cp"
                select="java:com.ibm.cs.bpws.model.CopyImpl.new ()"/>
  <xsl:if test="java:addCopy ($assign, $cp)"/>
  <xsl:variable name="from"
                select="java:com.ibm.cs.bpws.model.FromImpl.new ()"/>
  <xsl:if test="java:setFrom ($cp, $from)"/>
  <xsl:call-template name="parse-from">
    <xsl:with-param name="from-elem" select="bpws:from"/>
    <xsl:with-param name="from" select="$from"/>
  </xsl:call-template>

  <xsl:variable name="to"
                select="java:com.ibm.cs.bpws.model.ToImpl.new ()"/>
  <xsl:if test="java:setTo ($cp, $to)"/>
  <xsl:call-template name="parse-to">
    <xsl:with-param name="to-elem" select="bpws:to"/>
    <xsl:with-param name="to" select="$to"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="parse-transform">
  <xsl:param name="assign"/>

  <xsl:variable name="tr"
                select="java:com.ibm.cs.bpws.model.TransformImpl.new ()"/>
  <xsl:if test="java:addTransform ($assign, $tr)"/>

  <xsl:variable name="def"
                select="java:com.ibm.cs.bpws.model.DefinitionImpl.new ()"/>
  <xsl:if test="java:setDefinition ($tr, $def)"/>
  <xsl:choose>
    <xsl:when test="bpws:definition/@location">
      <xsl:if test="java:setLocation ($def, bpws:definition/@location)"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:if test="pu:setTransformInlinedDefinition ($def, 
                                                      bpws:definition/*[1])"/>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:variable name="input"
                select="java:com.ibm.cs.bpws.model.InputImpl.new ()"/>
  <xsl:if test="java:setInput ($tr, $input)"/>
  <xsl:for-each select="bpws:input/bpws:from">
    <xsl:variable name="from"
                  select="java:com.ibm.cs.bpws.model.FromImpl.new ()"/>
    <xsl:call-template name="parse-from">
      <xsl:with-param name="from-elem" select="."/>
      <xsl:with-param name="from" select="$from"/>
    </xsl:call-template>
    <xsl:if test="java:addFrom ($input, $from)"/>
  </xsl:for-each>

  <xsl:variable name="result"
                select="java:com.ibm.cs.bpws.model.ResultImpl.new ()"/>
  <xsl:if test="java:setResult ($tr, $result)"/>
  <xsl:for-each select="bpws:result/bpws:to">
    <xsl:variable name="to"
                  select="java:com.ibm.cs.bpws.model.ToImpl.new ()"/>
    <xsl:call-template name="parse-to">
      <xsl:with-param name="to-elem" select="."/>
      <xsl:with-param name="to" select="$to"/>
    </xsl:call-template>
    <xsl:if test="java:addTo ($result, $to)"/>
  </xsl:for-each>
</xsl:template>

<xsl:template match="bpws:assign">
  <xsl:param name="parent"/>
  <xsl:variable name="assign"
                select="java:com.ibm.cs.bpws.model.AssignImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$assign"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <!-- parse all the copy and transform elements -->
  <xsl:for-each select="bpws:copy | bpws:transform" >
    <xsl:choose>
      <xsl:when test="local-name() = 'copy'">
        <xsl:call-template name="parse-copy">
          <xsl:with-param name="assign" select="$assign"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="parse-transform">
          <xsl:with-param name="assign" select="$assign"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <sequence> -->

<xsl:template match="bpws:sequence">
  <xsl:param name="parent"/>
  <xsl:variable name="seq" 
                select="java:com.ibm.cs.bpws.model.SequenceImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$seq"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <xsl:apply-templates select="*">
    <xsl:with-param name="parent" select="$seq"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <switch> -->

<xsl:template match="bpws:case" mode="__switch__">
  <xsl:param name="switch"/>
  <xsl:variable name="case" 
                select="java:com.ibm.cs.bpws.model.CaseClauseImpl.new ()"/>
  <xsl:if test="java:addCaseClause ($switch, $case)"/>

  <xsl:if test="java:setCondition ($case, @condition)"/>
  <xsl:apply-templates select="*"> <!-- should be only one child! -->
    <xsl:with-param name="parent" select="$case"/>
  </xsl:apply-templates>
</xsl:template>

<xsl:template match="bpws:otherwise" mode="__switch__">
  <xsl:param name="switch"/>
  <xsl:variable name="ow" select="java:com.ibm.cs.bpws.model.OtherwiseImpl.new ()"/>
  <xsl:if test="java:setOtherwise ($switch, $ow)"/>

  <xsl:apply-templates select="*"> <!-- should be only one child! -->
    <xsl:with-param name="parent" select="$ow"/>
  </xsl:apply-templates>
</xsl:template>

<xsl:template match="bpws:switch">
  <xsl:param name="parent"/>
  <xsl:variable name="switch" 
                select="java:com.ibm.cs.bpws.model.SwitchBlockImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$switch"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <xsl:apply-templates select="*" mode="__switch__">
    <xsl:with-param name="switch" select="$switch"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <while> -->

<xsl:template match="bpws:while">
  <xsl:param name="parent"/>
  <xsl:variable name="while" 
                select="java:com.ibm.cs.bpws.model.WhileImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$while"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <xsl:if test="@condition">
    <xsl:if test="java:setCondition ($while, @condition)"/>
  </xsl:if>
  
  <xsl:apply-templates select="*">
    <xsl:with-param name="parent" select="$while"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <pick> -->

<xsl:template match="bpws:onMessage" mode="__pick__">
  <xsl:param name="pick"/>
  <xsl:variable name="onMsg" 
                select="java:com.ibm.cs.bpws.model.OnMessageImpl.new ()"/>

  <xsl:if test="java:addOnMessage ($pick, $onMsg)"/>

  <xsl:if test="@partner">
    <xsl:if test="java:setPartner ($onMsg,
                                   pu:getPartner ($process, @partner))"/>
  </xsl:if>
  <xsl:if test="@portType">
    <xsl:if test="java:setPortType ($onMsg,
                                    pu:getQNameFromString (., @portType))"/>
  </xsl:if>
  <xsl:if test="@operation">
    <xsl:if test="java:setOperation ($onMsg, @operation)"/>
  </xsl:if>
  <xsl:if test="@container">
    <xsl:if test="java:setContainer ($onMsg,
                                     pu:getContainer ($process, @container))"/>
  </xsl:if>

  <xsl:if test="bpws:correlations">
    <xsl:call-template name="parse-correlations">
      <xsl:with-param name="child" select="$onMsg"/>
      <xsl:with-param name="correlations" select="bpws:correlations"/>
    </xsl:call-template>
  </xsl:if>

  <xsl:apply-templates select="*"> <!-- should be only one child! -->
    <xsl:with-param name="parent" select="$onMsg"/>
  </xsl:apply-templates>
</xsl:template>

<xsl:template match="bpws:onAlarm" mode="__pick__">
  <xsl:param name="pick"/>
  <xsl:variable name="onAlarm" 
                select="java:com.ibm.cs.bpws.model.OnAlarmImpl.new ()"/>
  <xsl:if test="java:addOnAlarm ($pick, $onAlarm)"/>

  <xsl:if test="@for">
    <xsl:if test="java:setForExpr ($onAlarm, @for)"/>
  </xsl:if>
  <xsl:if test="@until">
    <xsl:if test="java:setUntil ($onAlarm, @until)"/>
  </xsl:if>

  <xsl:apply-templates select="*"> <!-- should be only one child! -->
    <xsl:with-param name="parent" select="$onAlarm"/>
  </xsl:apply-templates>
</xsl:template>

<xsl:template match="bpws:pick">
  <xsl:param name="parent"/>
  <xsl:variable name="pick"
                select="java:com.ibm.cs.bpws.model.PickImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$pick"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <xsl:if test="@createInstance">
    <xsl:choose>
      <xsl:when test="@createInstance='yes'">
        <xsl:if test="java:setCreateInstance ($pick, true ())"/>
      </xsl:when>
      <xsl:when test="@createInstance='no'">
        <xsl:if test="java:setCreateInstance ($pick, false ())"/>
      </xsl:when>
    </xsl:choose>
  </xsl:if>

  <xsl:apply-templates select="*" mode="__pick__">
    <xsl:with-param name="pick" select="$pick"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <flow> -->

<xsl:template match="bpws:links" mode="__flow__">
  <xsl:param name="flow"/>
  <xsl:variable name="links" 
                select="java:com.ibm.cs.bpws.model.LinksImpl.new ()"/>
  <xsl:if test="java:setLinks ($flow, $links)"/>

  <xsl:for-each select="bpws:link">
    <xsl:variable name="link"
                  select="java:com.ibm.cs.bpws.model.LinkImpl.new ()"/>
    <xsl:if test="@name">
      <xsl:if test="java:setName ($link, @name)"/>
    </xsl:if>
    <xsl:if test="java:addLink ($links, $link)"/>
  </xsl:for-each>
</xsl:template>

<xsl:template match="bpws:flow">
  <xsl:param name="parent"/>
  <xsl:variable name="flow" 
                select="java:com.ibm.cs.bpws.model.FlowImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$flow"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <xsl:apply-templates select="bpws:links" mode="__flow__">
    <xsl:with-param name="flow" select="$flow"/>
  </xsl:apply-templates>
 
  <xsl:apply-templates select="*[name() != 'bpws:links']">
    <xsl:with-param name="parent" select="$flow"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <scope> -->

<xsl:template match="bpws:scope">
  <xsl:param name="parent"/>
  <xsl:variable name="scope"
                select="java:com.ibm.cs.bpws.model.ScopeImpl.new ()"/>

  <!-- do standard stuff -->
  <xsl:call-template name="parse-standard-stuff">
    <xsl:with-param name="parent" select="$parent"/>
    <xsl:with-param name="child" select="$scope"/>
    <xsl:with-param name="current-element" select="."/>
  </xsl:call-template>

  <!-- set containerAccessSerializable thing (default of false) -->
  <xsl:if test="@containerAccessSerializable">
    <xsl:choose>
      <xsl:when test="@containerAccessSerializable='yes'">
        <xsl:if test="java:setContainerAccessSerializable ($scope, true ())"/>
      </xsl:when>
      <xsl:when test="@containerAccessSerializable='no'">
        <xsl:if test="java:setContainerAccessSerializable ($scope, false ())"/>
      </xsl:when>
    </xsl:choose>
  </xsl:if>

  <!-- process legal declarations -->
  <xsl:apply-templates select="bpws:faultHandlers |
                               bpws:compensationHandler">
    <xsl:with-param name="parent" select="$scope"/>
  </xsl:apply-templates>

  <!-- parse my activity (last element child) -->
  <xsl:apply-templates select="bpws:receive |
                               bpws:reply |
                               bpws:invoke |
                               bpws:throw |
                               bpws:wait |
                               bpws:empty |
                               bpws:sequence |
                               bpws:switch |
                               bpws:while |
                               bpws:pick |
                               bpws:flow |
                               bpws:scope |
                               bpws:assign |
                               bpws:terminate">
    <xsl:with-param name="parent" select="$scope"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->
<!-- parse <process> -->

<xsl:template match="bpws:process">
  <xsl:if test="@name">
    <xsl:if test="java:setName ($process, @name)"/>
  </xsl:if>
  <xsl:if test="@targetNamespace">
    <xsl:if test="java:setTargetNamespace ($process, @targetNamespace)"/>
  </xsl:if>
  <xsl:if test="@queryLanguage">
    <xsl:if test="java:setQueryLanguage ($process, @queryLanguage)"/>
  </xsl:if>
  <xsl:if test="@expressionLanguage">
    <xsl:if test="java:setExpressionLanguage ($process, @expressionLanguage)"/>
  </xsl:if>
  <xsl:if test="@suppressJoinFailure">
    <xsl:choose>
      <xsl:when test="@suppressJoinFailure='yes'">
        <xsl:if test="java:setSuppressJoinFailure ($process, true ())"/>
      </xsl:when>
      <xsl:when test="@suppressJoinFailure='no'">
        <xsl:if test="java:setSuppressJoinFailure ($process, false ())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unrecognized value '</xsl:text>
          <xsl:value-of select="@suppressJoinFailure"/>
          <xsl:text>' for "suppressJoinFailure" attribute of &lt;process&gt;.
          </xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
  <xsl:if test="@containerAccessSerializable">
    <xsl:choose>
      <xsl:when test="@containerAccessSerializable='yes'">
        <xsl:if test="java:setContainerAccessSerializable ($process,
                                                           true ())"/>
      </xsl:when>
      <xsl:when test="@containerAccessSerializable='no'">
        <xsl:if test="java:setContainerAccessSerializable ($process,
                                                           false ())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unrecognized value '</xsl:text>
          <xsl:value-of select="@containerAccessSerializable"/>
          <xsl:text>' for "containerAccessSerializable" attribute</xsl:text>
          <xsl:text> of &lt;process&gt;.</xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
  <xsl:if test="@enableInstanceCompensation">
    <xsl:choose>
      <xsl:when test="@enableInstanceCompensation='yes'">
        <xsl:if test="java:setEnableInstanceCompensation ($process,
                                                          true ())"/>
      </xsl:when>
      <xsl:when test="@enableInstanceCompensation='no'">
        <xsl:if test="java:setEnableInstanceCompensation ($process,
                                                          false ())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unrecognized value '</xsl:text>
          <xsl:value-of select="@enableInstanceCompensation"/>
          <xsl:text>' for "enableInstanceCompensation" attribute</xsl:text>
          <xsl:text> of &lt;process&gt;.</xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
  <xsl:if test="@abstractProcess">
    <xsl:choose>
      <xsl:when test="@abstractProcess='yes'">
        <xsl:if test="java:setAbstractProcess ($process, true ())"/>
      </xsl:when>
      <xsl:when test="@abstractProcess='no'">
        <xsl:if test="java:setAbstractProcess ($process, false ())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message terminate="yes">
          <xsl:text>Unrecognized value '</xsl:text>
          <xsl:value-of select="@abstractProcess"/>
          <xsl:text>' for "abstractProcess" attribute of &lt;process&gt;.
          </xsl:text>
        </xsl:message>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>

  <!-- process legal declarations -->
  <xsl:apply-templates select="bpws:faultHandlers |
                               bpws:compensationHandler |
                               bpws:containers | 
                               bpws:partners |
                               bpws:correlationSets">
    <xsl:with-param name="parent" select="$process"/>
  </xsl:apply-templates>

  <!-- parse my activity (last element child) -->
  <xsl:apply-templates select="bpws:receive |
                               bpws:reply |
                               bpws:invoke |
                               bpws:throw |
                               bpws:wait |
                               bpws:empty |
                               bpws:sequence |
                               bpws:switch |
                               bpws:while |
                               bpws:pick |
                               bpws:flow |
                               bpws:scope | 
                               bpws:terminate |
                               bpws:assign">
    <xsl:with-param name="parent" select="$process"/>
  </xsl:apply-templates>
</xsl:template>

<!-- ====================================================================== -->

<xsl:template match="/">
  <xsl:apply-templates select="/bpws:process"/>
</xsl:template>

</xsl:stylesheet>
