classdef Price
	%PRICE This class represent the price distribution of a cloud resource
	
	properties
		tsFrom;
		tsTo
		price;
		period; % hours
	end
	
	methods
		function [this] = Price(file, periodHours)
		% PRICE create an instance of Price based on the given file
		% file: CSV file containing price information
			[readDate, readPrice] = textread(file, '%s%f\n','delimiter',','); %#ok<DTXTRD>
			ts = datenum(readDate, 'yyyy-mm-dd HH:MM:ss');
			ts = ts*24; % convert days to hours
			ts = ts - ts(1); % change from absolute to relative seconds
			this.period = periodHours;
			len = length(readPrice)-1;
			this.tsFrom = ts(1:len);
			this.tsTo = ts(2:len+1);
			this.price = readPrice(1:len);
			
			% consider only the given period
			for ii=1:len
				this.tsFrom(ii) = mod(this.tsFrom(ii), this.period);
				this.tsTo(ii) = mod(this.tsTo(ii), this.period);
				
				if this.tsFrom(ii) > this.tsTo(ii)
					this.tsFrom(end+1) = 0;
					this.tsTo(end+1) = this.tsTo(ii);
					this.price(end+1) = this.price(ii);
					this.tsTo(ii) = this.period;
				end
			end
		end
		
		function [avgPrice, varPrice, maxPrice] = at(this, time) 
		% AT: gives the price at the given time
		% time: time to be checked
		% avgPrice: average cost
		% varPrice: variance of the cost
		% maxPrice = maximum price
		
			[avgPrice, varPrice, maxPrice] = this.between(time-1e-5, time+1e-5);
		end
		
		function [avgPrice, varPrice, maxPrice] = between(this, timeFrom, timeTo)
		% BETWEEN: gives statistics about the cost between two times
		% timeFrom: starting time(s)
		% timeTo: ending time(s)
		% avgPrice: average cost(s)
		% varPrice: variance of the cost(s)
		% maxPrice = maximum price(s)
			
			len = numel(timeFrom);
			avgPrice = timeFrom;
			varPrice = timeFrom;
			maxPrice = timeFrom;
			
			for n=1:len
		
				w = zeros(length(this.price),1);
				for ii=1:length(w)
					from = max([timeFrom(n) this.tsFrom(ii)]);
					to = min([timeTo(n) this.tsTo(ii)]);
					w(ii) = max([(to-from) 0]);					
				end

				avgPrice(n) = sum(this.price.*w)/sum(w);
				varPrice(n) = var(this.price, w);
				if ~isnan(avgPrice(n))
					maxPrice(n) = max(this.price(w>0));
				else
					maxPrice(n) = nan;
				end
			end
		end
		
		% SURVIVAL
	end
	
end

