classdef RatesProblem < handle
% RATESPROBLEM
% This class represent the partial problem to find the rates.
% This is used by findOptimalRatesFmincon

	properties (GetAccess = public, SetAccess = private)
		qn; % Queuing network
		constraints; % Constraints on the network
		initQNrates; % Initial queuing network rates
		evaluations; % Number of evaluations
	end
	
	methods (Access = public)
		
		function this = RatesProblem(qn, constraints)
			% This method creates an instance of the problem.
			% qn: queuing network
			% constraints: constraints on the response time
			this.qn = qn;
			this.constraints = constraints;
			this.initQNrates = avgRates(qn);
			this.evaluations = 0;
		end
		
		
		function [c,ceq]=nnlcon(this, x)
			% nnlcon: calculates the value for nonlinear constraints.
			% x: decision variable (rates for the resources of each station)
			% nondelayMx1.
			% c/ceq: non linear constraints for fmincon
			
			delayNodes = getDelayNodes(this.qn);
			qnRates = this.initQNrates;
			qnRates(~delayNodes,:) = this.initQNrates(~delayNodes,:).*repmat(x,1,this.qn.K);
			this.qn = setRates(this.qn, qnRates);
			[avgRT, cdfRTindex, cdfRTvalue] = evaluateRT(this.qn);
			this.evaluations = this.evaluations + 1;
			distance = this.constraints.distance(avgRT, cdfRTindex, cdfRTvalue);
			[u, v] = size(distance);
			
			ceq = [];
			%x
			c = reshape(distance, u*v, 1);
		end

		function f = objfun(~, x)
			% objfun: objective function for the RatesProblem. It is simply the sum
			% of all the rates.
			% x: decision variable (rates for the resources of each station)
			% nondelayMx1.
			% f: sum of all the rates
			f = sum(x);
		end
	end
end
