classdef Refactor
	%REFACTOR This class contains functions for refactoring queuing networks
	
	methods(Static)
		
		function [newQN] = removeClasses(qn, index, classes)
			% DESELECTCLASSES
			% Remove from the given node the possibility to process jobs for the
			% given class(es).
			% INPUT
			% qn: queuing network to be modified
			% index: index of the node that needs class removal
			% classes: indexes of the classes that will be removed
			% OUTPUT
			% newQN: queuing network after the transformation
			
			if index<2 || index>qn.M
				error(['index must be between 2 and ' num2str(qn.M)]);
			end
			
			if any(classes<1) || any(classes>qn.K)
				error(['classes must be between 1 and ' num2str(qn.K)]);
			end
			
			newQN = qn;
			
			for ii=classes
				p = newQN.P(ii:qn.K:end, ii:qn.K:end);
				p(:,index) = 0;
				p(index,:) = 0;
				p = p./repmat(sum(p,2),1,qn.M);
				p(isnan(p)) = 0;
				newQN.P(ii:qn.K:end, ii:qn.K:end) = p;
			end
			
		end
		
		function [newQN] = duplicateCenter(qn, index)
			% DUPLICATECENTER
			% Duplicate a node of the given queuing network.
			% INPUT
			% qn: queuing network to be modified
			% index: index of the node to be duplicated. It cannot be node 1 
			%         (the delay node).
			% OUTPUT
			% newQN: queuing network after the transformation
			
			if index<2 || index>qn.M
				error(['index must be between 2 and ' num2str(qn.M)]);
			end
			
			newQN = qn;
			newQN.rates(index,:) = newQN.rates(index,:)*2;
			newQN = splitStation(newQN,index,[0.5 0.5], [newQN.S(index) ...
																				newQN.S(index)]);
		end
		
		function [newQN] = shareLoad(qn, index1, index2)
			% SHARELOAD
			% Share the load between two nodes of the given queuing network by
			% modifying the routing probabilities.
			
			% INPUT
			% qn: queuing network to be modified
			% index1: index of the first node
			% index2: index of the second node
			% OUTPUT
			% newQN: queuing network after the transformation
			
			newQN = qn;
			
			for ii=1:qn.K
				p = newQN.P(ii:qn.K:end, ii:qn.K:end);
				pIn1 = p(:,index1);
				pIn2 = p(:,index2);
				pOut1 = p(index1,:);
				pOut2 = p(index2,:);
				
				pIn = (pIn1+pIn2)/2;
				pOut = (pOut1+pOut2)/2;
				
				p(:,index1) = pIn;
				p(:,index2) = pIn;
				p(index1,:) = pOut;
				p(index2,:) = pOut;
				
				newQN.P(ii:qn.K:end, ii:qn.K:end) = p;
			end
	
		end
		
		function [newQN] = incrementServers(qn, index, increment)
			% INCREMENTSERVERS
			% Increments the number of servers of one node of a queuing network.
			
			% INPUT
			% qn: queuing network to be modified
			% index: index of the target node
			% increment: optional increment value (default: 1)
			% OUTPUT
			% newQN: queuing network after the transformation	
			newQN = qn;
			
			if index<2 || index>qn.M
				error(['index must be between 2 and ' num2str(qn.M)]);
			end
			
			if nargin < 3
				increment = 1;
			end
			
			newQN.S(index) = newQN.S(index) + increment;
		end
		
		function [newQN] = increaseComputingPower(qn, index, speedup)
			% INCREASECOMPUTINGPOWER
			% Increments the computing power (the rates) of the given
			% queuing network.
			
			% INPUT
			% qn: queuing network to be modified
			% index: index of the target node
			% speedup: percentage of speedup with repsect of the previous node (1
			% means +100%, 0.1 means +10%, -0.1 means -10%)
			% OUTPUT
			% newQN: queuing network after the transformation	
			newQN = qn;
			
			if index<2 || index>qn.M
				error(['index must be between 2 and ' num2str(qn.M)]);
			end
			
			newQN.rates(index,:) = newQN.rates(index,:).*(1+speedup);
		end
		
	end
	
end

