function [rates, demands] = findFeasibleRates(qn, constraints)
	% FINDFEASIBLERATES Find non-optimal rates that satisfy the constraint
	% This considers the case in which the upper bound of the achievable
	% response time is not greater than all the constraints
	% INPUT
	% qn: Queuing network
	% constraints: Quality constraints on the response time
	% OUTPUT
	% resourceRates: Nominal rates of each resource
	% demands: Demands of each resource
	
	% STEP 1: Calculate the lowest acceptable response time from the constraints
	% for each class
	% This is just an aid to reduce the number of algorithm iterations.
	rates = zeros(qn.M, 1);
	initQNrates = avgRates(qn);
	
	minRT = zeros(1, qn.K);
	for ii=1:qn.K
		minRT(1, ii) = min([constraints.maxAverageRT(ii) min(constraints.maxPercentileValueRT(ii))]);
	end
	
	% STEP 2: Calculate the visits and the demands of the queuing network for each class
	visits = avgVisits(qn);
	demands = avgDemands(qn, visits);
	
	% STEP 3: Calculate the total delay of the network for each class, except
	% for the delay of the reference node
	delayNodes = getDelayNodes(qn);
	currentRates = avgRates(qn);
	totalDelay = 1./sum(currentRates(delayNodes,:),1);
	for ii=1:qn.K
		totalDelay(ii) = totalDelay(ii) - 1/currentRates(qn.refNodes(ii),ii);
	end
	
	% STEP 4: Calculate the resource rate for each class such that the upper
	% bound of the response time for each class is lower than the minimum allowed
	% response time.
	% The rate formula is derived from Rk<=NDk+Zk
	rates(delayNodes,1) = 1;
	for ii=1:qn.K
		currentRates = qn.NK(ii)*sum(~delayNodes)*demands(~delayNodes,ii)/(minRT(1,ii)-totalDelay(ii));
		rates(~delayNodes,1) = max([rates(~delayNodes,1) currentRates],[],2);
	end
	
	% STEP 5: Scale the violated constraints solution to make it feasible
	%
	% In the case of multiclass, and on certain response time distributions, 
	% the bounds may be incorrect, so we need to scale-up
	% the rates.
	done = false;
	iterations = 0;
	while ~done
		iterations = iterations + 1;
		currentRates = initQNrates;
		currentRates(~delayNodes,:) = initQNrates(~delayNodes,:).*repmat(rates(~delayNodes, 1),1,qn.K);
		qn = setRates(qn, currentRates);
		[avgRT, cdfRTindex, cdfRTvalue] = evaluateRT(qn);
		verifiedConstraints = constraints.verify(avgRT, cdfRTindex, cdfRTvalue);
		done = all(verifiedConstraints);
		
		if ~done
			%disp('findFeasibleRates() did not find a feasible solution. Doubling the rates.');
			rates(~delayNodes, 1) = 2*rates(~delayNodes, 1);
		end
	end
	
end
