classdef QNValue
	% QNVALUE This class is used to store the value of a queueing network
	% Properties:
	% qn: queueing network
	% avgRT: average response time
	% cdfRTindex: indexes of the response time distribution
	% cdfRTvalue: values of the response time distribution
	% U: utilization (vector qn.M)
	% X: throughput (matrix qn.M X qn.K)
	% rates: average rates of the QN
	% visits: average visits of the QN
	% demands: average demands of the QN
	
	properties
		qn;
		rt;
		avgRT;
		cdfRTindex;
		cdfRTvalue;
		U;
		X;
		rates;
		visits;
		demands;
	end
	
	methods
		function obj = QNValue(qn,avgRT,cdfRTindex,cdfRTvalue,U,X)
			% QNVALUE Create an instance of QNValue class
			% QNValue(qn)
			% QNValue(qn,avgRT,cdfRTindex,cdfRTvalue,U,X)
			%
			% qn: queueing network to evaluate
			%
			% If the other parameters are not provided, then the queuing network
			% is evaluated using the evaluateRT function.
			%
			% avgRT: average response time
			% cdfRTindex: indexes of the response time distribution
			% cdfRTvalue: values of the response time distribution
			% U: utilization (vector qn.M)
			% X: throughput (matrix qn.M X qn.K)
			
			% Check if we have all the parameters
			if nargin<6
				avgRT = nan;
				cdfRTindex = nan;
				cdfRTvalue = nan;
				U = nan;
				X = nan;
			end
			
			obj.qn = qn;
			obj.avgRT = avgRT;
			obj.cdfRTindex = cdfRTindex;
			obj.cdfRTvalue = cdfRTvalue;
			obj.U = U;
			obj.X = X;
			obj.rates = avgRates(qn);
			obj.visits = avgVisits(qn);
			obj.demands = avgDemands(qn,obj.visits);
		end
		
		function evaluatedQNvalue = evaluate(this)
			% EVALUATE - Evaluate the queuing network
			[myAvgRT, myCdfRTindex, myCdfRTvalue, myU, myX] = evaluateRT(this.qn);
			evaluatedQNvalue = this;
			evaluatedQNvalue.avgRT = myAvgRT;
			evaluatedQNvalue.cdfRTindex = myCdfRTindex;
			evaluatedQNvalue.cdfRTvalue = myCdfRTvalue;
			evaluatedQNvalue.U = myU;
			evaluatedQNvalue.X = myX;
		end
			
	end
	

	
end

