#include "stdafx.h"
#include "matrix.h"

#define new DEBUG_NEW

///////////////////////////////////////////////////////////////////////////////
// Complex number methods

// CComplex::MagSquared() const
//
// Return the magnitude squared
double CComplex::MagSquared() const
{
    return Real * Real + Imag * Imag;
}

/* Function no longer used... (use *= as it is more efficient)

// operator*( const CComplex& c1, const CComplex &c2 )
//
// Multiply two complex numbers
CComplex operator*( const CComplex& c1, const CComplex &c2 )
{
    CComplex ret;

    ret.Real = c1.Real * c2.Real - c1.Imag * c2.Imag;
    ret.Imag = c1.Real * c2.Imag + c1.Imag * c2.Real;

    return ret;
}
*/


// CComplex operator+( const CComplex& c1, const CComplex &c2 )
//
// Add two complex numbers
CComplex operator+( const CComplex& c1, const CComplex &c2 )
{
    CComplex ret;

    ret.Real = c1.Real + c2.Real;
    ret.Imag = c1.Imag + c2.Imag;

    return ret;
}

// CComplex operator-( const CComplex& c1, const CComplex &c2 )
//
// returns c1 - c2
CComplex operator-( const CComplex& c1, const CComplex &c2 )
{
    CComplex ret;

    ret.Real = c1.Real - c2.Real;
    ret.Imag = c1.Imag - c2.Imag;

    return ret;
}

// CComplex operator*( const CComplex& c, const double &d )
//
// return c * d
CComplex operator*( const CComplex& c, const double &d )
{
    CComplex ret;

    ret.Real = c.Real * d;
    ret.Imag = c.Imag * d;

    return ret;
}

// CComplex operator+( const CComplex& c, const double &d )
//
// return c + d
CComplex operator+( const CComplex& c, const double &d )
{
    CComplex ret;

    ret.Real = c.Real + d;
    ret.Imag = c.Imag;

    return ret;
}

// CComplex operator-( const CComplex& c, const double &d )
//
// return c - d
CComplex operator-( const CComplex& c, const double &d )
{
    CComplex ret;

    ret.Real = c.Real - d;
    ret.Imag = c.Imag;

    return ret;
}


// CComplex operator*( const double &d, const CComplex& c )
//
// return d * c
CComplex operator*( const double &d, const CComplex& c )
{
    CComplex ret;

    ret.Real = c.Real * d;
    ret.Imag = c.Imag * d;

    return ret;
}

// CComplex operator+( const double &d, const CComplex& c )
//
// return d + c
CComplex operator+( const double &d, const CComplex& c )
{
    CComplex ret;

    ret.Real = c.Real + d;
    ret.Imag = c.Imag;

    return ret;
}

// CComplex operator-( const double &d, const CComplex& c )
//
// return d - c
CComplex operator-( const double &d, const CComplex& c )
{
    CComplex ret;

    ret.Real = d - c.Real;
    ret.Imag = - c.Imag;

    return ret;
}

// CComplex& CComplex::operator=( const double &d )
//
// Create a complex number from a real floating point number
CComplex& CComplex::operator=( const double &d )
{
    Real = d;
    Imag = 0;

    return *this;
}

// CComplex& CComplex::operator=( const double &d )
//
// Create a complex number from a real integer number
CComplex& CComplex::operator=( int i )
{
    Real = double( i );
    Imag = 0;

    return *this;
}

// CComplex& CComplex::operator*=( const double &d )
//
// Multiply ourselves by d
CComplex& CComplex::operator*=( const double &d )
{
    Imag *= d;
    Real *= d;

    return *this;
}

// CComplex& CComplex::operator+=( const double &d )
//
// Add d to ourselves
CComplex& CComplex::operator+=( const double &d )
{
    Real += d;

    return *this;
}

// CComplex& CComplex::operator-=( const double &d )
//
// Subract d from ourselves
CComplex& CComplex::operator-=( const double &d )
{
    Real -= d;

    return *this;
}

// CComplex& CComplex::operator*=( const CComplex &c )
//
// Multiply ourselves by c
CComplex& CComplex::operator*=( const CComplex &c )
{
    double Im = Real * c.Imag + Imag * c.Real;

    Real = Real * c.Real - Imag * c.Imag;
    Imag = Im;

    return *this;
}


// CComplex& CComplex::operator-=( const CComplex &c )
//
// Subract c from ourselves
CComplex& CComplex::operator-=( const CComplex &c )
{
    Real -= c.Real;
    Imag -= c.Imag;

    return *this;
}


// operator==( const CComplex &c1, const CComplex &c2 )
//
// Comparison. return tt iff c1 and c2 are _nearly_ equal
BOOL operator==( const CComplex &c1, const CComplex &c2 )
{
    return FLOAT_EQ( c1.Real, c2.Real ) && FLOAT_EQ( c1.Imag, c2.Imag );
}

// Output functions
ostream& operator<<( ostream &os, const CComplex &c )
{
    if( c.Imag > -0.0001 && c.Imag < 0.0001 )
        return os << c.Real;

    if( c.Real < -0.0001 )
        os << "- " << (c.Real*-1);
    else if( c.Real > 0.0001 )
        os << c.Real;

    if( c.Imag > 0.0 )
        os << " + " << c.Imag;
    else 
        os << " - " << (c.Imag*-1);

    return os << 'i';
}

///////////////////////////////////////////////////////////////////////////////
// Vector stuff

void CBaseComplexVector::Reset()
{
    m_iLength = 0;
}

CBaseComplexVector& CBaseComplexVector::operator=( CBaseComplexVector &cv )
{
    SetLength( cv.Length() );
    for( int i = 0; i < Length(); i++ )
        SetElement( i, cv.GetElement(i) );

    return *this;
}

// void CComplexVector::Zero( )
//
// Set the elements to 0.0
void CBaseComplexVector::Zero( )
{
    for( int i = 0; i < Length(); i++ )
        SetElement(i, CComplex(0.0));
}

void CBaseComplexVector::SetLength( int iLength, BOOL bZero )
{
    m_iLength = iLength;
    if( bZero )
        Zero();
}

void CBaseComplexVector::ScaleElement( int i, const CComplex &cv )
{
    SetElement( i, GetElement( i ) * cv );
}

void CBaseComplexVector::AddToElement( int i, const CComplex &cv )
{
    SetElement( i, GetElement( i ) + cv );
}

CComplex CBaseComplexVector::DotProduct( const CBaseComplexVector &cv1 ) const
{
    CComplex cvReturn = 0;
    for( int i = 0; i < Length(); i++ )
        cvReturn += GetElement( i ) * cv1.GetElement( i );

    return cvReturn;
}

void CBaseComplexVector::Add( const CBaseComplexVector &cv1 )
{
    for( int i = 0; i < Length(); i++ )
        SetElement( i, GetElement( i ) + cv1.GetElement( i ) );
}

CComplex CBaseComplexVector::MagSquared() const
{
    return DotProduct( *this );
}

// CComplexVector::CComplexVector()
//
// default constructor. Don't allocate any memory for the array of complex numbers
CComplexVector::CComplexVector()
: m_pComplex( NULL )
, m_iRealLength( 0 )
{
}

// CComplexVector::CComplexVector( int iLength, BOOL bZero )
//
// Construct a complex vector of length iLength. Set the elements to 0.0 if
// bZero is true.
CComplexVector::CComplexVector( int iLength, BOOL bZero )
: m_pComplex( NULL )
, m_iRealLength( 0 )
{
    SetLength( iLength, bZero );
}

// CComplexVector::CComplexVector( const CComplexVector &cv )
//
// Copy a vector, overwriting our current <null> data
CComplexVector::CComplexVector( const CComplexVector &cv )
: m_pComplex( NULL )
, m_iRealLength( 0 )
{
    m_pComplex = NULL;
    Copy( cv );
}

// CComplexVector::~CComplexVector()
//
// Destroy our data
CComplexVector::~CComplexVector()
{
    Reset();
}

// void CComplexVector::Reset()
//
// Dekete the array of complex numbers. Set our length to 0
void CComplexVector::Reset()
{
    if( m_pComplex ){
        delete [] m_pComplex;
        m_pComplex = NULL;
        m_iRealLength = 0;
        SetLength( 0, FALSE );
    }
}

// void CComplexVector::Copy( const CComplexVector &cv )
//
// Copy from cv, overwriting our current data without deleting it first
void CComplexVector::Copy( const CComplexVector &cv )
{
    ASSERT( cv.Length() > 0 );
    SetLength( cv.Length(), FALSE );
    for( int i = 0; i < Length(); i++ )
        m_pComplex[ i ] = cv.m_pComplex[ i ];
}

// CComplexVector& CComplexVector::operator=( const CComplexVector &cv )
//
// Copy cv
CComplexVector& CComplexVector::operator=( const CComplexVector &cv )
{
    Reset();                            // delete our current vector
    Copy( cv );                         // Copy the new stuff

    return *this;
}

CComplexVector& CComplexVector::operator=( const CBaseComplexVector &cv )
{
    Reset();                            // delete our current vector
    SetLength( cv.Length(), FALSE );
    for( int i = 0; i < cv.Length(); i++ )
        (*this)[i] = cv[i];

    return *this;
}

const CComplex& CComplexVector::GetElement( int i ) const
{
    ASSERT( m_pComplex );
    ASSERT( i >=0 && i < Length() );
    return m_pComplex[i];
}

void CComplexVector::SetElement( int i, const CComplex &cv )
{
    ASSERT( m_pComplex );
    ASSERT( i >=0 && i < Length() );
    m_pComplex[i] = cv;
}

// void CComplexVector::SetLength( int iLength, BOOL bZero )
//
// Set our length to iLength. Set the elements to 0.0 if bZero
// is true
void CComplexVector::SetLength( int iLength, BOOL bZero )
{
    if( iLength > m_iRealLength ){
        Reset();
        if( iLength > 0 )
            m_pComplex = new CComplex[ iLength ];
        m_iRealLength = iLength;
    }
    CBaseComplexVector::SetLength( iLength, bZero );
}

// CComplex operator*( const CComplexVector &cv1, const CComplexVector &cv2 )
//
// Return the 'dot' product of cv1 and cv2
CComplex operator*( const CComplexVector &cv1, const CComplexVector &cv2 )
{
    ASSERT( cv1.Length() > 0 );
    ASSERT( cv1.Length() == cv2.Length() );

    CComplex ret;

    for( int i = 0; i < cv1.Length(); i++ ){
        ret += cv1[i] * cv2[i];
    }

    return ret;
}

// CComplexVector operator+( const CComplexVector &cv1, const CComplexVector &cv2 )
//
// Sum the vectors
CComplexVector operator+( const CComplexVector &cv1, const CComplexVector &cv2 )
{
    ASSERT( cv1.Length() > 0 );
    ASSERT( cv1.Length() == cv2.Length() );

    CComplexVector ret( cv1.Length() );
    for( int i = 0; i < cv1.Length(); i++ )
        ret[i] = cv1[i] + cv2[i];

    return ret;
}

// CComplexVector operator-( const CComplexVector &cv1, const CComplexVector &cv2 )
//
// Return the difference of the two vectors
CComplexVector operator-( const CComplexVector &cv1, const CComplexVector &cv2 )
{
    ASSERT( cv1.Length() > 0 );
    ASSERT( cv1.Length() == cv2.Length() );

    CComplexVector ret( cv1.Length() );
    for( int i = 0; i < cv1.Length(); i++ )
        ret[i] = cv1[i] - cv2[i];

    return ret;
}

// CComplexVector operator*( const CComplex &c, const CComplexVector &cv )
//
// Multiply each element of cv by c
CComplexVector operator*( const CComplex &c, const CComplexVector &cv )
{
    ASSERT( cv.Length() > 0 );

    CComplexVector ret( cv.Length() );
    for( int i = 0; i < cv.Length(); i++ )
        ret[i] = cv[i] * c;

    return ret;
}

// CComplexVector operator*( const CComplexVector &cv, const CComplex &c )
//
// Multiply each element of cv by c
CComplexVector operator*( const CComplexVector &cv, const CComplex &c )
{
    return c * cv;
}


// CComplexVector& CComplexVector::operator*=( const CComplex &c )
//
// Multiply each element of ourselves by c
CComplexVector& CComplexVector::operator*=( const CComplex &c )
{
    ASSERT( m_pComplex );
    ASSERT( Length() > 0 );

    for( int i = 0; i < Length(); i++ ){
        m_pComplex[i] *= c;
    }

    return (*this);
}

// CComplexVector& CComplexVector::operator+=( const CComplexVector &cv )
//
// Add cv to ourselves
CComplexVector& CComplexVector::operator+=( const CComplexVector &cv )
{
    ASSERT( m_pComplex );
    ASSERT( Length() > 0 );
    ASSERT( Length() == cv.Length() );

    for( int i = 0; i < Length(); i++ ){
        m_pComplex[i] += cv.m_pComplex[i];
    }

    return (*this);
}

// CComplexVector& CComplexVector::operator-=( const CComplexVector &cv )
//
// Subtract cv from ourselves
CComplexVector& CComplexVector::operator-=( const CComplexVector &cv )
{
    ASSERT( Length() > 0 );
    ASSERT( Length() == cv.Length() );
    ASSERT( m_pComplex );

    for( int i = 0; i < Length(); i++ ){
        m_pComplex[i] -= cv.m_pComplex[i];
    }

    return (*this);
}

// ostream& operator<<( ostream &os, const CComplexVector &cv )
//
// Dump out data to os
ostream& operator<<( ostream &os, const CComplexVector &cv )
{
    if( !cv.Length() )
        return os << "[ ]";

    os << "[ ";
    int i;

    for( i = 0; i + 1 < cv.Length(); i++ )
        os << cv[i] << ", \t";

    return os << cv[i] << "\t ]";
}

CSparseComplexVector::CSparseComplexVector()
: m_ppComplexVector( NULL )
, m_bZero( FALSE )
, m_complexZero( CComplex( 0.0 ) )
{
}

CSparseComplexVector::~CSparseComplexVector()
{
    Reset();
}

void CSparseComplexVector::Reset()
{
    if( m_ppComplexVector ){
        for( int i = 0; i < Sections(); i++ ){
            if( m_ppComplexVector[i] )
                delete m_ppComplexVector[i];
        }
        delete [] m_ppComplexVector;
        m_ppComplexVector = NULL;
    }
    CBaseComplexVector::Reset();
}

// Copy from cv, overwriting our current data
void CSparseComplexVector::Copy( const CSparseComplexVector &cv )
{
    ASSERT( cv.Length() > 0 );
    SetLength( cv.Length(), FALSE );
    for( int i = 0; i < Sections(); i++ ){
        if( cv.m_ppComplexVector[ i ] ){
            m_ppComplexVector[i] = new CComplexVector( *cv.m_ppComplexVector[i] );
        }
    }
}

// Copy cv
CSparseComplexVector& CSparseComplexVector::operator=( const CSparseComplexVector &cv )
{
    Reset();                            // delete our current vector
    Copy( cv );                         // Copy the new stuff

    return *this;
}

// Copy cv
CSparseComplexVector& CSparseComplexVector::operator=( const CBaseComplexVector &cv )
{
    Reset();                            // delete our current vector

    ASSERT( cv.Length() > 0 );
    SetLength( cv.Length(), FALSE );

    for( int i = 0; i < cv.Length(); i++ ){
        if( cv[i] != m_complexZero )
            SetElement(i, cv[i]);
    }

    return *this;
}

void CSparseComplexVector::ScaleElement( int i, const CComplex &cv )
{
    ASSERT( m_ppComplexVector );
    ASSERT( i >= 0 && i <= Length() );

    if( m_ppComplexVector[Section(i)] ){
        Value(i) *= cv;
    }
}

void CSparseComplexVector::AddToElement( int i, const CComplex &cv )
{
    ASSERT( m_ppComplexVector );
    ASSERT( i >= 0 && i <= Length() );

    LoadSection( Section(i) );
    
    ASSERT( m_ppComplexVector[Section(i)] );

    Value(i) += cv;
}

void CSparseComplexVector::SetLength( int iLength, BOOL bZero )
{
    m_bZero = bZero;
    Reset();

    if( iLength > 0 ){
        int iSections =  Section( iLength - 1 ) + 1;
        m_ppComplexVector = new CComplexVector*[ iSections ];
        for( int i = 0; i < iSections; i++ )
            m_ppComplexVector[i] = NULL;
    }
    CBaseComplexVector::SetLength( iLength, bZero );
}

void CSparseComplexVector::LoadSection( int i )
{
    if( !m_ppComplexVector[i] ){
        m_ppComplexVector[i] = new CComplexVector( ELEMENTS_PER_SECTION, m_bZero );
    }
}

void CSparseComplexVector::SetElement( int i, const CComplex &cv )
{
    ASSERT( m_ppComplexVector );
    ASSERT( i >= 0 && i <= Length() );

    if( !m_ppComplexVector[Section(i)] && cv == m_complexZero ){
        ASSERT( m_bZero );
        return;
    }

    LoadSection( Section(i) );
    
    ASSERT( m_ppComplexVector[Section(i)] );

    Value(i) = cv;
}

void CSparseComplexVector::Zero( )
{
    ASSERT( m_ppComplexVector );

    for( int i = 0; i < Sections(); i++ ){
        if( m_ppComplexVector[i] ){
            delete m_ppComplexVector[i];
            m_ppComplexVector[i] = 0;
        }
    }
    m_bZero = TRUE;
}

int CSparseComplexVector::MemUsage() const
{
    int iSections = 0;
    for( int i = 0; i < Sections(); i++ )
        if( m_ppComplexVector[i] )
            iSections++;

    return iSections * sizeof(CComplexVector *) + iSections * ELEMENTS_PER_SECTION * sizeof CComplex;
}

void CSparseComplexVector::ShrinkMem()
{
    ASSERT( !"Not tested yet" );

    ASSERT( m_ppComplexVector );
    ASSERT( m_bZero );

    for( int i = 0; i < Sections(); i++ ){
        if( m_ppComplexVector[i] ){
            for( int j = 0; j < ELEMENTS_PER_SECTION; j++ )
                if( (*m_ppComplexVector[i])[j] != m_complexZero )
                    break;
            if( j == ELEMENTS_PER_SECTION ){
                delete m_ppComplexVector[i];
                m_ppComplexVector[i] = NULL;
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
// Matrix stuff

// CComplexMatrix::CComplexMatrix()
//
// Construct an empty matrix
CComplexMatrix::CComplexMatrix()
: m_iColumns( 0 )
, m_iRows( 0 )
, m_pRow( NULL )
{
}

// CComplexMatrix::CComplexMatrix( int iRows, int iColumns, BOOL bZero )
//
// Construct a matrix of iRows x iColumns, zero if bZero is true
CComplexMatrix::CComplexMatrix( int iRows, int iColumns, BOOL bZero )
: m_iColumns( iColumns )
, m_iRows( iRows )
{
    ASSERT( iColumns > 0 && iRows > 0 );
    m_pRow = new CComplexVector[ iRows ];

    for( int i = 0; i < m_iRows; i++ )
        m_pRow[i].SetLength( iColumns, bZero );
}

// CComplexMatrix::CComplexMatrix( const CComplexMatrix &cm )
//
// Construct a matrix from cm
CComplexMatrix::CComplexMatrix( const CComplexMatrix &cm )
{
    Copy( cm );
}

// CComplexMatrix::~CComplexMatrix()
//
// Delete our data
CComplexMatrix::~CComplexMatrix()
{
    Reset();
}

// void CComplexMatrix::Reset()
//
// Delete all of the vectors. Set out length to 0x0
void CComplexMatrix::Reset()
{
    if( m_pRow ){
        delete [] m_pRow;
        m_pRow = NULL;
        m_iColumns = m_iRows = 0;
    }
}

// void CComplexMatrix::Copy( const CComplexMatrix &cm )
//
// Copy cm, overwriting any current data without deleting it first
void CComplexMatrix::Copy( const CComplexMatrix &cm )
{
    m_pRow = new CComplexVector[ cm.m_iRows ];

    for( int i = 0; i < cm.m_iRows; i++ )
        m_pRow[i].SetLength( cm.m_iColumns, FALSE );

    m_iRows = cm.m_iRows;
    m_iColumns = cm.m_iColumns;

    for( i = 0; i < m_iRows; i++ )
        m_pRow[ i ] = cm.m_pRow[ i ];
}

// void CComplexMatrix::Zero()
//
// Zero the whole matrix
void CComplexMatrix::Zero()
{
    ASSERT( m_pRow );
    for( int i = 0; i < m_iRows; i++ )
        m_pRow[ i ].Zero();
}

// void CComplexMatrix::SetSize( int iRows, int iColumns, BOOL bZero )
//
// Set out size to iRows x iColumns, zero if bZero
void CComplexMatrix::SetSize( int iRows, int iColumns, BOOL bZero )
{
    ASSERT( iColumns > 0 && iRows > 0 );
    Reset();

    m_pRow = new CComplexVector[ iRows ];

    for( int i = 0; i < iRows; i++ )
        m_pRow[i].SetLength( iColumns, bZero );

    m_iRows = iRows;
    m_iColumns = iColumns;
}

// BOOL CComplexMatrix::IsUnitary( ) const
//
// Return true iff this is a unitary matrix
BOOL CComplexMatrix::IsUnitary( ) const
{
    // Unitary means that the matrix's inverse is equal to its conjugate inverse
    // This involves calculating the inverse - (i.e., writing Gauss Elimination
    // and complex division routines). Quite a bit of work
    //
    // I introduce a (hopefully) sufficient test for unitary matrices,
    // that for each Ax = b, where A is the matrix and x is a unit vector, then b
    // must also be a unit vector. 
    //
    // Assumption: it is sufficient to test x = (1, 0, 0, ..., 0), (0, 1, 0, ..., 0)
    //
    // TODO: proof of above assumption or guartunteed method

    // Obviously has to be square
    ASSERT( Rows() == Columns() );

    for( int i = 0; i < Rows(); i++ ){
        CComplexVector cvBasis( Rows(), TRUE );

        cvBasis[ i ] = 1.0;

        if( fabs( ((*this) * cvBasis).MagSquared().MagSquared() - 1.0 ) > FLOAT_EQUALITY )
            return FALSE;
    }

    return TRUE;
}

// CComplexMatrix& CComplexMatrix::operator=( const CComplexMatrix &cm )
//
// Copy matrix cm
CComplexMatrix& CComplexMatrix::operator=( const CComplexMatrix &cm )
{
    Reset();
    Copy( cm );

    return *this;
}

// CComplexVector CComplexMatrix::Row( int iRow ) const
//
// Return row iRow
CComplexVector CComplexMatrix::Row( int iRow ) const
{
    ASSERT( m_pRow );
    ASSERT( iRow >=0 && iRow < m_iRows );

    return m_pRow[ iRow ];
}

// CComplexVector CComplexMatrix::Column( int iColumn ) const
//
// Return Column iColumn
CComplexVector CComplexMatrix::Column( int iColumn ) const
{
    ASSERT( m_pRow );
    ASSERT( iColumn >= 0 && iColumn < m_iColumns );

    CComplexVector col( m_iRows );

    for( int i = 0; i < m_iRows; i++ )
        col[ i ] = (*this)[i][ iColumn ];

    return col;
}

// void CComplexMatrix::Multiply( CComplexVector& ret, const CComplexVector &cv ) const
//
// Multiply ourselves by vector cv, returning the reult in ret.
void CComplexMatrix::Multiply( CComplexVector& ret, const CComplexVector &cv ) const
{
    ASSERT( Columns() == cv.Length() );
    ASSERT( ret.Length() == cv.Length() );

    /* Old method
    for( int j = 0; j < ret.Length(); j++ )
        ret[j] = 0.0;

    for( int i = 0; i < cv.Length(); i++ )
        for( int j = 0; j < cv.Length(); j++ )
            ret[i] += (*this)[i][j] * cv[j];
     */

    // Slightly more efficient version.
    ASSERT( cv.Length() >= 1 );

    for( int i = 0; i < cv.Length(); i++ )
        ret[i] = (*this)[i][0] * cv[0];

    for( int j = 1; j < cv.Length(); j++ )
        for( i = 0; i < cv.Length(); i++ )
            ret[i] += (*this)[i][j] * cv[j];


}

// CComplexMatrix operator-( const CComplexMatrix &cm1, const CComplexMatrix &cm2 )
//
// Subtract two matrices
CComplexMatrix operator-( const CComplexMatrix &cm1, const CComplexMatrix &cm2 )
{
    ASSERT( cm1.Rows() && cm1.Rows()==cm2.Rows() );
    ASSERT( cm1.Columns() && cm1.Columns()==cm2.Columns() );

    CComplexMatrix ret( cm1.Columns(), cm1.Rows(), FALSE );

    for( int i = 0; i < cm1.Rows(); i++ )
        ret[ i ] = cm1[ i ] - cm2[ i ];

    return ret;
}

// CComplexMatrix operator+( const CComplexMatrix &cm1, const CComplexMatrix &cm2 )
//
// Add two matrices
CComplexMatrix operator+( const CComplexMatrix &cm1, const CComplexMatrix &cm2 )
{
    ASSERT( cm1.Rows() && cm1.Rows()==cm2.Rows() );
    ASSERT( cm1.Columns() && cm1.Columns()==cm2.Columns() );

    CComplexMatrix ret( cm1.Columns(), cm1.Rows(), FALSE );

    for( int i = 0; i < cm1.Rows(); i++ )
        ret[ i ] = cm1[ i ] + cm2[ i ];

    return ret;
}

// CComplexMatrix operator*( const CComplexMatrix &cm1, const CComplexMatrix &cm2 )
//
// Multiply two matrices
CComplexMatrix operator*( const CComplexMatrix &cm1, const CComplexMatrix &cm2 )
{
    ASSERT( cm1.Columns() > 0 && cm2.Columns() > 0 );
    ASSERT( cm1.Rows()    > 0 && cm2.Rows()    > 0 );

    ASSERT( cm1.Columns() == cm2.Rows() );

    CComplexMatrix ret( cm1.Rows(), cm2.Columns(), TRUE );

    for( int i = 0; i < ret.Rows(); i++ )
        for( int j = 0; j < ret.Columns(); j++ )
            for( int k = 0; k < cm1.Columns(); k++ )
                ret[i][j] += cm1[i][k] * cm2[k][j];

    return ret;
}

// CComplexVector operator*( const CComplexMatrix &cm, const CComplexVector &cv )
//
// Multiply cm by vector cv
CComplexVector operator*( const CComplexMatrix &cm, const CComplexVector &cv )
{
    ASSERT( cm.Columns() == cv.Length() );

    CComplexVector ret( cv.Length(), TRUE );

    for( int i = 0; i < cv.Length(); i++ )
        for( int j = 0; j < cv.Length(); j++ )
            ret[i] += cm[i][j] * cv[j];

    return ret;
}

// CComplexMatrix& CComplexMatrix::operator+=( const CComplexMatrix &cm )
//
// Add cm to ourselves
CComplexMatrix& CComplexMatrix::operator+=( const CComplexMatrix &cm )
{
    ASSERT( m_iColumns && m_iColumns == cm.Columns() );
    ASSERT( m_iRows    && m_iRows    == cm.Rows()    );

    for( int i = 0; i < m_iRows; i++ )
        (*this)[i] += cm[i];

    return *this;
}

// CComplexMatrix& CComplexMatrix::operator-=( const CComplexMatrix &cm )
//
// subtract cm from ourselved
CComplexMatrix& CComplexMatrix::operator-=( const CComplexMatrix &cm )
{
    ASSERT( m_iColumns && m_iColumns == cm.Columns() );
    ASSERT( m_iRows    && m_iRows    == cm.Rows()    );

    for( int i = 0; i < m_iRows; i++ )
        (*this)[i] -= cm[i];

    return *this;
}

// CComplexMatrix& CComplexMatrix::operator*=( const CComplexMatrix &cm )
//
// Return ourselves * cm
CComplexMatrix& CComplexMatrix::operator*=( const CComplexMatrix &cm )
{
    // We have to make a copy of our matrix in order to 
    // perform the multiply, so we may as well do this the
    // long way

    (*this) = (*this) * cm;

    return *this;
}

// CComplexMatrix operator*( const CComplexMatrix &cm, const CComplex &c )
//
// Multiply a matrix cm by c
CComplexMatrix operator*( const CComplexMatrix &cm, const CComplex &c )
{
    CComplexMatrix ret( cm.Columns(), cm.Rows(), FALSE );

    for( int i = 0; i < cm.Rows(); i++ )
        ret[ i ] = cm[ i ] * c;

    return ret;
}

// CComplexMatrix operator*( const CComplex &c, const CComplexMatrix &cm )
//
// Multiply a matrix cm by c
CComplexMatrix operator*( const CComplex &c, const CComplexMatrix &cm )
{
    // Call the other operator
    return cm * c;
}

// CComplexMatrix& CComplexMatrix::operator*=( const CComplex &c )
//
// Multiply ourselves by c
CComplexMatrix& CComplexMatrix::operator*=( const CComplex &c )
{
    ASSERT( m_pRow );

    for( int i = 0; i < m_iRows; i++ )
        (*this)[i] * c;

    return *this;
}


// ostream& operator<<( ostream &os, const CComplexMatrix &cm )
//
//
ostream& operator<<( ostream &os, const CComplexMatrix &cm )
{
    for( int i = 0; i < cm.Rows(); i++ )
        os << cm[i] << endl;

    return os;
}

