
#ifndef _METAVERSE_H_
#define _METAVERSE_H_

class CUniverse;
class CCircuit;
class CGate;
class CGate;
class CGateGroup;
class CCalculationThreads;

#include "qlist.h"
#include "matrix.h"

typedef void (*SimCallbackFunc)(int iProgress, void *, CBaseComplexVector *);
// Parameters passed to the simulation
struct SimParams{
    BOOL        m_bSimulateSimply;      // Use simple simulation algorithm?
    BOOL        m_bStoreInitialAmps;    // Keep the initial sim amps for future reference?
	BOOL	    m_bOutputToFile;        // Send output to file?
    const char  *m_szInputFile;         // Input file name, if required
	const char  *m_szOutputFile;        // Output file name, if required
	int		    m_iBitsToOutput;        // Output bits option (all, selected)
	int		    m_iBitsFrom;            // Selected output starting bit, if required
	int		    m_iBitsTo;              // Selected output ending bit, if required
	int		    m_iCircuitInputs;       // Type of input
	int		    m_iCircuitOutputs;      // Type of output
    int         m_iNumberToFactorise;   // Number to factorise if performing factorisation
    int         m_iOutputFormat;        // Format of the output numbers (bin or dec) 
    int         m_iMaxThreads;          // max threads in complex sims.
    int         m_iThreadPriority;      // Priority of said threads
    BOOL        m_bSparseVectors;       // Type of vectors
};

// The metaverse is responsible for running the universes! (ie simuulations)
class CMetaverse : protected SimParams {
    ostream     &m_os;                  // Text output stream

    SimCallbackFunc m_pSimCallback;     // Callback to inform of progress
    void        *m_pID;                 // ID used in callbak

    BOOL        m_bStopSim;             // Will stop simulating when true
    BOOL        m_bSimulating;          // True if we are simulating

    SimParams   m_SimParams;            // Simulation Parameters

    int         m_iRandomNumber;        // Random number required for factorisation

    CCalculationThreads *m_pCalcThreads;
public:
    // Enumerations for SimParams::m_iCircuitInputs
    enum { InputFromDefault = 0, InputFromFile = 1, InputForQFactorisation = 2 };

    // Enumerations for SimParams::m_iCircuitOutputs
    enum { OutputProbabilities = 0, OutputAmplitudes = 1, OutputHistogram = 2, OutputNothing = 3 };

    // Enumerations for SimParams::m_iOutputBits
    enum { OutputAllBits = 0, OutputSelectedBits = 1 };

    // Enumeration for SimParams::m_iOutputFormat;
    enum { OutputBinary = 0, OutputDecimal = 1 };

    // Input / output amplitudes for simulation
public:
    CBaseComplexVector *m_pcomplexInitialAmplitudes;
    CBaseComplexVector *m_pcomplexOutputAmplitudes;

    // Construction / destruction
public:
    CMetaverse( ostream &os );
    ~CMetaverse( );

    // Simulation methods
public:
    void StartSimulation( CCircuit      *pCircuit
                        , SimCallbackFunc pSimCallback
                        , void          *pID 
                        , SimParams     simParams
                        );

    // Stop event set / reset
    void ResetStopEvent(){ m_bStopSim = FALSE; }
    void StopSimulation( ){ m_bStopSim = TRUE; }

    BOOL SimulationStopped(){ return m_bStopSim; }
    BOOL Simulating(){ return m_bSimulating; }

protected:
    // Free memory allocated for simulation amplitudes
    void FreeAmplitudes();

    // Helper methods for the simulation
protected:
    // Amplitude initialisation
    void SetInitialAmplitudes( CBaseComplexVector& vectorAmps, const CCircuit *pCircuit );
	void InitialiseForFactorisation( CBaseComplexVector &cvAmps, const CCircuit *pCircuit );
    void InitialiseFromDefault( CBaseComplexVector &cvAmps, const CCircuit *pCircuit );

    // Simulate - initialises and performs simulation, called by StartSimulation
    void Simulate( CCircuit *pCircuit, CBaseComplexVector &complexOutputProbs );

    // Alternate simulation algorithms - called by Simulate
    void SimulateCircuitSimply( CBaseComplexVector& vectorAmps, CCircuit *pCircuit );
    void SimulateCircuitComplexly( CBaseComplexVector& cvAmps, CCircuit *pCircuit );

    // Partitions a group of circuits in to a list of groups
    void PartitionGroup( const CGateGroup& ggIn
                       , int iMaxBits
                       , CQList<CGateGroup> &listGroupsOut
                       );

    // Called by SimulateCircuitComplexly to perform a simulation on a group
    void SimulateGroup( CBaseComplexVector& cvAmps
                      , CCircuit *pCircuit
                      , const CGateGroup& groups 
                      , int& iWorkDone
                      );

    // Called by Simulate group to simulate a sub group within a group
    void SimulateSubGroup( CBaseComplexVector& cvAmps
                         , CCircuit *pCircuit
                         , const CGateGroup& groups 
                         , int& iWorkDone
                         );


    // Validation of circuit
protected:
    void VerifyCircuitSize( CCircuit *pCircuit );

    // Output methods
protected:
    void OutputResults( CBaseComplexVector &cvAmps, const CCircuit *pCircuit );
    void DumpAmplitdues( const CBaseComplexVector &vectorAmps, int iBits ) const;

private:
    // Test stuff
/*
    void ListTest();
    void CircuitTest();
    void VectorTest();
    void ComplexTest();
    void MatrixTest();
    */
};

#endif