#include "stdafx.h"

#include "qlist.h"
#include "gate.h"
#include "GateGroup.h"


// CGateGroup::CGateGroup( CQList<CGate *>& listGates )
//
// Create a gate group from a list of gates
CGateGroup::CGateGroup( CQList<CGate *>& listGates )
{
    m_listGates = *((CQList<const CGate *>*)&listGates);
    for( int i = 0; i < listGates.Length(); i++ ){
        ComputeTransitiveEnclosure( listGates[i] );
    }
}

// CGateGroup::CGateGroup( const CGate *pGate )
// 
// Create a gate group with only one gate
CGateGroup::CGateGroup( const CGate *pGate )
{
    AddGate( pGate );
}

// CGateGroup::ComputeTransitiveEnclosure( const CGate *pGate )
//
// Compute the transitive enclosure. I.e. if we depend the output of
// gate x and it depends on gate y then note that we also depend on
// gate y
void CGateGroup::ComputeTransitiveEnclosure( const CGate *pGate )
{

    for( int i = 0; i < pGate->InputPins(); i++ ){
        ASSERT( pGate->InputPin( i )->IsConnected() );
        CGate *pPrevious = pGate->InputPin( i )->ConnectedTo()->ParentGate();
        if( m_listDependsOn.Find( pPrevious ) < 0 ){
            m_listDependsOn += pPrevious;
            ComputeTransitiveEnclosure( pPrevious );
        }
    }
}

// void CGateGroup::AddGate( const CGate *pGate )
//
// Add a new gate to the group
void CGateGroup::AddGate( const CGate *pGate )
{
    m_listGates += pGate;

    ComputeTransitiveEnclosure( pGate );
}

// CGateGroup::ExpandGatesToSeparateGroups( CQList<CGateGroup> &listGateGroups ) const
//
// Take the current gates and return a list of groups each with one gate per group.
void CGateGroup::ExpandGatesToSeparateGroups( CQList<CGateGroup> &listGateGroups ) const
{
    listGateGroups.Empty();

    for( int i = 0; i < m_listGates.Length(); i++ ){
        CGateGroup gg;

        gg.AddGate( const_cast<CGate *>(m_listGates[i]) );
        listGateGroups += gg;
    }

    // Sort the group
    listGateGroups.PartialOrderSort();
}

// operator==( const CGateGroup &gg1, const CGateGroup &gg2 )
//
// Two groups are equal if they have the same gates (their list of dependencies
// must also be the same by definition)
BOOL operator==( const CGateGroup &gg1, const CGateGroup &gg2 )
{
    return (gg1.m_listGates == gg2.m_listGates);
}

// Dependance. Return tt iff gg2 depends on gg1
BOOL operator<( const CGateGroup &gg1, const CGateGroup &gg2 )
{
    for( int i = 0; i < gg2.NoDependsOn(); i++ ){
        if( gg1.m_listGates.Find( gg2.DependsOn( i ) ) >= 0 )
            return TRUE;
    }
    return FALSE;
}

// Dependance. Return tt iff gg2 depends on gg1 or they are concurrent
BOOL operator<=( const CGateGroup &gg1, const CGateGroup &gg2 )
{
    for( int i = 0; i < gg2.NoDependsOn(); i++ ){
        if( gg1.m_listGates.Find( gg2.DependsOn( i ) ) >= 0 )
            return TRUE;
    }

    for( i = 0; i < gg1.NoDependsOn(); i++ ){
        if( gg2.m_listGates.Find( gg1.DependsOn( i ) ) >= 0 )
            return FALSE;
    }

    return TRUE;
}


// operator<<( ostream& os, const CGateGroup& gg2 )
//
// Dump the information for debugging purposes.
ostream& operator<<( ostream& os, const CGateGroup& gg2 )
{
    os << "\nGates: ";
    for( int i = 0; i < gg2.NoGates(); i++ )
        os << gg2.Gate(i)->GateNumber() << " ";

    os << ", Dependencies ";
    for( i = 0; i < gg2.NoDependsOn(); i++ )
        os << gg2.DependsOn(i)->GateNumber() << " ";

    return os;
}
