/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import uk.ac.imperial.doc.kenya.builtIns.BuiltInMethodsLoader;
import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.gui.KenyaGUIApplication;
import uk.ac.imperial.doc.kenya.gui.editor.utils.EditingWindowUtilsFactory;
import uk.ac.imperial.doc.kenya.gui.editor.utils.FileLoaderSaver;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterLastPointReachedCatcher;
import uk.ac.imperial.doc.kenya.mediator.Mediator;
import uk.ac.imperial.doc.kenya.mediator.stackMachine.StackMachineFactory;
import uk.ac.imperial.doc.kenya.mediator.util.StreamHandler;
import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.stackMachine.ClosureMiscHelper;
import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.scope.ClosureScope;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IClosureScope;
import uk.ac.imperial.doc.kenya.stackMachine.types.ArrayTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.StringTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;
import uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode;
import uk.ac.imperial.doc.kenya.styleCheckers.IFixSuggestion;
import uk.ac.imperial.doc.kenya.styleCheckers.IStyleCheckResult;
import uk.ac.imperial.doc.kenya.types.IBuiltInMethod;

/**
 * @author toa02, ms02
 * 
 */
public class Kenya implements KenyaGUIApplication, KenyaConsoleApplication {

    private static final String VERSION = new String("@VERSION@");
    
    public String getVersion(){
    	return VERSION;
    }

    public static final boolean DEBUG_OUTPUT = new Boolean(System.getProperty(
            "KENYA_DEBUG", "false")).booleanValue();

    private static boolean exiting = false;

    private static final ExecutorService executor = Executors
            .newFixedThreadPool(16);

    private static final List<Runnable> exitListeners = new ArrayList<Runnable>();

    private static final Runnable exitJob = new Runnable() {

        public void run() {
            List<Runnable> exiters = null;
            synchronized (exitListeners) {
                exiters = new ArrayList<Runnable>(exitListeners);
            }
            for (Runnable exitingJob : exiters) {
                if (null != exitingJob) exitingJob.run();
            }
            executor.shutdown();
            while (!executor.isTerminated()) {
                try {
                    executor.awaitTermination(1, TimeUnit.SECONDS);
                } catch (InterruptedException e) {
                    // fail silently
                    System.exit(0);
                }
            }
            System.exit(0);
        }
    };

    private static final String license = IJavaCode.NEWLINE
            + "   Kenya"
            + IJavaCode.NEWLINE
            + "   Copyright (C) 2004 Tristan Allwood,"
            + IJavaCode.NEWLINE
            + "                 2004 Matthew Sackman"
            + IJavaCode.NEWLINE
            + IJavaCode.NEWLINE
            + "   This program is free software; you can redistribute it and/or"
            + IJavaCode.NEWLINE
            + "   modify it under the terms of the GNU General Public License"
            + IJavaCode.NEWLINE
            + "   as published by the Free Software Foundation; either version 2"
            + IJavaCode.NEWLINE
            + "   of the License, or (at your option) any later version."
            + IJavaCode.NEWLINE
            + IJavaCode.NEWLINE
            + "   This program is distributed in the hope that it will be useful,"
            + IJavaCode.NEWLINE
            + "   but WITHOUT ANY WARRANTY; without even the implied warranty of"
            + IJavaCode.NEWLINE
            + "   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"
            + IJavaCode.NEWLINE
            + "   GNU General Public License for more details."
            + IJavaCode.NEWLINE
            + IJavaCode.NEWLINE
            + "   You should have received a copy of the GNU General Public License"
            + IJavaCode.NEWLINE
            + "   along with this program; if not, write to the Free Software"
            + IJavaCode.NEWLINE
            + "   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA."
            + IJavaCode.NEWLINE + IJavaCode.NEWLINE
            + "   The authors can be contacted by email at toa02@doc.ic.ac.uk"
            + IJavaCode.NEWLINE
            + "                                             ms02@doc.ic.ac.uk"
            + IJavaCode.NEWLINE;

    private static final String help = IJavaCode.NEWLINE
            + "Kenya: 	A teaching language designed by Tristan Allwood and Matthew Sackman (c) 2004."
            + IJavaCode.NEWLINE
            + "        Based upon work previously done by Robert Chatley and others, in 2001"
            + IJavaCode.NEWLINE
            + IJavaCode.NEWLINE
            + "Usage:"
            + IJavaCode.NEWLINE
            + "        kenya [options] kenyaFile [arguments]"
            + IJavaCode.NEWLINE
            + IJavaCode.NEWLINE
            + "If no options are specified, then the Kenya GUI will be used."
            + IJavaCode.NEWLINE
            + IJavaCode.NEWLINE
            + "Any options will cause Kenya to run on the command line, valid options are:"
            + IJavaCode.NEWLINE
            + "-v|--version           Display the Kenya version number"
            + IJavaCode.NEWLINE
            + "-s|--translate         Translate the Kenya program to Java if it is valid"
            + IJavaCode.NEWLINE
            + "-c|--execute <path>    Execute the Kenya program, using Path as a place to save "
            + IJavaCode.NEWLINE
            + "                           the .java and .class file"
            + IJavaCode.NEWLINE
            + "-i|--interpret         Interpret the Kenya program"
            + IJavaCode.NEWLINE
            + "-h|--help              Show this help message."
            + IJavaCode.NEWLINE
            + "-d|--debug             Print out debugging information."
            + IJavaCode.NEWLINE
            + "-x|--stylefix <sep>    Print out versions of the source code created by applying style fixes, each separated by <sep>."      
            + IJavaCode.NEWLINE
            + "-p|--parse             Stop after the parsing stage."
            + IJavaCode.NEWLINE
            + "-f|--typefind          Stop after the type-finding stage."
            + IJavaCode.NEWLINE
            + "-e|--definitionfind    Stop after the definition finding stage."
            + IJavaCode.NEWLINE
            + "-l|--builtinload       Stop after the built-in method loading stage."
            + IJavaCode.NEWLINE
            + "-t|--typecheck         Stop after the type-checking stage."
            + IJavaCode.NEWLINE
            + "-y|--structcheck       Stop after the variable-value checking stage."
            + IJavaCode.NEWLINE
            + "-b|--builtins          Just Dump the names and signatures of the built in methods."
            + IJavaCode.NEWLINE
            + "-n|--license           Show the licensing information for Kenya."
            + IJavaCode.NEWLINE;

    
    private static KenyaGUIApplication kenyaGUI;
    private static KenyaConsoleApplication kenyaConsole; 
    
    /**
     * Main entry point into Kenya.
     */
    public static void main(String[] args) {
    	Kenya kenya = new Kenya();
    	kenyaGUI = kenya;
    	kenyaConsole = kenya;

        Getopt op = new Getopt("Kenya", args, "hdpfeltysc:ibnvx:", new LongOpt[] {
                new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
                new LongOpt("debug", LongOpt.NO_ARGUMENT, null, 'd'),
                new LongOpt("parse", LongOpt.NO_ARGUMENT, null, 'p'),
                new LongOpt("typefind", LongOpt.NO_ARGUMENT, null, 'f'),
                new LongOpt("definitionfind", LongOpt.NO_ARGUMENT, null, 'e'),
                new LongOpt("builtinload", LongOpt.NO_ARGUMENT, null, 'l'),
                new LongOpt("typecheck", LongOpt.NO_ARGUMENT, null, 't'),
                new LongOpt("structcheck", LongOpt.NO_ARGUMENT, null, 'y'),
                new LongOpt("translate", LongOpt.NO_ARGUMENT, null, 's'),
                new LongOpt("execute", LongOpt.REQUIRED_ARGUMENT, null, 'c'),
                new LongOpt("interpret", LongOpt.NO_ARGUMENT, null, 'i'),
                new LongOpt("builtins", LongOpt.NO_ARGUMENT, null, 'b'),
                new LongOpt("license", LongOpt.NO_ARGUMENT, null, 'n'),
                new LongOpt("version", LongOpt.NO_ARGUMENT, null, 'v'),
                new LongOpt("stylefix", LongOpt.REQUIRED_ARGUMENT, null, 'x')});

        int opt;

        int debugOptions = 0;
        boolean translate = false; // Should we translate?
        boolean execute = false; // Should we excute?
        String path = null; // Where should we compile to?
        
        boolean stylefix = false; // Should we apply style fixer?
        String stylefixSeparator = null; // what separates style fix output


        boolean interpret = false; // interpret the thing [ takes precedence ]
        while ((opt = op.getopt()) != -1) {
            switch (opt) {
            case 'h':
                /* Help */
                printHelp();
                System.exit(0);
                break;
            case 'v':
                System.out.println("Kenya version " + VERSION);
                System.exit(0);
                break;
            case 'n':
                printLicense();
                System.exit(0);
                break;
            case 'd':
                /* Debug */
                debugOptions |= Mediator.DBG_MESSAGES;
                break;
            case 'p':
                /* Parse */
                debugOptions |= Mediator.DBG_PARSE;
                break;
            case 'f':
                /* TypeFind */
                debugOptions |= Mediator.DBG_TYPEFIND;
                break;
            case 'e':
                /* Definiton Find */
                debugOptions |= Mediator.DBG_DEFINITIONFIND;
                break;
            case 'l':
                /* Builtin Loader */
                debugOptions |= Mediator.DBG_BUILTINLOAD;
                break;
            case 't':
                /* TypeCheck */
                debugOptions |= Mediator.DBG_TYPECHECK;
                break;
            case 'y':
                /* Structual Check */
                debugOptions |= Mediator.DBG_STRUCTURECHECK;
                break;
            case 's':
                /* Translate */
                translate = true;
                break;
            case 'c':
                /* Compile */
                execute = true;
                path = op.getOptarg();
                break;
            case 'i':
                /* Interpret */
                interpret = true;
                break;
            case 'x':
                /* Print out all style fix variants */
                stylefix = true;
                stylefixSeparator = op.getOptarg();
                break;
            case 'b':
                /* Dump builtin methods */
                dumpBuiltinMethods();
                System.exit(0);
            case '?':
            /* Error message from getopt */
            default:
                /* Not recognised */
                printHelp();
                System.exit(1);
                break;
            }
        }

        int argIndex = op.getOptind();
        if (argIndex < args.length) {
            /* There are file(s) to process */
            if (debugOptions == 0 && !translate && !execute && !interpret && !stylefix) {
                for (int idx = argIndex; idx < args.length; idx++) {
                    String fileName = args[idx];
                    FileLoaderSaver fls = new FileLoaderSaver();
                    fls.setPath(fileName);
                    EditingWindowUtilsFactory.createAndOpenEditorWindow(fls, kenyaGUI);
                }
            } else {
                String[] argsLeft = new String[args.length - argIndex];
                System.arraycopy(args, argIndex, argsLeft, 0, argsLeft.length);
                kenyaConsole.doCommandLine(argsLeft, debugOptions, translate, execute,
                        interpret, path, stylefix, stylefixSeparator);
            }
        } else {
            /* There are no files to process */
            if (debugOptions == 0 && !translate && !execute && !interpret && !stylefix) {
                EditingWindowUtilsFactory.createAndOpenEditorWindow(kenyaGUI);
            } else {
                printError("Command Line Kenya needs files to operate upon.");
            }
        }
    }

    public void doCommandLine(String[] files, int dbg,
            boolean translate, boolean execute, boolean interpret,
            String filePath, boolean stylefix, String stylefixSeparator) {

        final IStackMachine sm = (new StackMachineFactory()).createStackMachine(executor);

        try {
            FileReader fis = new FileReader(files[0]);
            ICheckedCode icc = Mediator.check(fis, dbg, System.out);

            if (icc.isErroredCode()) {
                Iterator<ISourceCodeError> it = icc.getErrors().iterator();
                while (it.hasNext()) {
                    SourceCodeException sce = (SourceCodeException) it.next();
                    System.err.println(sce.getMessage());
                }
                System.out.println(files[0] + ": INVALID CODE.");
            } else {
                System.out.println(files[0] + ": VALID CODE.");

                if (interpret) {
                    runInterpreter(files, sm, icc);
                } else if (translate) {
                    runTranslater(icc);
                } else if (execute) {
                    runJavaExecute(files, filePath, icc);
                } else if (stylefix) {
                    runStylefixer(stylefixSeparator, icc);
                }
            }

        } catch (FileNotFoundException e) {
            System.err.println("File Not Found: " + files[0]);
        } catch (IOException ioe) {
            System.err.println("IOException on: " + files[0]);
        }

        exit();
    }

    private void runJavaExecute(String[] files, String filePath,
            ICheckedCode icc) throws IOException {
        long v = 0;
        String s = "_Command";
        IJavaCode ijc = icc.translate();
        while (!ijc.isValidClassName(s + v)) {
            v++;
        }

        String[] args = new String[files.length - 1];
        System.arraycopy(files, 1, args, 0, args.length);

        final PipedOutputStream stdInPOS = new PipedOutputStream();
        PipedInputStream stdIn = new PipedInputStream(stdInPOS);

        BufferedWriter stdInWriter = new BufferedWriter(
                new OutputStreamWriter(stdInPOS));
        BufferedWriter stdOutWriter = new BufferedWriter(
                new OutputStreamWriter(System.out));
        BufferedWriter stdErrWriter = new BufferedWriter(
                new OutputStreamWriter(System.err));

        final BufferedReader stdInReader = new BufferedReader(
                new InputStreamReader(stdIn));

        final StreamHandler sh = new StreamHandler(
                new BufferedReader(new InputStreamReader(System.in)),
                stdInWriter);

        executor.execute(sh);

        executor.execute(new Runnable() {

            public void run() {
                try {
                    synchronized (sh) {
                        while (!sh.isFinished()) {
                            sh.wait();
                        }
                    }
                    if (uk.ac.imperial.doc.kenya.Kenya.DEBUG_OUTPUT) {
                        System.out
                                .println("stream handler B finished");
                    }
                    stdInPOS.close();
                    if (uk.ac.imperial.doc.kenya.Kenya.DEBUG_OUTPUT) {
                        System.out
                                .println("stream handler BB finished");
                    }
                } catch (InterruptedException ie) {
                    ie.printStackTrace();
                } catch (IOException ioe) {
                    ioe.printStackTrace();
                }

            }
        });

        if (DEBUG_OUTPUT) {
            System.out.println("about to execute");
        }

        boolean res = icc.translate().execute(filePath, s + v,
                stdInReader, stdInWriter, stdOutWriter,
                stdErrWriter, args, executor);

        if (res == false && DEBUG_OUTPUT) {
            System.err.println("Translate & Execute failed");
        }

        System.in.close();
        System.out.flush();
        System.err.flush();

        exit();
    }

    private static void runTranslater(ICheckedCode icc) throws IOException {
        BufferedReader r = new BufferedReader(icc.translate()
                .getPlaceHeldCode());

        String s;
        while ((s = r.readLine()) != null) {
            System.out.println(s);
        }
    }

    private void runInterpreter(String[] files, final IStackMachine sm,
            ICheckedCode icc) {
        IStackMachineInformationProvider ismp = icc
                .getBaseStackMachine();

        InterpreterLastPointReachedCatcher ilprc = new InterpreterLastPointReachedCatcher(
                sm);
        sm.addPositionReachedListener(ilprc);
        sm.setStepMode(false);

        try {
            IClosureScope scope = new ClosureScope();
            ClosureMiscHelper.executeClosureInScope(scope, ismp
                    .getPreInitClosure(), sm);

            IType[] array = new IType[files.length - 1];
            for (int idx = 1; idx < files.length; idx++) {
                array[idx - 1] = StringTypeFactory
                        .createStringType(files[idx]);
            }
            sm.push(ArrayTypeFactory.createArrayType(array));
            sm.invokeMethod(ismp.getEntryPointClass(), ismp
                    .getEntryPoint());
        } catch (Throwable t) {
            sm.getErr().println(
                    t + " at " + ilprc.getLastPointData());
        } finally {
            exit();
        }
    }

    private static void runStylefixer(String separator, ICheckedCode code) {

        List<String> outputCode = new ArrayList<String>(); 
        
        for(IStyleCheckResult res : code.getStyleCheckResults()) {
            for(IFixSuggestion fix : res.getPossibleFixes()) {
                StringWriter sw = new StringWriter();
                PrintWriter pw = new PrintWriter(sw);
                
                pw.println(fix.getFixedSourceCode());
                pw.println("/*");
                pw.println("Style fix result");
                pw.println(res.toString());
                pw.println(fix.getName());
                pw.println(fix.getDescription());
                pw.println("*/");
                
                outputCode.add(sw.toString());
            }
        }

        for (Iterator<String> it = outputCode.iterator(); it.hasNext();) {
            String thecode = it.next()
            ;
            System.out.print(thecode);
            
            if(it.hasNext()) {
                System.out.println(separator);
            }
        }
    }
    
    public static void printError(String message) {
        System.err.println("Kenya:  " + message);
        System.err.println(help);
        System.exit(1);
    }

    public static void printHelp() {
        System.out.println(help);
    }

    public static void printLicense() {
        System.out.println(license);
    }

    public synchronized void exit() {
        if (!exiting) {
            exiting = true;
            new Thread(exitJob).run();
        }
    }

    public synchronized boolean isExiting() {
        return exiting;
    }

    public void addExitListener(Runnable listener) {
        if (listener == null) {
            System.err.println("Attempt made to add a null exit listener.");
            throw new NullPointerException();
        } else {
            synchronized (exitListeners) {
                exitListeners.add(listener);
            }
        }
    }

    public synchronized void removeExitListener(Runnable listener) {
        exitListeners.remove(listener);
    }
    
    private static void dumpBuiltinMethods() {
        System.out.println("Builtin Methods:");
        Set<IBuiltInMethod> builtinMethods = BuiltInMethodsLoader.getBuiltInMethods();
        for (Iterator<IBuiltInMethod> it = builtinMethods.iterator(); it.hasNext();) {
            IBuiltInMethod method = (IBuiltInMethod) it.next();
            System.out.println('<' + method.getTemplateParameters().keySet().toString() + '>' + method.getReturnType().getName() +  ' ' + method.getName() + '(' + method.getMethodParameters().toString() + ')');
        }
    }
}
