/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 01-Jul-2004
 */
package uk.ac.imperial.doc.kenya.builtIns;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import uk.ac.imperial.doc.kenya.types.IBuiltInMethod;
import uk.ac.imperial.doc.kenya.types.KBasicType;
import uk.ac.imperial.doc.kenya.types.KParamType;
import uk.ac.imperial.doc.kenya.types.KType;

/**
 * Takes the IBuiltInMethod interface and adds setters for every getter. Thus
 * aiding the way in which built in method classes construct the methods they
 * support.
 * 
 * @author Matthew Sackman (ms02)
 * @edited by Tristan Allwood (toa02)
 * @version 1
 */
public class BuiltInMethod implements IBuiltInMethod {

    protected static final String[] emptyString = { "" };
    
    private String name = "";
    private KType returnType = KBasicType.getVoid();
    private List<KType> parameters = new LinkedList<KType>();
    private IGetMethodCode methodCode = null;
    private Set<String> reservedClasses = new HashSet<String>();
    private Set<String> imports = new HashSet<String>();
    private static final Map<String,KParamType> emptyMap = Collections
    .unmodifiableMap(new HashMap<String,KParamType>());
    private Map<String,KParamType> templateParameters = emptyMap;
    private IInterpretedMethod smMethod = null;

    // Constructor
    public BuiltInMethod(String name,
    					 KType returnType,
    					 List<KType> params,
    					 IGetMethodCode codeHook,
    					 Set<String> reservedClasses,
    					 Set<String> imports,
    					 IInterpretedMethod method,
    					 Map<String,KParamType> templateParams)
    {
    	// Non-empty parameters
    	setName(name);
    	setReturnType(returnType);
    	setCodeHook(codeHook);
    	setInterpretedMethod(method);
    	
    	// Optional parameters
    	if (params == null) params = Collections.unmodifiableList(new LinkedList<KType>());;
    	setMethodParameters(params);
    	
    	if (reservedClasses == null) reservedClasses = Collections.unmodifiableSet(new HashSet<String>());
    	setReservedClasses(reservedClasses);
    	
    	if (imports == null) imports = Collections.unmodifiableSet(new HashSet<String>());
    	setImports(imports);
    	
    	if (templateParams == null) templateParams = Collections.unmodifiableMap(new HashMap<String,KParamType>());
    	setTemplateParameters(templateParams);
    }
    
    // Comparator
    public synchronized boolean equals(Object obj) {
        if (obj instanceof BuiltInMethod) {
            BuiltInMethod abimobj = (BuiltInMethod) obj;

            return getName().equals(abimobj.getName())
                    && getMethodParameters().equals(
                            abimobj.getMethodParameters());
        } else {
            return false;
        }
    }
    
    // Utility
    public String toString() {
        return getName();
    }
    
    // Accessors
    public synchronized String getName() {return name;}
    public synchronized KType getReturnType() {return returnType;}
    
    public synchronized List<KType> getMethodParameters() {
    	return Collections.unmodifiableList(parameters);
    }
    
    public synchronized Set<String> getReservedClasses() {
        return Collections.unmodifiableSet(reservedClasses);
    }
    
    public synchronized Set<String> getImports() {
        return Collections.unmodifiableSet(imports);
    }
    
    public synchronized IInterpretedMethod getInterpretedMethod() {
        return smMethod;
    }
    
    public synchronized String[] getCode(List<String[]> parameters) {
        if (methodCode == null) {
            return emptyString;
        } else {
            return methodCode.getCode(parameters, getMethodParameters());
        }
    }
    
    public synchronized Map<String,KParamType> getTemplateParameters() {
        return templateParameters;
    }
    
    // Mutators
    public synchronized void setName(String n) {name = n;}
    public synchronized void setReturnType(KType rt) {returnType = rt;}
    
    public synchronized void setMethodParameters(List<KType> params) {
        parameters.clear();
        parameters.addAll(params);
    }

    public synchronized void setCodeHook(IGetMethodCode mC) {
        methodCode = mC;
    }

    public synchronized void setReservedClasses(Set<String> i) {
        reservedClasses.clear();
        reservedClasses.addAll(i);
    }

    public synchronized void setImports(Set<String> i) {
        imports.clear();
        imports.addAll(i);
    }
 
    public synchronized void setTemplateParameters(Map<String,KParamType> tp) {
        if (templateParameters == emptyMap) {
            templateParameters = new HashMap<String,KParamType>();
        } else {
            templateParameters.clear();
        }
        templateParameters.putAll(tp);
    }
    
    public synchronized void setInterpretedMethod(IInterpretedMethod method) {
        smMethod = method;
    }

    // Static functions
    public static String[] concatenate(String[] lhs, String[] middle,
            String[] rhs) {
        String[] res = new String[lhs.length + middle.length + rhs.length];
        System.arraycopy(lhs, 0, res, 0, lhs.length);
        System.arraycopy(middle, 0, res, lhs.length, middle.length);
        System.arraycopy(rhs, 0, res, lhs.length + middle.length, rhs.length);
        return res;
    }

}