/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004
 */
package uk.ac.imperial.doc.kenya.gui.editor;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.window.ApplicationWindow;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;

import uk.ac.imperial.doc.kenya.gui.editor.highlighting.DefaultHighlightingManager;
import uk.ac.imperial.doc.kenya.gui.editor.highlighting.IHighlightingManager;
import uk.ac.imperial.doc.kenya.gui.editor.highlighting.IKenyaHighlightedText;
import uk.ac.imperial.doc.kenya.gui.editor.icons.ButtonManager;
import uk.ac.imperial.doc.kenya.gui.editor.icons.IButtonManager;
import uk.ac.imperial.doc.kenya.gui.editor.indenter.IndentationStrategy;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.AboutAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.AlwaysSaveJavaAutomatically;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.CloseAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.CopyTextAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.CutTextAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.EditorBigFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.EditorFixedFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.EditorHugeFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.EditorMediumFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.EditorProportionalFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.EditorSmallFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.ExitAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.FormatAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.IOBigFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.IOFixedFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.IOHugeFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.IOMediumFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.IOProportionalFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.IOSmallFontAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.ISaveKenyaActionHandler;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.ISaveKenyaContinuation;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.JavaSaveAsAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.KenyaSaveAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.KenyaSaveAsAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.ManuallySaveJava;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.NewAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.OpenAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.OpenInNewAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.PasteTextAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.PromptToSaveJava;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.SavePromptAction;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.SwitchToInterpreter;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.SwitchToJava;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.SwitchToKenya;
import uk.ac.imperial.doc.kenya.gui.editor.menuActions.UniqueCheckedManager;
import uk.ac.imperial.doc.kenya.gui.editor.sourceCodeInformationTable.HumanErrorLabelContentProvider;
import uk.ac.imperial.doc.kenya.gui.editor.sourceCodeInformationTable.IKenyaTextProvider;
import uk.ac.imperial.doc.kenya.gui.editor.sourceCodeInformationTable.ISourceCodeInformationTableManager;
import uk.ac.imperial.doc.kenya.gui.editor.sourceCodeInformationTable.JavacErrorLabelContentProvider;
import uk.ac.imperial.doc.kenya.gui.editor.sourceCodeInformationTable.KenyaErrorContentProvider;
import uk.ac.imperial.doc.kenya.gui.editor.sourceCodeInformationTable.SourceCodeInformationTableManager;
import uk.ac.imperial.doc.kenya.gui.editor.utils.CarrotPainter;
import uk.ac.imperial.doc.kenya.gui.editor.utils.EditingWindowUtilsFactory;
import uk.ac.imperial.doc.kenya.gui.editor.utils.EditorControlListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.FileLoaderSaver;
import uk.ac.imperial.doc.kenya.gui.editor.utils.JavaExecuteButtonSelectionListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.JavaLineStyler;
import uk.ac.imperial.doc.kenya.gui.editor.utils.JavaPaintListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.JavaTerminateButtonSelectionListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.KenyaKeyListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.KenyaLineStyler;
import uk.ac.imperial.doc.kenya.gui.editor.utils.KenyaPaintListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.KenyaTableSelectionChangedListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.KenyaTextChangedListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.NumbersSelectionListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterContinueButtonSelectionListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterInspectorContentProvider;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterInspectorLabelProvider;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterInspectorListContentProvider;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterInspectorListLabelProvider;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterInterpretSelectionListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterMouseListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterPaintListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.InterpreterStepModeButtonSelectionListener;
import uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter.TabSelectionAdapter;
import uk.ac.imperial.doc.kenya.mediator.Mediator;
import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeInformation;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IMethodScope;
import uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode;
import uk.ac.imperial.doc.kenya.styleCheckers.IStyleCheckResult;

/**
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class EditingWindow extends ApplicationWindow implements IKenyaTextProvider,IKenyaHighlightedText{

    public static final int KENYATABINDEX = 0;

    public static final int JAVATABINDEX = 1;

    public static final int INTERPRETERTABINDEX = 2;

    private final Runnable runGetKenyaCaretOffset = new Runnable() {

        public void run() {
            getKenyaCaretOffset();
        }
    };

    private final Runnable runLoadFromKenyaFile = new Runnable() {

        public void run() {
            loadFromKenyaFile();
        }
    };

    private final Runnable runSafeRedraw = new Runnable() {

        public void run() {
            safeRedraw();
        }
    };

    private final Runnable runIsVisible = new Runnable() {

        public void run() {
            isVisible();
        }
    };

    private final Runnable runGetKenyaText = new Runnable() {

        public void run() {
            getKenyaText();
        }
    };

    private final Runnable runRebuildInterpreterNumbers = new Runnable() {

        public void run() {
            rebuildInterpreterNumbers();
        }
    };

    private final Runnable runRebuildJavaNumbers = new Runnable() {

        public void run() {
            rebuildJavaNumbers();
        }
    };

    private final Runnable runCloseWindow = new Runnable() {

        public void run() {
            closeWindow();
        }
    };

    private final Runnable runRebuildKenyaNumbers = new Runnable() {

        public void run() {
            rebuildKenyaNumbers();
        }
    };

    private final Object secondaryLock = new Object();

    private final StringBuffer kenyaNumbersBuffer = new StringBuffer();

    private int kenyaNumbersLineCount = -1;

    private final StringBuffer javaNumbersBuffer = new StringBuffer();

    private int javaNumbersLineCount = -1;

    private final StringBuffer interpreterNumbersBuffer = new StringBuffer();

    private int interpreterNumbersLineCount = -1;

    private volatile IButtonManager buttonManager = null;

    private volatile TabFolder topTabs = null;

    private volatile SelectionListener tabSelectionListener = null;

    private volatile StyledText interpreterText = null;

    private volatile StyledText interpreterNumbers = null;

    private volatile StyledText javaText = null;

    private volatile StyledText javaNumbers = null;

    private volatile StyledText kenyaText = null;

    private volatile StyledText kenyaNumbers = null;

    private volatile ISourceCodeInformationTableManager kenyaHumanErrorTableManager = null;

    private volatile ISourceCodeInformationTableManager kenyaJavacErrorTable = null;

    private final KenyaErrorContentProvider kenyaTableContents = new KenyaErrorContentProvider();

    private volatile Button javaExecuteButton = null;

    private volatile Button javaEOFButton = null;

    private volatile Button javaTerminateButton = null;

    private volatile Text javaStdInText = null;

    private volatile StyledText javaStdOutText = null;

    private volatile Text javaArgsText = null;

    private volatile Label javaArgsLabel = null;

    private String[] javaArgsResult = null;

    private volatile IJavaCode javaCode = null;

    private final String javaDefaultClassName;

    private volatile String javaClassName = null;

    private volatile boolean javaRunning = false;

    private volatile String kenyaWidgetText = "";

    private volatile boolean isVisible = false;

    private volatile int caretOffset = 0;

    private final Date lastKeyEvent = new Date();

    private volatile boolean reparse = false;

    private volatile FileLoaderSaver javaFileLoaderSaver = null;

    private volatile FileLoaderSaver kenyaFileLoaderSaver = null;

    private volatile KenyaLineStyler kenyaLineStyler = null;

    private volatile JavaLineStyler javaLineStyler = null;

    private volatile IHighlightingManager highlights = null;

    private volatile boolean modified = false;

    private final IndentationStrategy is = new IndentationStrategy();

    private volatile CarrotPainter carrotPainter = null;

    private volatile IStackMachineInformationProvider baseStackMachine = null;

    private volatile IStackMachine stackMachine = null;

    private volatile Button interpreterInterpretButton = null;

    private volatile StyledText interpreterStdOutText = null;

    private volatile Text interpreterStdInText = null;

    private volatile Button interpreterTrackButton = null;

    private volatile Button interpreterEOFButton = null;

    private volatile Button interpreterTerminateButton = null;

    private volatile Button interpreterStepModeButton = null;

    private volatile boolean interpreterStepMode = false;

    private volatile boolean interpreterTrackMode = false;

    private volatile Button interpreterContinueButton = null;

    private volatile TreeViewer interpreterInspector = null;

    private volatile ListViewer interpreterList = null;

    private volatile Text interpreterArgsText = null;

    private volatile Label interpreterArgsLabel = null;

    private String[] interpreterArgsResult = null;

    private volatile String lastDir = null;

    private final ExecutorService executor = Executors.newFixedThreadPool(16);

    private ISaveKenyaActionHandler kenyaSaveHandler = null;

    private IAction javaSaveAsAction = null;

    private boolean saveableJava = false;

    public EditingWindow(String dcn) {
        super(null);
        addStatusLine();
        addMenuBar();
        javaDefaultClassName = dcn;
    }

    // only the GUI thread will call this
    protected synchronized Control createContents(Composite parent) {
        buttonManager = new ButtonManager(this);

        topTabs = new TabFolder(parent, SWT.NULL);

        final TabItem kenyaTab = new TabItem(topTabs, SWT.NULL);
        kenyaTab.setText("&Kenya");

        final TabItem javaTab = new TabItem(topTabs, SWT.NULL);
        javaTab.setText("&Java");

        final TabItem interpreterTab = new TabItem(topTabs, SWT.NULL);
        interpreterTab.setText("&Debugger");

        final SashForm kenyaTopControl = new SashForm(topTabs, SWT.VERTICAL
                | SWT.BORDER);
        kenyaTab.setControl(kenyaTopControl);
        buildKenya(kenyaTopControl);

        final SashForm javaTopControl = new SashForm(topTabs, SWT.VERTICAL
                | SWT.BORDER);
        javaTab.setControl(javaTopControl);
        buildJava(javaTopControl);

        final SashForm interpreterTopControl = new SashForm(topTabs,
                SWT.VERTICAL | SWT.BORDER);
        interpreterTab.setControl(interpreterTopControl);
        buildInterpreter(interpreterTopControl);

        tabSelectionListener = new TabSelectionAdapter(this);
        topTabs.addSelectionListener(tabSelectionListener);

        FontData fontdata = kenyaText.getFont().getFontData()[0];
        fontdata.setName("Courier");
        fontdata.setHeight(12);
        Font ioFont = new Font(topTabs.getDisplay(), fontdata);

        setIOFonts(ioFont);

        fontdata = kenyaText.getFont().getFontData()[0];
        fontdata.setName("arial");
        fontdata.setHeight(12);
        Font editorFont = new Font(topTabs.getDisplay(), fontdata);

        setEditorFonts(editorFont);

        parent.setSize(600, 800);

        return topTabs;
    }

    protected void handleShellCloseEvent() {
        closeWindow();
    }

    private void buildKenya(SashForm kenyaTopControl) {
        kenyaTopControl.setLayout(new FillLayout());
    
        setKenyaLineStyler(new KenyaLineStyler());
    
        final SashForm editorSash = new SashForm(kenyaTopControl,
                SWT.HORIZONTAL);
    
        kenyaNumbers = new StyledText(editorSash, SWT.NO_FOCUS | SWT.READ_ONLY);
        kenyaNumbers.setEditable(false);
        kenyaNumbers.setDoubleClickEnabled(false);
        kenyaNumbers.setEnabled(false);
        kenyaNumbers.addSelectionListener(new NumbersSelectionListener(
                kenyaNumbers));
    
        kenyaText = new StyledText(editorSash, SWT.NULL | SWT.V_SCROLL
                | SWT.H_SCROLL);
        kenyaText.addVerifyKeyListener(new KenyaKeyListener(this));
        kenyaText.addPaintListener(new KenyaPaintListener(this));
        kenyaText.addLineStyleListener(kenyaLineStyler);
        kenyaText.getContent().addTextChangeListener(
                new KenyaTextChangedListener(this));
    
        FileLoaderSaver fls = getKenyaFile();
        if (fls == null) {
            kenyaText.setText("");
            getShell().setText("Kenya IDE");
            setModified(false);
        } else {
            loadFromKenyaFile();
        }
    
        rebuildKenyaNumbers();
        editorSash.setWeights(new int[] { 1, 18 });
    
        if (highlights != null) {
            kenyaText.addMouseListener(highlights
                    .createKenyaMouseListener(this));
            kenyaNumbers.addLineBackgroundListener(highlights
                    .createKenyaLineBackgroundListener(this));
        }
    
        TabFolder kenyaTabs = new TabFolder(kenyaTopControl, SWT.NULL);
    
        TabItem humanErrorsTab = new TabItem(kenyaTabs, SWT.NULL);
        humanErrorsTab.setText("Kenya errors");
    
        TabItem javacErrorsTab = new TabItem(kenyaTabs, SWT.NULL);
        javacErrorsTab.setText("Java errors");
    
        /* human error table */
    
        ISelectionChangedListener tableSelectionChanged = new KenyaTableSelectionChangedListener(
                this);
        
        kenyaHumanErrorTableManager = new SourceCodeInformationTableManager(this,
                kenyaTabs, new HumanErrorLabelContentProvider(buttonManager, kenyaText.getLineDelimiter()), 
                tableSelectionChanged, kenyaTableContents); 
        humanErrorsTab.setControl(kenyaHumanErrorTableManager.getControl());
    
        /* javac error table */
        
        kenyaJavacErrorTable = new SourceCodeInformationTableManager(this,
               kenyaTabs, new JavacErrorLabelContentProvider(buttonManager, kenyaText.getLineDelimiter()),
                        tableSelectionChanged, kenyaTableContents);
        javacErrorsTab.setControl(kenyaJavacErrorTable.getControl());
    
        try {
            ICheckedCode checker = Mediator.check(new StringReader(kenyaText
                    .getText()));
            setKenyaCheckedCode(checker);
        } catch (IOException e) {
        }
    
        kenyaTopControl.setWeights(new int[] { 3, 1 });
    
        kenyaText.setFocus();
    
        editorSash.addControlListener(new EditorControlListener(editorSash,
                kenyaNumbers, kenyaText));
    }

    private void buildJava(SashForm javaTopControl) {
        javaTopControl.setLayout(new FillLayout());

        final SashForm editorSash = new SashForm(javaTopControl, SWT.HORIZONTAL);

        javaNumbers = new StyledText(editorSash, SWT.NO_FOCUS | SWT.READ_ONLY);
        javaNumbers.setEditable(false);
        javaNumbers.setDoubleClickEnabled(false);
        javaNumbers.setEnabled(false);
        javaNumbers.addSelectionListener(new NumbersSelectionListener(
                javaNumbers));
        javaNumbers.setBackground(DefaultHighlightingManager.WHITE);

        javaText = new StyledText(editorSash, SWT.NULL | SWT.V_SCROLL
                | SWT.H_SCROLL | SWT.READ_ONLY);
        javaText.setEditable(false);
        javaText.addPaintListener(new JavaPaintListener(this));
        javaLineStyler = new JavaLineStyler();
        javaText.addLineStyleListener(javaLineStyler);
        javaText.addKeyListener(new KeyListener() {

            private static final String statusText = "Switch to the Kenya tab to edit the code.";

            public void keyPressed(KeyEvent e) {
                safeSetStatus(statusText);
            }

            public void keyReleased(KeyEvent e) {
                safeSetStatus(statusText);
            }
        });

        rebuildJavaNumbers();
        editorSash.setWeights(new int[] { 1, 18 });

        Composite ioComposite = new Composite(javaTopControl, SWT.BORDER);
        FormLayout fl = new FormLayout();
        ioComposite.setLayout(fl);

        javaExecuteButton = new Button(ioComposite, SWT.PUSH);
        javaExecuteButton.setText("E&xecute Java");

        javaTerminateButton = new Button(ioComposite, SWT.PUSH);
        javaTerminateButton.setText("Terminate");
        javaTerminateButton.setEnabled(false);
        javaTerminateButton
                .addSelectionListener(new JavaTerminateButtonSelectionListener(
                        this));

        javaStdOutText = new StyledText(ioComposite, SWT.MULTI | SWT.WRAP
                | SWT.V_SCROLL | SWT.BORDER);
        javaStdOutText.setEditable(false);
        javaStdOutText.setTabs(8);

        javaStdInText = new Text(ioComposite, SWT.SINGLE | SWT.BORDER);
        javaStdInText.setEditable(true);
        javaStdInText.setEnabled(false);
        javaStdInText.setTabs(8);

        javaEOFButton = new Button(ioComposite, SWT.PUSH);
        javaEOFButton.setText("Input EOF");
        javaEOFButton.setEnabled(false);

        javaArgsLabel = new Label(ioComposite, SWT.LEFT);
        javaArgsLabel.setText("Arguments");

        javaArgsText = new Text(ioComposite, SWT.SINGLE | SWT.BORDER);

        // top left
        FormData fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.left = new FormAttachment(0, 1);
        fd.right = new FormAttachment(javaTerminateButton, -5);
        javaExecuteButton.setLayoutData(fd);

        // top right
        fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.right = new FormAttachment(100, -1);
        fd.width = 100;
        fd.bottom = new FormAttachment(javaExecuteButton, 0, SWT.BOTTOM);
        javaTerminateButton.setLayoutData(fd);

        // middle
        fd = new FormData();
        fd.top = new FormAttachment(javaExecuteButton, 5);
        fd.bottom = new FormAttachment(javaStdInText, -5);
        fd.left = new FormAttachment(0, 1);
        fd.right = new FormAttachment(100, -1);
        javaStdOutText.setLayoutData(fd);

        // bottom left
        fd = new FormData();
        fd.left = new FormAttachment(0, 1); // 0%, 0 offset
        fd.bottom = new FormAttachment(javaArgsText, -5);
        fd.right = new FormAttachment(javaEOFButton, -5);
        javaStdInText.setLayoutData(fd);

        // bottom right
        fd = new FormData();
        fd.right = new FormAttachment(100, -1); // 100%, 0 offset
        fd.bottom = new FormAttachment(javaArgsText, -5);
        fd.width = 100;
        fd.top = new FormAttachment(javaStdInText, 0, SWT.TOP);
        javaEOFButton.setLayoutData(fd);

        fd = new FormData();
        fd.left = new FormAttachment(0, 1);
        fd.bottom = new FormAttachment(100, -5);
        fd.top = new FormAttachment(javaArgsText, 5, SWT.TOP);
        javaArgsLabel.setLayoutData(fd);

        fd = new FormData();
        fd.right = new FormAttachment(100, -1);
        fd.left = new FormAttachment(javaArgsLabel, 5);
        fd.bottom = new FormAttachment(100, -5);
        javaArgsText.setLayoutData(fd);

        SelectionListener executeJavaButtonSelection = new JavaExecuteButtonSelectionListener(
                this);
        javaExecuteButton.addSelectionListener(executeJavaButtonSelection);

        javaTopControl.setWeights(new int[] { 2, 1 });

        javaTopControl.addControlListener(new EditorControlListener(editorSash,
                javaNumbers, javaText));

    }

    private void buildInterpreter(SashForm interpreterTopControl) {
        final SashForm editorSash = new SashForm(interpreterTopControl,
                SWT.HORIZONTAL);

        interpreterNumbers = new StyledText(editorSash, SWT.NO_FOCUS
                | SWT.READ_ONLY);
        interpreterNumbers.setEditable(false);
        interpreterNumbers.setDoubleClickEnabled(false);
        interpreterNumbers.setEnabled(false);
        interpreterNumbers.addSelectionListener(new NumbersSelectionListener(
                interpreterNumbers));
        interpreterNumbers.setBackground(DefaultHighlightingManager.WHITE);

        interpreterText = new StyledText(editorSash, SWT.NULL | SWT.V_SCROLL
                | SWT.H_SCROLL | SWT.READ_ONLY);
        interpreterText.setEditable(false);
        interpreterText.addPaintListener(new InterpreterPaintListener(this));
        interpreterText.addKeyListener(new KeyListener() {

            public void keyPressed(KeyEvent e) {
            }

            public void keyReleased(KeyEvent e) {
                if (e.character >= ' ') {
                    int caretOffset = interpreterText.getCaretOffset();
                    topTabs.setSelection(KENYATABINDEX);
                    kenyaText.setSelection(caretOffset);
                    kenyaText.showSelection();
                    kenyaText.insert("" + e.character);
                    kenyaText.setCaretOffset(caretOffset + 1);
                    kenyaText.setFocus();
                }
            }
        });

        carrotPainter = new CarrotPainter(this);

        interpreterText.addPaintListener(carrotPainter);
        InterpreterMouseListener iml = new InterpreterMouseListener(this);
        interpreterText.addMouseListener(iml);

        rebuildInterpreterNumbers();
        editorSash.setWeights(new int[] { 1, 18 });

        SashForm inspectorSash = new SashForm(interpreterTopControl,
                SWT.HORIZONTAL | SWT.BORDER);

        interpreterList = new ListViewer(inspectorSash, SWT.SINGLE
                | SWT.V_SCROLL | SWT.H_SCROLL);
        interpreterList
                .setContentProvider(new InterpreterInspectorListContentProvider());
        interpreterList
                .setLabelProvider(new InterpreterInspectorListLabelProvider(
                        this));
        interpreterList
                .addSelectionChangedListener(new ISelectionChangedListener() {

                    private Object firstElement = null;

                    public void selectionChanged(SelectionChangedEvent event) {
                        IStructuredSelection selection = (IStructuredSelection) event
                                .getSelection();
                        if (selection.getFirstElement() != firstElement) {
                            getInterpreterInspector().setInput(
                                    selection.getFirstElement());
                            firstElement = selection.getFirstElement();
                        }

                        if (selection.getFirstElement() instanceof IMethodScope) {
                            IFunction function = getBaseStackMachine()
                                    .lookupFunction(
                                            ((IMethodScope) selection
                                                    .getFirstElement())
                                                    .getCurrentMethod()
                                                    .getName());
                            StyledText text = getInterpreterTextWidget();
                            int offset = text.getOffsetAtLine(function
                                    .getPosition().getLineNumber() - 1);
                            offset += function.getPosition().getColumnNumber() - 1;
                            text.setCaretOffset(offset);
                            text.setSelection(offset, offset
                                    + function.getPosition().getTokenLength());
                        }
                    }
                });

        interpreterInspector = new TreeViewer(inspectorSash, SWT.SINGLE
                | SWT.V_SCROLL | SWT.H_SCROLL);
        interpreterInspector
                .setContentProvider(new InterpreterInspectorContentProvider());
        interpreterInspector
                .setLabelProvider(new InterpreterInspectorLabelProvider(this));
        interpreterInspector.setUseHashlookup(true);

        Composite ioComposite = new Composite(interpreterTopControl, SWT.BORDER);
        FormLayout fl = new FormLayout();
        ioComposite.setLayout(fl);

        interpreterInterpretButton = new Button(ioComposite, SWT.PUSH);
        interpreterInterpretButton.setText("&Run");
        interpreterInterpretButton.setEnabled(false);

        interpreterStdOutText = new StyledText(ioComposite, SWT.MULTI
                | SWT.WRAP | SWT.V_SCROLL | SWT.BORDER);
        interpreterStdOutText.setText("");
        interpreterStdOutText.setEditable(false);
        interpreterStdOutText.setTabs(8);

        interpreterStdInText = new Text(ioComposite, SWT.SINGLE | SWT.BORDER);
        interpreterStdInText.setText("");
        interpreterStdInText.setEditable(true);
        interpreterStdInText.setEnabled(false);
        interpreterStdInText.setTabs(8);

        interpreterEOFButton = new Button(ioComposite, SWT.PUSH);
        interpreterEOFButton.setText("Input EOF");
        interpreterEOFButton.setEnabled(false);

        interpreterTerminateButton = new Button(ioComposite, SWT.PUSH);
        interpreterTerminateButton.setText("Terminate");
        interpreterTerminateButton.setEnabled(false);

        interpreterStepModeButton = new Button(ioComposite, SWT.TOGGLE);
        interpreterStepModeButton.setText("Step Mode");
        interpreterStepModeButton.setEnabled(false);
        interpreterStepModeButton
                .addSelectionListener(new InterpreterStepModeButtonSelectionListener(
                        this));
        setInterpreterStepMode(interpreterStepModeButton.getSelection());

        interpreterContinueButton = new Button(ioComposite, SWT.PUSH);
        interpreterContinueButton.setText("Continue");
        interpreterContinueButton.setEnabled(false);

        interpreterContinueButton
                .addSelectionListener(new InterpreterContinueButtonSelectionListener(
                        this));

        interpreterTrackButton = new Button(ioComposite, SWT.TOGGLE);
        interpreterTrackButton.setText("Track");
        interpreterTrackButton.setEnabled(true);
        interpreterTrackButton.addSelectionListener(new SelectionAdapter() {

            public void widgetSelected(SelectionEvent e) {
                int pri = Thread.currentThread().getPriority();
                Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
                setInterpreterTrackMode(interpreterTrackButton.getSelection());
                Thread.currentThread().setPriority(pri);
            }
        });
        interpreterTrackButton.setSelection(true);
        setInterpreterTrackMode(interpreterTrackButton.getSelection());

        interpreterArgsLabel = new Label(ioComposite, SWT.LEFT);
        interpreterArgsLabel.setText("Arguments");

        interpreterArgsText = new Text(ioComposite, SWT.SINGLE | SWT.BORDER);

        // top left
        FormData fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.left = new FormAttachment(0, 1);
        fd.right = new FormAttachment(interpreterTrackButton, -5);
        interpreterInterpretButton.setLayoutData(fd);

        // top middle left left
        fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.right = new FormAttachment(interpreterContinueButton, -5);
        fd.width = 100;
        fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
                SWT.BOTTOM);
        interpreterTrackButton.setLayoutData(fd);

        // top middle left
        fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.right = new FormAttachment(interpreterStepModeButton, -5);
        fd.width = 100;
        fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
                SWT.BOTTOM);
        interpreterContinueButton.setLayoutData(fd);

        // top middle right
        fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.right = new FormAttachment(interpreterTerminateButton, -5);
        fd.width = 100;
        fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
                SWT.BOTTOM);
        interpreterStepModeButton.setLayoutData(fd);

        // top right
        fd = new FormData();
        fd.top = new FormAttachment(0, 0);
        fd.right = new FormAttachment(100, -1);
        fd.width = 100;
        fd.bottom = new FormAttachment(interpreterInterpretButton, 0,
                SWT.BOTTOM);
        interpreterTerminateButton.setLayoutData(fd);

        // middle
        fd = new FormData();
        fd.top = new FormAttachment(interpreterInterpretButton, 5);
        fd.bottom = new FormAttachment(interpreterStdInText, -5);
        fd.left = new FormAttachment(0, 1);
        fd.right = new FormAttachment(100, -1);
        interpreterStdOutText.setLayoutData(fd);

        // bottom left
        fd = new FormData();
        fd.left = new FormAttachment(0, 1); // 0%, 0 offset
        fd.bottom = new FormAttachment(interpreterArgsText, -5);
        fd.right = new FormAttachment(interpreterEOFButton, -5);
        interpreterStdInText.setLayoutData(fd);

        // bottom right
        fd = new FormData();
        fd.right = new FormAttachment(100, -1); // 100%, 0 offset
        fd.bottom = new FormAttachment(interpreterArgsText, -5);
        fd.width = 100;
        fd.top = new FormAttachment(interpreterStdInText, 0, SWT.TOP);
        interpreterEOFButton.setLayoutData(fd);

        fd = new FormData();
        fd.left = new FormAttachment(0, 1);
        fd.bottom = new FormAttachment(100, -5);
        fd.top = new FormAttachment(interpreterArgsText, 5, SWT.TOP);
        interpreterArgsLabel.setLayoutData(fd);

        fd = new FormData();
        fd.right = new FormAttachment(100, -1);
        fd.left = new FormAttachment(interpreterArgsLabel, 5);
        fd.bottom = new FormAttachment(100, -5);
        interpreterArgsText.setLayoutData(fd);

        SelectionListener interpretButtonListener = new InterpreterInterpretSelectionListener(
                this);
        interpreterInterpretButton
                .addSelectionListener(interpretButtonListener);

        interpreterTopControl.setWeights(new int[] { 1, 1, 1 });

        interpreterTopControl.addControlListener(new EditorControlListener(
                editorSash, interpreterNumbers, interpreterText));
    }
    
    // only the GUI thread will call this
    protected synchronized MenuManager createMenuManager() {
        MenuManager bar_menu = new MenuManager("");

        MenuManager file_menu = new MenuManager("&File");
        bar_menu.add(file_menu);
        file_menu.add(new NewAction(this));
        file_menu.add(new OpenAction(this));
        file_menu.add(new OpenInNewAction(this));
        file_menu.add(new KenyaSaveAction(this));
        file_menu.add(new KenyaSaveAsAction(this));

        MenuManager javaSaveOptions = new MenuManager("Java Save Options");
        file_menu.add(javaSaveOptions);

        UniqueCheckedManager javaSaveOptionsManager = new UniqueCheckedManager();
        IAction alwaysSaveJavaAutomatically = new AlwaysSaveJavaAutomatically(
                this);
        javaSaveOptions.add(alwaysSaveJavaAutomatically);
        javaSaveOptionsManager.addAction(alwaysSaveJavaAutomatically);

        IAction promptToSaveJava = new PromptToSaveJava(this);
        javaSaveOptions.add(promptToSaveJava);
        javaSaveOptionsManager.addAction(promptToSaveJava);

        IAction manuallySaveJava = new ManuallySaveJava(this);
        javaSaveOptions.add(manuallySaveJava);
        javaSaveOptionsManager.addAction(manuallySaveJava);

        javaSaveAsAction = new JavaSaveAsAction(this);
        file_menu.add(javaSaveAsAction);
        javaSaveAsAction.setEnabled(false);

        file_menu.add(new CloseAction(this));
        file_menu.add(new ExitAction(this));

        MenuManager edit_menu = new MenuManager("&Edit");
        bar_menu.add(edit_menu);
        edit_menu.add(new CopyTextAction(this));
        edit_menu.add(new CutTextAction(this));
        edit_menu.add(new PasteTextAction(this));
        edit_menu.add(new FormatAction(this));

        MenuManager view_menu = new MenuManager("&View");
        bar_menu.add(view_menu);

        MenuManager editing_fonts_menu = new MenuManager("&Editor Fonts");
        view_menu.add(editing_fonts_menu);

        MenuManager editing_fonts_size_menu = new MenuManager("Size");
        editing_fonts_menu.add(editing_fonts_size_menu);
        UniqueCheckedManager ucm = new UniqueCheckedManager();

        IAction fontAction = new EditorSmallFontAction(this);
        ucm.addAction(fontAction);
        editing_fonts_size_menu.add(fontAction);

        fontAction = new EditorMediumFontAction(this);
        ucm.addAction(fontAction);
        editing_fonts_size_menu.add(fontAction);

        fontAction = new EditorBigFontAction(this);
        ucm.addAction(fontAction);
        editing_fonts_size_menu.add(fontAction);

        fontAction = new EditorHugeFontAction(this);
        ucm.addAction(fontAction);
        editing_fonts_size_menu.add(fontAction);

        MenuManager editing_fonts_type_menu = new MenuManager("Type");
        editing_fonts_menu.add(editing_fonts_type_menu);
        ucm = new UniqueCheckedManager();

        fontAction = new EditorProportionalFontAction(this);
        ucm.addAction(fontAction);
        editing_fonts_type_menu.add(fontAction);

        fontAction = new EditorFixedFontAction(this);
        ucm.addAction(fontAction);
        editing_fonts_type_menu.add(fontAction);

        MenuManager java_io_fonts_menu = new MenuManager("&I/O Fonts");
        view_menu.add(java_io_fonts_menu);

        MenuManager java_io_fonts_size_menu = new MenuManager("Size");
        java_io_fonts_menu.add(java_io_fonts_size_menu);
        ucm = new UniqueCheckedManager();

        fontAction = new IOSmallFontAction(this);
        ucm.addAction(fontAction);
        java_io_fonts_size_menu.add(fontAction);

        fontAction = new IOMediumFontAction(this);
        ucm.addAction(fontAction);
        java_io_fonts_size_menu.add(fontAction);

        fontAction = new IOBigFontAction(this);
        ucm.addAction(fontAction);
        java_io_fonts_size_menu.add(fontAction);

        fontAction = new IOHugeFontAction(this);
        ucm.addAction(fontAction);
        java_io_fonts_size_menu.add(fontAction);

        MenuManager java_io_fonts_type_menu = new MenuManager("Type");
        java_io_fonts_menu.add(java_io_fonts_type_menu);
        ucm = new UniqueCheckedManager();

        fontAction = new IOProportionalFontAction(this);
        ucm.addAction(fontAction);
        java_io_fonts_type_menu.add(fontAction);

        fontAction = new IOFixedFontAction(this);
        ucm.addAction(fontAction);
        java_io_fonts_type_menu.add(fontAction);

        view_menu.add(new SwitchToKenya(this));
        view_menu.add(new SwitchToJava(this));
        view_menu.add(new SwitchToInterpreter(this));

        MenuManager help_menu = new MenuManager("&Help");
        bar_menu.add(help_menu);
        help_menu.add(new AboutAction(this));

        return bar_menu;
    }

    public synchronized boolean isJavaRunning() {
        return javaRunning;
    }

    public synchronized void setJavaRunning(boolean value) {
        javaRunning = value;
    }

    public void safeSetStatus(final String message) {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                setStatus(message);
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {
                    safeSetStatus(message);
                }
            });
        }
    }

    public synchronized void positionKenyaNumbers() {
        if (kenyaNumbers.getTopPixel() != kenyaText.getTopPixel()) {
            kenyaNumbers.setTopPixel(kenyaText.getTopPixel());
            kenyaNumbers.redraw();
        }
    }

    public synchronized void positionJavaNumbers() {
        if (javaNumbers.getTopPixel() != javaText.getTopPixel()) {
            javaNumbers.setTopPixel(javaText.getTopPixel());
            javaNumbers.redraw();
        }
    }

    public synchronized void positionInterpreterNumbers() {
        if (interpreterNumbers.getTopPixel() != interpreterText.getTopPixel()) {
            interpreterNumbers.setTopPixel(interpreterText.getTopPixel());
            interpreterNumbers.redraw();
        }
    }

    public void rebuildKenyaNumbers() {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (kenyaNumbersLineCount != kenyaText.getLineCount()) {
                    int topPixel = kenyaNumbers.getTopPixel();
                    kenyaNumbersBuffer.setLength(0);
                    for (int idx = 1; idx <= kenyaText.getLineCount(); idx++) {
                        kenyaNumbersBuffer.append(idx);
                        kenyaNumbersBuffer.append(SWT.CR);
                    }
                    kenyaNumbers.setText(kenyaNumbersBuffer.toString());
                    kenyaNumbers.setTopPixel(topPixel);
                    kenyaNumbersLineCount = kenyaText.getLineCount();
                }
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(runRebuildKenyaNumbers);
        }
    }

    public void rebuildJavaNumbers() {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (javaNumbersLineCount != javaText.getLineCount()) {
                    int topPixel = javaNumbers.getTopPixel();
                    javaNumbersBuffer.setLength(0);
                    for (int idx = 1; idx <= javaText.getLineCount(); idx++) {
                        javaNumbersBuffer.append(idx);
                        javaNumbersBuffer.append(SWT.CR);
                    }
                    javaNumbers.setText(javaNumbersBuffer.toString());
                    javaNumbers.setTopPixel(topPixel);
                    javaNumbersLineCount = javaText.getLineCount();
                }
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(runRebuildJavaNumbers);
        }
    }

    public void rebuildInterpreterNumbers() {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (interpreterNumbersLineCount != interpreterText
                        .getLineCount()) {
                    int topPixel = interpreterNumbers.getTopPixel();
                    interpreterNumbersBuffer.setLength(0);
                    for (int idx = 1; idx <= interpreterText.getLineCount(); idx++) {
                        interpreterNumbersBuffer.append(idx);
                        interpreterNumbersBuffer.append(SWT.CR);
                    }
                    interpreterNumbers.setText(interpreterNumbersBuffer
                            .toString());
                    interpreterNumbers.setTopPixel(topPixel);
                    interpreterNumbersLineCount = interpreterText
                            .getLineCount();
                }
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(runRebuildInterpreterNumbers);
        }
    }

    public void setKenyaCheckedCode(final ICheckedCode checker) {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                kenyaHumanErrorTableManager.setInput(checker);
                kenyaJavacErrorTable.setInput(checker);
                setJavaGoodToSave(!checker.isErroredCode());
                if (checker.isErroredCode()) {
                    if (getTopTabs().getSelectionIndex() != KENYATABINDEX) {
                        switchToTab(KENYATABINDEX);
                    }
                }
                List<ISourceCodeWarning> warnings = checker.getInfos();

                if (checker.isErroredCode()) {
                    List<ISourceCodeError> errors = checker.getErrors();
                    for (int idx = 0; idx < errors.size(); idx++) {
                        ISourceCodeError error = (ISourceCodeError) errors
                                .get(idx);
                        getHighlightingManager().addSourceCodeError(error);
                    }
                    if (errors.size() == 1)
                        safeSetStatus("1 error found.");
                    else
                        safeSetStatus("" + errors.size() + " errors found.");
                } else {
                    if (warnings.size() == 0)
                        safeSetStatus("No errors or warnings found.");
                    else if (warnings.size() == 1)
                        safeSetStatus("1 warning found; no errors found.");
                    else
                        safeSetStatus("" + warnings.size()
                                + " warnings found; no errors found.");
                }

                for (int idx = 0; idx < warnings.size(); idx++) {
                    ISourceCodeInformation warning = (ISourceCodeInformation) warnings
                            .get(idx);
                    getHighlightingManager().addSourceCodeWarning(warning);
                }
                
                List<IStyleCheckResult> styleChecks = checker.getStyleCheckResults();
                for(IStyleCheckResult scr : styleChecks) {
                    getHighlightingManager().addStyleCheckResult(scr);
                }
                
                getKenyaTextWidget().redraw();
                getKenyaLineNumberWidget().redraw();
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {
                    setKenyaCheckedCode(checker);
                }
            });
        }

    }

    public void selectInKenyaErrorTable(final ISourceCodeInformation error) {
        if (!isVisible()) return;
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (kenyaJavacErrorTable == null
                        || kenyaHumanErrorTableManager == null) return;
                if (error == null && !getTopTabs().isDisposed()) {
                    kenyaJavacErrorTable.deselectAll();
                    kenyaHumanErrorTableManager.deselectAll();
                } else {
                    int index = kenyaTableContents.getIndex(error);
                    if (index == -1) {
                        kenyaJavacErrorTable.deselectAll();
                        kenyaHumanErrorTableManager.deselectAll();
                    } else {
                        kenyaJavacErrorTable.setSelection(index);
                        kenyaJavacErrorTable.showSelection();
                        kenyaHumanErrorTableManager.setSelection(index);
                        kenyaHumanErrorTableManager.showSelection();
                    }
                }
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {
                    selectInKenyaErrorTable(error);
                }
            });
        }
    }

    public synchronized StyledText getKenyaTextWidget() {
        return kenyaText;
    }

    public synchronized StyledText getJavaTextWidget() {
        return javaText;
    }

    public synchronized StyledText getInterpreterTextWidget() {
        return interpreterText;
    }

    public synchronized StyledText getInterpreterLineNumberWidget() {
        return interpreterNumbers;
    }

    public synchronized StyledText getKenyaLineNumberWidget() {
        return kenyaNumbers;
    }

    public synchronized StyledText getJavaLineNumberWidget() {
        return javaNumbers;
    }

    public String getKenyaText() {
        if (!isVisible()) return "";
        if (Thread.currentThread().equals(kenyaText.getDisplay().getThread())) {
            synchronized (this) {
                setKenyaWidgetText(kenyaText.getText());
            }
        } else {
            kenyaText.getDisplay().syncExec(runGetKenyaText);
        }
        return getKenyaWidgetText();
    }

    public void setInterpreterStepMode(boolean result) {
        synchronized (secondaryLock) {
            interpreterStepMode = result;
        }
    }

    public boolean getInterpreterStepMode() {
        synchronized (secondaryLock) {
            return interpreterStepMode;
        }
    }

    private void setWindowVisible(boolean isVisible) {
        synchronized (secondaryLock) {
            this.isVisible = isVisible;
        }
    }

    private boolean getWindowVisible() {
        synchronized (secondaryLock) {
            return isVisible;
        }
    }

    public boolean isVisible() {
        TabFolder topTabs = getTopTabs();
        if (topTabs == null || topTabs.isDisposed() || EditingWindowUtilsFactory.kenya.isExiting())
            return false;

        if (Thread.currentThread().equals(topTabs.getDisplay().getThread())) {
            synchronized (this) {
                setWindowVisible(topTabs.isVisible());
            }
        } else {
            topTabs.getDisplay().syncExec(runIsVisible);
        }
        return getWindowVisible();
    }

    public synchronized void resetTimer() {
        lastKeyEvent.setTime(System.currentTimeMillis());
    }

    public synchronized Date getLastKeyEvent() {
        return lastKeyEvent;
    }

    public void safeRedraw() {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                getTopTabs().redraw();
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(runSafeRedraw);
        }
    }

    public synchronized void reparse(boolean value) {
        reparse = value;
    }

    public synchronized boolean reparse() {
        return reparse;
    }

    public synchronized void setKenyaFile(FileLoaderSaver fls) {
        kenyaFileLoaderSaver = fls;
    }

    public void loadFromKenyaFile() {
        if (getKenyaTextWidget() != null && getKenyaFile() != null) {
            if (Thread.currentThread().equals(
                    kenyaText.getDisplay().getThread()))
                synchronized (this) {
                    FileLoaderSaver fls = getKenyaFile();
                    String contents = fls.loadFile(fls.getPath());
                    getShell().setText("Kenya IDE: " + fls.getPath());
                    kenyaText.setText(contents);
                    switchToTab(KENYATABINDEX);
                    getIndentationStrategy().parseText(kenyaText);
                    kenyaText.setCaretOffset(0);
                    kenyaText.setSelection(0);
                    kenyaText.setFocus();
                    reparse(true);
                    setModified(false);
                    setJavaFileLoaderSaver(null);
                }
            else
                kenyaText.getDisplay().syncExec(runLoadFromKenyaFile);
        }
    }

    public synchronized FileLoaderSaver getKenyaFile() {
        return kenyaFileLoaderSaver;
    }

    public synchronized KenyaLineStyler getKenyaLineStyler() {
        return kenyaLineStyler;
    }

    public synchronized JavaLineStyler getJavaLineStyler() {
        return javaLineStyler;
    }

    synchronized void setKenyaLineStyler(KenyaLineStyler lineStyler) {
        this.kenyaLineStyler = lineStyler;
    }

    public synchronized IHighlightingManager getHighlightingManager() {
        return highlights;
    }

    public synchronized void setHighlightingManager(
            IHighlightingManager highlights) {
        this.highlights = highlights;
    }

    public synchronized TabFolder getTopTabs() {
        return topTabs;
    }

    private void setKCaretOffset(int offset) {
        synchronized (secondaryLock) {
            caretOffset = offset;
        }
    }

    private int getKCaretOffset() {
        synchronized (secondaryLock) {
            return caretOffset;
        }
    }

    public int getKenyaCaretOffset() {
        if (!isVisible()) return 0;
        if (Thread.currentThread().equals(kenyaText.getDisplay().getThread())) {
            synchronized (this) {
                if (kenyaText.isDisposed()) {
                    setKCaretOffset(0);
                } else {
                    setKCaretOffset(kenyaText.getCaretOffset());
                }
            }
        } else {
            kenyaText.getDisplay().syncExec(runGetKenyaCaretOffset);
        }
        return getKCaretOffset();
    }

    public synchronized boolean isModified() {
        return modified;
    }

    public synchronized void setModified(boolean modified) {
        if (!this.modified && modified) {
            if (kenyaFileLoaderSaver != null
                    && kenyaFileLoaderSaver.getPath() != null
                    && !kenyaFileLoaderSaver.getPath().equals("")) {
                getShell().setText(
                        "Kenya IDE: " + kenyaFileLoaderSaver.getPath()
                                + " (modified)");
            } else {
                getShell().setText("Kenya IDE (modified)");
            }
        } else if (this.modified && !modified) {
            if (kenyaFileLoaderSaver != null
                    && kenyaFileLoaderSaver.getPath() != null
                    && !kenyaFileLoaderSaver.getPath().equals("")) {
                getShell().setText(
                        "Kenya IDE: " + kenyaFileLoaderSaver.getPath());
            } else {
                getShell().setText("Kenya IDE");
            }
        }
        this.modified = modified;
    }

    public void closeWindow() {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (unloadKenyaFile()) {
                    close();
                }
            }
        } else {
            if (!isVisible()) return;
            getTopTabs().getDisplay().syncExec(runCloseWindow);
        }
    }

    public boolean unloadKenyaFile() {
        if (isModified()) {
            SavePromptAction spa = new SavePromptAction(this);
            spa.run();
            if (spa.isCancelled()) return false;
        }
        return true;
    }

    public IndentationStrategy getIndentationStrategy() {
        return is;
    }

    public synchronized String getJavaClassName() {
        return javaClassName;
    }

    public synchronized void setJavaClassName(String javaClassName) {
        this.javaClassName = javaClassName;
    }

    public synchronized IJavaCode getJavaCode() {
        return javaCode;
    }

    public synchronized void setJavaCode(IJavaCode javaCode) {
        this.javaCode = javaCode;
    }

    public synchronized String getJavaDefaultClassName() {
        return javaDefaultClassName;
    }

    public synchronized void setBaseStackMachine(
            IStackMachineInformationProvider smip) {
        baseStackMachine = smip;
    }

    public synchronized IStackMachineInformationProvider getBaseStackMachine() {
        return baseStackMachine;
    }

    public synchronized Text getJavaStdInText() {
        return javaStdInText;
    }

    public synchronized StyledText getJavaStdOutText() {
        return javaStdOutText;
    }

    public synchronized CarrotPainter getCarrotPainter() {
        return carrotPainter;
    }

    public void positionBlueCarrot(final ISourceCodeLocation loc) {
        if (Thread.currentThread().equals(
                getInterpreterTextWidget().getDisplay().getThread())) {

            synchronized (this) {
                int row = loc.getLineNumber() - 1;
                int col = loc.getColumnNumber() - 1;
                int offset = getInterpreterTextWidget().getOffsetAtLine(row)
                        + col;
                getCarrotPainter().setBlueCarrotOffset(offset);
                getInterpreterInspector().refresh();
            }
        } else {
            getInterpreterTextWidget().getDisplay().syncExec(new Runnable() {

                public void run() {
                    positionBlueCarrot(loc);
                }
            });
        }
    }

    public void positionBlueCarrot(final int offset) {
        if (Thread.currentThread().equals(
                getInterpreterTextWidget().getDisplay().getThread())) {

            synchronized (this) {
                getCarrotPainter().setBlueCarrotOffset(offset);
                getInterpreterInspector().refresh();
            }
        } else {
            getInterpreterTextWidget().getDisplay().syncExec(new Runnable() {

                public void run() {
                    positionBlueCarrot(offset);
                }
            });
        }
    }

    public synchronized Button getInterpreterInterpretButton() {
        return interpreterInterpretButton;
    }

    public synchronized StyledText getInterpreterStdOutText() {
        return interpreterStdOutText;
    }

    public synchronized Text getInterpreterStdInText() {
        return interpreterStdInText;
    }

    public synchronized Button getInterpreterEOFButton() {
        return interpreterEOFButton;
    }

    public synchronized Button getInterpreterTerminateButton() {
        return interpreterTerminateButton;
    }

    public synchronized SelectionListener getTabSelectionListener() {
        return tabSelectionListener;
    }

    public synchronized Button getInterpreterStepModeButton() {
        return interpreterStepModeButton;
    }

    public synchronized Button getInterpreterContinueButton() {
        return interpreterContinueButton;
    }

    public synchronized IStackMachine getStackMachine() {
        return stackMachine;
    }

    public void setStackMachine(final IStackMachine stackMachine) {
        if (Thread.currentThread()
                .equals(
                        getInterpreterInspector().getControl().getDisplay()
                                .getThread())) {
            synchronized (this) {
                this.stackMachine = stackMachine;
                interpreterInspector.setInput(stackMachine);
                interpreterList.setInput(stackMachine);
            }
        } else {
            getInterpreterInspector().getControl().getDisplay().syncExec(
                    new Runnable() {

                        public void run() {
                            setStackMachine(stackMachine);
                        }
                    });
        }
    }

    public synchronized IButtonManager getButtonManager() {
        return buttonManager;
    }

    public synchronized Button getJavaExecuteButton() {
        return javaExecuteButton;
    }

    public synchronized Button getJavaEOFButton() {
        return javaEOFButton;
    }

    public synchronized Button getJavaTerminateButton() {
        return javaTerminateButton;
    }

    public synchronized void setEditorFonts(Font font) {
        if (kenyaText != null) {
            kenyaText.setFont(font);
            kenyaNumbers.setFont(font);
        }
        if (kenyaHumanErrorTableManager != null) {
            kenyaHumanErrorTableManager.getControl().setFont(font);
            kenyaJavacErrorTable.getControl().setFont(font);
        }
        if (javaText != null) {
            javaText.setFont(font);
            javaNumbers.setFont(font);
        }
        if (interpreterText != null) {
            interpreterText.setFont(font);
            interpreterNumbers.setFont(font);
        }
        if (interpreterInspector != null) {
            interpreterInspector.getControl().setFont(font);
            interpreterList.getControl().setFont(font);
        }
    }

    public synchronized void setIOFonts(Font font) {
        if (javaText != null) {
            javaStdOutText.setFont(font);
            javaStdInText.setFont(font);
        }
        if (interpreterText != null) {
            interpreterStdOutText.setFont(font);
            interpreterStdInText.setFont(font);
        }
    }

    public synchronized void switchToTab(int tab) {
        if (tab == KENYATABINDEX || tab == JAVATABINDEX
                || tab == INTERPRETERTABINDEX) {
            if (topTabs != null && getTopTabs().getSelectionIndex() != tab) {
                getTopTabs().setSelection(tab);
                tabSelectionListener.widgetSelected(null);
            }
        }
    }

    public synchronized FileLoaderSaver getJavaFileLoaderSaver() {
        return javaFileLoaderSaver;
    }

    public synchronized void setJavaFileLoaderSaver(
            FileLoaderSaver javaFileLoaderSaver) {
        this.javaFileLoaderSaver = javaFileLoaderSaver;
    }

    public synchronized Button getInterpreterTrackButton() {
        return interpreterTrackButton;

    }

    public boolean getInterpreterTrackMode() {
        synchronized (secondaryLock) {
            return interpreterTrackMode;
        }
    }

    private void setInterpreterTrackMode(boolean interpreterTrackMode) {
        synchronized (secondaryLock) {
            this.interpreterTrackMode = interpreterTrackMode;
        }
    }

    private String getKenyaWidgetText() {
        synchronized (secondaryLock) {
            return kenyaWidgetText;
        }
    }

    private void setKenyaWidgetText(String kenyaWidgetText) {
        synchronized (secondaryLock) {
            this.kenyaWidgetText = kenyaWidgetText;
        }
    }

    public synchronized TreeViewer getInterpreterInspector() {
        return interpreterInspector;
    }

    public synchronized ListViewer getInterpreterList() {
        return interpreterList;
    }

    public synchronized Text getInterpreterArgsText() {
        return interpreterArgsText;
    }

    public synchronized Text getJavaArgsText() {
        return javaArgsText;
    }

    private String[] getArgs(String input) {
        Pattern p = Pattern.compile("[ ]*([^ ]+)[ ]*");
        Matcher m = p.matcher(input);
        m.reset();
        List<String> args = new ArrayList<String>();
        while (m.find()) {
            args.add(m.group(1));
        }
        return args.toArray(new String[args.size()]);
    }

    public String[] getJavaArgs() {
        if (Thread.currentThread().equals(
                getJavaArgsText().getDisplay().getThread())) {
            setJavaArgsResult(getArgs(getJavaArgsText().getText()));
        } else {
            getJavaArgsText().getDisplay().syncExec(new Runnable() {

                public void run() {
                    getJavaArgs();
                }
            });
        }
        return getJavaArgsResult();
    }

    public String[] getInterpreterArgs() {
        if (Thread.currentThread().equals(
                getInterpreterArgsText().getDisplay().getThread())) {
            setInterpreterArgsResult(getArgs(getInterpreterArgsText().getText()));
        } else {
            getInterpreterArgsText().getDisplay().syncExec(new Runnable() {

                public void run() {
                    getInterpreterArgs();
                }
            });
        }
        return getInterpreterArgsResult();
    }

    private void setJavaArgsResult(String[] javaArgs) {
        synchronized (secondaryLock) {
            this.javaArgsResult = javaArgs;
        }
    }

    private String[] getJavaArgsResult() {
        synchronized (secondaryLock) {
            return javaArgsResult;
        }
    }

    private void setInterpreterArgsResult(String[] interpreterArgs) {
        synchronized (secondaryLock) {
            this.interpreterArgsResult = interpreterArgs;
        }
    }

    private String[] getInterpreterArgsResult() {
        synchronized (secondaryLock) {
            return interpreterArgsResult;
        }
    }

    public void saveKenyaAs(final ISaveKenyaContinuation continuation) {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                Shell shell = new Shell();
                FileDialog dialog = new FileDialog(shell, SWT.SAVE);
                dialog.setFilterExtensions(new String[] { "*.k" });
                dialog.setText("Save Kenya file");

                if (lastDir == null) {
                    if (getKenyaFile() != null) {
                        lastDir = getKenyaFile().getDir();
                        if (lastDir != null && lastDir.length() == 0)
                            lastDir = null;
                    }
                }
                if (lastDir != null) dialog.setFilterPath(lastDir);

                if (dialog.open() != null) {
                    lastDir = dialog.getFilterPath();

                    File file = new File(dialog.getFilterPath(), dialog
                            .getFileName());
                    if (getKenyaFile() == null) {
                        FileLoaderSaver fls = new FileLoaderSaver();
                        fls.setPath(file.getAbsolutePath());
                        setKenyaFile(fls);
                    } else {
                        FileLoaderSaver fls = getKenyaFile();
                        fls.setPath(file.getAbsolutePath());
                    }
                    setJavaFileLoaderSaver(null);
                    saveKenya(continuation);
                } else if (null != continuation) {
                    continuation.invoke(false, this);
                }
            }
        } else {
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {

                    saveKenyaAs(continuation);
                }
            });
        }
    }

    public void saveKenya(final ISaveKenyaContinuation continuation) {
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                FileLoaderSaver fls = getKenyaFile();
                if (fls == null || fls.getPath() == null) {
                    saveKenyaAs(continuation);
                } else {
                    fls.saveFile(getKenyaTextWidget().getText());
                    setModified(false);
                    safeSetStatus("Kenya code saved to " + fls.getPath());
                    if (null != continuation) {
                        continuation.invoke(true, this);
                    }
                }
            }
        } else {
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {
                    saveKenya(continuation);
                }
            });
        }
    }

    public void saveJavaAs() {
        if (!isJavaGoodToSave()) {
            return;
        }
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (!isJavaGoodToSave()) {
                    return;
                }
                Shell shell = new Shell();
                FileDialog dialog = new FileDialog(shell, SWT.SAVE);
                dialog.setFilterExtensions(new String[] { "*.java" });
                dialog.setText("Save Java translation file");

                if (lastDir == null) {
                    if (getJavaFileLoaderSaver() != null) {
                        lastDir = getJavaFileLoaderSaver().getDir();
                        if (lastDir.length() == 0) lastDir = null;
                    }
                }
                if (lastDir != null) dialog.setFilterPath(lastDir);

                if (dialog.open() != null) {
                    lastDir = dialog.getFilterPath();

                    File file = new File(dialog.getFilterPath(), dialog
                            .getFileName());
                    if (getJavaFileLoaderSaver() == null) {
                        FileLoaderSaver fls = new FileLoaderSaver();
                        fls.setPath(file.getAbsolutePath());
                        setJavaFileLoaderSaver(fls);
                    } else {
                        FileLoaderSaver fls = getJavaFileLoaderSaver();
                        fls.setPath(file.getAbsolutePath());
                    }
                    saveJava();
                }
            }
        } else {
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {

                    saveJavaAs();
                }
            });
        }
    }

    public void saveJava() {
        if (!isJavaGoodToSave()) {
            return;
        }
        if (Thread.currentThread()
                .equals(getTopTabs().getDisplay().getThread())) {
            synchronized (this) {
                if (!isJavaGoodToSave()) {
                    return;
                }
                Shell shell = new Shell();
                FileLoaderSaver fls = getJavaFileLoaderSaver();
                if (fls == null || fls.getPath() == null) {
                    saveJavaAs();
                    return;
                } else {
                    try {
                        String name = fls.getName();
                        if (name.indexOf('.') != -1) {
                            name = name.substring(0, name.indexOf('.'));
                        }
                        if (getJavaCode() != null) {
                            BufferedReader javaReader = new BufferedReader(
                                    getJavaCode().getCode(name));
                            StringBuffer javaBuffer = new StringBuffer();
                            int c = javaReader.read();
                            while (c != -1) {
                                javaBuffer.append((char) c);
                                c = javaReader.read();
                            }
                            javaReader.close();
                            fls.saveFile(javaBuffer.toString());
                            safeSetStatus("Java translation saved to "
                                    + fls.getPath());
                        } else {
                            MessageBox message = new MessageBox(shell, SWT.OK);
                            message.setText("Unable to save file.");
                            message
                                    .setMessage("Can't save translation of faulty code.");
                            message.open();
                            safeSetStatus("Can't save translation of faulty code.");
                        }
                    } catch (IOException e) {
                        safeSetStatus("IOException occured when translating Kenya.");
                    } catch (IllegalArgumentException e) {
                        MessageBox message = new MessageBox(shell, SWT.OK);
                        message.setText("Unable to save file.");
                        message
                                .setMessage("Can not save Java into invalid file name '"
                                        + fls.getName() + "'");
                        message.open();
                        safeSetStatus("Can not save Java into invalid file name '"
                                + fls.getName() + "'");
                    }
                }
            }
        } else {
            getTopTabs().getDisplay().syncExec(new Runnable() {

                public void run() {

                    saveJava();
                }
            });
        }
    }

    public ExecutorService getExecutorService() {
        return executor;
    }

    public synchronized void setSaveKenyaActionHandler(
            ISaveKenyaActionHandler saveHandler) {
        kenyaSaveHandler = saveHandler;
    }

    public synchronized ISaveKenyaActionHandler getSaveKenyaActionHandler() {
        return kenyaSaveHandler;
    }

    public synchronized void setSaveableJava(boolean b) {
        saveableJava = b;
        if (null != javaSaveAsAction) {
            javaSaveAsAction.setEnabled(isJavaGoodToSave() && saveableJava);
        }
    }

    private synchronized void setJavaGoodToSave(boolean b) {
        if (null != javaSaveAsAction) {
            javaSaveAsAction.setEnabled(b && saveableJava);
        }
    }

    public synchronized boolean isJavaGoodToSave() {
        try {
            ICheckedCode checker = Mediator.check(new StringReader(
                    getKenyaText()));
            setKenyaCheckedCode(checker);
            if (!checker.isErroredCode()) {
                setJavaCode(checker.translate());
            } else {
                setJavaCode(null);
            }
            return !checker.isErroredCode();
        } catch (IOException e) {
            return false;
        }
    }
}