/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on Aug 4, 2004
 *
 */
package uk.ac.imperial.doc.kenya.gui.editor.indenter;

import java.util.regex.Matcher;

/**
 * @author Matthew Sackman
 * @version 1
 */
public class LineIndentation implements ILineIndentation {

    private ILineIndentation next = null;

    private ILineIndentation previous = null;

    private boolean complete = false;

    private boolean reduce = false;

    private int depth = 0;

    private String indentation = "";

    private boolean blockStart = false;

    private boolean blockEnd = false;

    private int lineNumber = 0;

    public synchronized void setNextLine(ILineIndentation nextLine) {
        if (next != null) {
            next.setPreviousLine(null);
        }
        next = nextLine;
        recount();
    }

    public synchronized void setPreviousLine(ILineIndentation previousLine) {
        if (previous != null) {
            previous.setNextLine(null);
        }
        previous = previousLine;
        recount();
    }

    public synchronized ILineIndentation getNextLine() {
        return next;
    }

    public synchronized ILineIndentation getPreviousLine() {
        return previous;
    }

    public synchronized boolean isComplete() {
        return complete;
    }

    public synchronized String getNextLineIndentation() {
        return indentation;
    }

    public synchronized void parselineText(String thisLineText,
            String nextLineText) {
        complete = false;
        reduce = false;
        indentation = "";
        depth = 0;
        if (BLOCKSTART.matcher(thisLineText).matches()) {
            blockStart = true;
        } else {
            blockStart = false;
        }
        if (BLOCKEND.matcher(thisLineText).matches()) {
            blockEnd = true;
        } else {
            blockEnd = false;
        }
        if (isInBlock()) {
            complete = true;
            if (previous == null)
                depth = 0;
            else
                depth = previous.getDepth();

        }
        if ((isInBlock() && blockStart) || !isInBlock()) {
            if (SEMICOLONEND.matcher(thisLineText).matches()) {
                complete = true;
            } else if (DECREASINGEND.matcher(thisLineText).matches()) {
                complete = true;
            } else if (INCREASINGEND.matcher(thisLineText).matches()) {
                complete = true;
            } else if (SPACELINE.matcher(thisLineText).matches()) {
                if (previous == null)
                    complete = true;
                else
                    complete = previous.isComplete();
            }

            int braceSkew = findBraceSkew(thisLineText);

            if (nextLineText != null
                    && INCREASINGSTART.matcher(nextLineText).matches()
                    && !complete
                    && findBraceSkew(nextLineText) > 0) reduce = true;

            if (previous == null) {
                depth = braceSkew;
            } else {
                depth = previous.getDepth() + braceSkew;
            }

            if (nextLineText != null
                    && DECREASINGSTART.matcher(nextLineText).matches()) {
                reduce = true;
            }
        }

        depth = Math.max(0, depth);

        indentation = "";
        int count = 0;
        if (reduce) count++;
        if (!complete) count--;
        for (; count < depth; count++)
            indentation += UNIT;

    }

    private int findBraceSkew(String text) {
        int result = 0;
        Matcher inc = INCREASEREMOVE.matcher(text);
        while (inc.matches()) {
            result++;
            text = inc.group(1) + inc.group(2);
            inc = INCREASEREMOVE.matcher(text);
        }
        Matcher dec = DECREASEREMOVE.matcher(text);
        while (dec.matches()) {
            result--;
            text = dec.group(1) + dec.group(2);
            dec = DECREASEREMOVE.matcher(text);
        }
        return result;
    }

    public synchronized int getLineNumber() {
        return lineNumber;
    }

    public synchronized int getDepth() {
        return depth;
    }

    public synchronized ILineIndentation getLine(int lineNumber) {
        int myLineNumber = getLineNumber();
        if (myLineNumber < lineNumber) {
            if (next == null)
                return null;
            else
                return next.getLine(lineNumber);
        } else if (myLineNumber > lineNumber) {
            if (previous == null)
                return null;
            else
                return previous.getLine(lineNumber);
        } else {
            return this;
        }
    }

    public synchronized boolean isBlockStart() {
        return blockStart;
    }

    public synchronized boolean isBlockEnd() {
        return blockEnd;
    }

    public synchronized boolean isInBlock() {
        if (previous == null) {
            return (blockStart || blockEnd);
        } else {
            if (blockStart) return true;
            if (blockEnd) return true;
            if (previous.isBlockEnd()) return false;
            return previous.isInBlock();
        }
    }

    public synchronized void recount() {
        if (previous != null) {
            previous.recount();
        } else {
            setLineNumber(0);
        }
    }

    public synchronized void setLineNumber(int lineNumber) {
        this.lineNumber = lineNumber;
        if (next != null) {
            next.setLineNumber(lineNumber + 1);
        }
    }

}