/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 *               Matthew Sackman - addition of Generic handling code
 ******************************************************************************/
package uk.ac.imperial.doc.kenya.gui.editor.utils;

import java.io.IOException;
import java.io.StringReader;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import org.eclipse.swt.custom.LineStyleListener;
import org.eclipse.swt.graphics.Color;

public class JavaLineStyler extends LineStyler implements LineStyleListener {

    Vector<int[]> blockComments = new Vector<int[]>();

    public JavaLineStyler() {
        initializeJavaColors();
        scanner = new JavaScanner();
    }

    Color getColor(int type) {
        if (type < 0 || type >= tokenColors.length) {
            return null;
        }
        return colors[tokenColors[type]];
    }

    boolean inBlockComment(int start, int end) {
        for (int i = 0; i < blockComments.size(); i++) {
            int[] offsets = blockComments.elementAt(i);
            // start of comment in the line
            if ((offsets[0] >= start) && (offsets[0] <= end))
                return true;
            // end of comment in the line
            if ((offsets[1] >= start) && (offsets[1] <= end))
                return true;
            if ((offsets[0] <= start) && (offsets[1] >= end))
                return true;
        }
        return false;
    }

    void initializeJavaColors() {
        initializeColors(9);
    }
    
    void disposeColors() {
        for (int i = 0; i < colors.length; i++) {
            colors[i].dispose();
        }
    }

    public void parseBlockComments(String text) {
        blockComments = new Vector<int[]>();
        StringReader buffer = new StringReader(text);
        int ch;
        boolean blkComment = false;
        int cnt = 0;
        int[] offsets = new int[2];
        boolean done = false;

        try {
            while (!done) {
                switch (ch = buffer.read()) {
                case -1: {
                    if (blkComment) {
                        offsets[1] = cnt;
                        blockComments.addElement(offsets);
                    }
                    done = true;
                    break;
                }
                case '/': {
                    ch = buffer.read();
                    if ((ch == '*') && (!blkComment)) {
                        offsets = new int[2];
                        offsets[0] = cnt;
                        blkComment = true;
                        cnt++;
                    } else {
                        cnt++;
                    }
                    cnt++;
                    break;
                }
                case '*': {
                    if (blkComment) {
                        ch = buffer.read();
                        cnt++;
                        if (ch == '/') {
                            blkComment = false;
                            offsets[1] = cnt;
                            blockComments.addElement(offsets);
                        }
                    }
                    cnt++;
                    break;
                }
                default: {
                    cnt++;
                    break;
                }
                }
            }
        } catch (IOException e) {
            // ignore errors
        }
    }

    /**
     * A simple fuzzy scanner for Java
     */
    public class JavaScanner extends GenericScanner implements Scanner{

        protected Hashtable<String, Integer> fgKeys = null;

        protected StringBuffer fBuffer = new StringBuffer();

        protected String fDoc;

        protected int fPos;

        protected int fEnd;

        protected int fStartToken;

        protected boolean fEofSeen = false;

        private String[] fgKeywords = { "abstract", "boolean", "break", "byte",
                "case", "catch", "char", "class", "continue", "default", "do",
                "double", "else", "extends", "false", "final", "finally",
                "float", "for", "if", "implements", "import", "instanceof",
                "int", "interface", "long", "native", "new", "null", "package",
                "private", "protected", "public", "return", "short", "static",
                "super", "switch", "synchronized", "this", "throw", "throws",
                "transient", "true", "try", "void", "volatile", "while" };

        public JavaScanner() {
            initialize();
        }

        /**
         * Returns the ending location of the current token in the document.
         */
        public final int getLength() {
            return fPos - fStartToken;
        }

        /**
         * Initialize the lookup table.
         */
        void initialize() {
            fgKeys = new Hashtable<String, Integer>();
            Integer k = new Integer(KEY);
            for (int i = 0; i < fgKeywords.length; i++)
                fgKeys.put(fgKeywords[i], k);
        }

        /**
         * Returns the starting location of the current token in the document.
         */
        public final int getStartOffset() {
            return fStartToken;
        }

        /**
         * Returns the next lexical token in the document.
         */
        public int nextToken() {
            int c;
            fStartToken = fPos;
            while (true) {
                switch (c = read()) {
                case EOF:
                    return EOF;
                case '<': // generic
                    return nextGenericToken(c, GENERIC, EOF, EOL, OTHER);
                case '/': // comment
                    c = read();
                    if (c == '/') {
                        while (true) {
                            c = read();
                            if ((c == EOF) || (c == EOL)) {
                                unread(c);
                                return COMMENT;
                            }
                        }
                    } else {
                        unread(c);
                    }
                    return OTHER;
                case '\'': // char const
                    for (;;) {
                        c = read();
                        switch (c) {
                        case '\'':
                            return STRING;
                        case EOF:
                            unread(c);
                            return STRING;
                        case '\\':
                            c = read();
                            break;
                        }
                    }

                case '"': // string
                    for (;;) {
                        c = read();
                        switch (c) {
                        case '"':
                            return STRING;
                        case EOF:
                            unread(c);
                            return STRING;
                        case '\\':
                            c = read();
                            break;
                        }
                    }

                case '0':
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9':
                    return nextDigitToken(NUMBER);
                default:
                    if (Character.isWhitespace((char) c)) {
                        do {
                            c = read();
                        } while (Character.isWhitespace((char) c));
                        unread(c);
                        return WHITE;
                    }
                    if (Character.isJavaIdentifierStart((char) c)) {
                        fBuffer.setLength(0);
                        do {
                            fBuffer.append((char) c);
                            c = read();
                        } while (Character.isJavaIdentifierPart((char) c));
                        unread(c);
                        Integer i = fgKeys.get(fBuffer.toString());
                        if (i != null)
                            return i.intValue();
                        return WORD;
                    }
                    return OTHER;
                }
            }
        }

        /**
         * Returns next character.
         */
        protected int read() {
            if (fPos <= fEnd) {
                return fDoc.charAt(fPos++);
            }
            return EOF;
        }

        public void setRange(String text) {
            fDoc = text;
            fPos = 0;
            fEnd = fDoc.length() - 1;
        }

        protected void unread(int c) {
            if (c != EOF)
                fPos--;
        }

        protected void unreadBuffer(List<Integer> buff) {
            for (int idx = buff.size() - 1; idx >= 0; idx--) {
                unread(buff.get(idx).intValue());
            }
        }

        public int nextToken(int lineOffset) {
            return nextToken();
        }
    }

}

