/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation, Matthew
 * Sackman - customisation to kenya
 ******************************************************************************/
package uk.ac.imperial.doc.kenya.gui.editor.utils;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;

import org.eclipse.swt.custom.LineStyleListener;
import org.eclipse.swt.custom.StyledTextContent;
import org.eclipse.swt.graphics.Point;

import uk.ac.imperial.doc.kenya.gui.editor.indenter.ILineIndentation;
import uk.ac.imperial.doc.kenya.io.VariablePushbackReader;

public class KenyaLineStyler extends LineStyler implements LineStyleListener {

    private List<Point> blockComments = new ArrayList<Point>();
    public static final int OPERATOR = 9;


    public KenyaLineStyler() {
        initializeKenyaColors();
        scanner = new KenyaScanner();
    }


    boolean inBlockComment(int start, int end) {
        for (int idx = 0; idx < blockComments.size(); idx++) {
            Point p = blockComments.get(idx);
            if (start >= p.x && start <= p.y && end >= p.x && end <= p.y)
                return true;
        }
        return false;
    }

    void initializeKenyaColors() {
        initializeColors(10);
        tokenColors[OPERATOR] = 5;

    }

    void disposeColors() {
        for (int i = 0; i < colors.length; i++) {
            colors[i].dispose();
        }
    }

    public void parseBlockComments(StyledTextContent content) {
        blockComments.clear();
        Point currentPoint = null;

        for (int lineNumber = 0; lineNumber < content.getLineCount(); lineNumber++) {
            String line = content.getLine(lineNumber);
            int offset = content.getOffsetAtLine(lineNumber);
            currentPoint = extractCommentsFromLine(line, currentPoint, offset,
                    content);
        }

        if (currentPoint != null) {
            currentPoint.y = content.getCharCount() - 1;
            blockComments.add(currentPoint);
        }
    }

    private Point extractCommentsFromLine(String line, Point currentPoint,
            int offset, StyledTextContent content) {
        Matcher blockEnd = ILineIndentation.BLOCKEND.matcher(line);
        Matcher blockStart = ILineIndentation.BLOCKSTART.matcher(line);
        if (blockEnd.matches()) {
            String prefix = blockEnd.group(1);
            if (currentPoint == null) {
                blockStart = ILineIndentation.BLOCKSTART.matcher(prefix);
                if (blockStart.matches()) {
                    String prePrefix = blockStart.group(1);
                    currentPoint = new Point(offset + prePrefix.length(),
                            offset + prefix.length() + 2);
                    blockComments.add(currentPoint);
                    currentPoint = null;
                }
            } else {
                currentPoint.y = offset + prefix.length() + 2;
                blockComments.add(currentPoint);
                currentPoint = null;
            }
            String suffix = blockEnd.group(2);
            if (suffix.length() > 0) {
                currentPoint = extractCommentsFromLine(suffix, currentPoint,
                        offset + prefix.length() + 2, content);
            }
        } else if (blockStart.matches()) {
            if (currentPoint == null) {
                int prefixLength = blockStart.group(1).length();
                currentPoint = new Point(offset + prefixLength, 0);
            }
        }
        return currentPoint;
    }

    /**
     * A simple fuzzy scanner for Kenya
     */
    public class KenyaScanner extends GenericScanner implements Scanner {

        protected Map<String, Integer> fgKeys = null;

        protected final StringBuffer fBuffer = new StringBuffer();

        protected VariablePushbackReader reader;

        protected int charsRead;

        protected int length;

        protected int currentTokenStartPosition;

        protected boolean fEofSeen = false;

        protected boolean lastCharWasWhitespace = false;

        private final String[] fgKeywords = { "boolean", "char", "int",
                "double", "String", "void", "class", "const", "null" };

        private final String[] fgOperators = { "if", "else", "while", "return",
                "switch", "case", "break", "default", "for", "assert", "new",
                "enum", "true", "false", };

        public KenyaScanner() {
            initialize();
        }

        /**
         * Returns the ending location of the current token in the document.
         */
        public final int getLength() {
            return charsRead - currentTokenStartPosition;
        }

        /**
         * Initialize the lookup table.
         */
        void initialize() {
            fgKeys = new Hashtable<String, Integer>();
            Integer k = new Integer(KEY);
            for (String key: fgKeywords)
                fgKeys.put(key, k);

            Integer o = new Integer(OPERATOR);
            for (String op: fgOperators)
                fgKeys.put(op, o);
        }

        /**
         * Returns the starting location of the current token in the document.
         */
        public final int getStartOffset() {
            return currentTokenStartPosition;
        }

        /**
         * Returns the next lexical token in the document.
         */
        public int nextToken(int lineOffset) {
            boolean prevWhiteSpace = lastCharWasWhitespace;
            lastCharWasWhitespace = false;
            int c;
            int preReadCount = 0;
            currentTokenStartPosition = charsRead;

            if (currentTokenStartPosition == 0
                    && inBlockComment(lineOffset, lineOffset)) {
                int d = read();
                if (d == EOF) return EOF;
                while (d != '/' && d != '*' && d != EOL && d != EOF) {
                    d = read();
                    preReadCount++;
                }
                unread(d);
                if (d == EOL || d == EOF) return COMMENT;
            }

            while (true) {

                switch (c = read()) {
                case EOF:
                    return EOF;
                case '<': // generic
                    return nextGenericToken(c, GENERIC, EOF, EOL, OTHER);
                case '/': // comment
                    c = read();
                    if (c == '/') {
                        while (true) {
                            c = read();
                            if ((c == EOF) || (c == EOL)) {
                                unread(c);
                                return COMMENT;
                            }
                        }
                    } else if (c == '*') { // block comment
                        for (int idx = 0; idx < blockComments.size(); idx++) {
                            Point p = blockComments.get(idx);
                            if (p.x == currentTokenStartPosition + lineOffset) {
                                int target = p.y;
                                int count = 2; // already read '/' and '*'
                                while (currentTokenStartPosition + lineOffset
                                        + count <= target
                                        && c != EOL && c != EOF) {
                                    c = read();
                                    count++;
                                }
                                return COMMENT;
                            }
                        }
                        unread(c);
                    } else {
                        unread(c);
                    }
                    return OTHER;
                case '*': // end block comment
                    c = read();
                    if (c == '/') {
                        for (int idx = 0; idx < blockComments.size(); idx++) {
                            Point p = blockComments.get(idx);
                            if (p.y == currentTokenStartPosition + lineOffset
                                    + preReadCount + 2) return COMMENT;
                        }
                        unread(c);
                    } else {
                        unread(c);
                    }
                    return OTHER;
                case '\'': // char const
                    for (;;) {
                        c = read();
                        switch (c) {
                        case '\'':
                            return STRING;
                        case EOF:
                            unread(c);
                            return STRING;
                        case '\\':
                            c = read();
                            break;
                        }
                    }

                case '"': // string
                    for (;;) {
                        c = read();
                        switch (c) {
                        case '"':
                            return STRING;
                        case EOF:
                            unread(c);
                            return STRING;
                        case '\\':
                            c = read();
                            break;
                        }
                    }

                case '0':
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9':
                    return nextDigitToken(NUMBER);
                default:
                    if (Character.isWhitespace((char) c)) {
                        do {
                            c = read();
                        } while (Character.isWhitespace((char) c));
                        unread(c);
                        lastCharWasWhitespace = true;
                        return WHITE;
                    }
                    if (isKenyaIdentifierStart((char) c) && (prevWhiteSpace || charsRead == 1)) {
                        fBuffer.setLength(0);
                        do {
                            fBuffer.append((char) c);
                            c = read();
                        } while (isKenyaIdentifierPart((char) c));
                        unread(c);
                        if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z')
                                || (c >= 'A' && c <= 'Z') || c == '_')
                            return OTHER;
                        Integer i = fgKeys.get(fBuffer.toString());
                        if (i != null) return i.intValue();
                        return WORD;
                    }
                    return OTHER;
                }
            }
        }

        private boolean isKenyaIdentifierStart(char c) {
            for (String key: fgKeys.keySet()) {
                if (key.startsWith("" + c)) return true;
            }
            return false;
        }

        private boolean isKenyaIdentifierPart(char c) {
            for (String key: fgKeys.keySet()) {
                if (key.indexOf(c) != -1) return true;
            }
            return false;
        }

        /**
         * Returns next character.
         */
        protected int read() {
            try {
                if (reader.ready()) {
                    charsRead++;
                    return reader.read();
                }
            } catch (IOException e) {
                return EOF;
            }
            return EOF;
        }

        public void setRange(String text) {
            reader = new VariablePushbackReader(new StringReader(text));
            charsRead = 0;
            currentTokenStartPosition = 0;
            length = text.length();
        }

        protected void unread(int c) {
            if (c != EOF) reader.unread(c);

            charsRead--;
            if (charsRead < 0) {
                System.err.println("CharsRead goes -ve (" + charsRead
                        + ") when unreading " + c);
            }
        }

        protected void unreadBuffer(List<Integer> buff) {
            for (int idx = buff.size() - 1; idx >= 0; idx--) {
                unread(buff.get(idx).intValue());
            }
        }
    }

}