/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on Sep 7, 2004
 *
 */
package uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;

import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IMethodScope;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IArrayType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IClassInstanceType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IClassStaticType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IEnumType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IGenericType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.INonPrimitiveType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.INullType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IPrimitiveType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IStringType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * @author matthew
 */
public class InterpreterInspectorContentProvider implements
        ITreeContentProvider {

    private static final Object[] EMPTY_ARRAY = new Object[] {};

    private final Map<Map.Entry<String, IType>, Object> parentMap = new HashMap<Map.Entry<String, IType>, Object>();

    public synchronized Object[] getChildren(Object parentElement) {
        if (parentElement == null)
            return null;
        // Should be List<Map.Entry<String, IType>> ?
        
        List<Map.Entry<String, IType>> children = new ArrayList<Map.Entry<String, IType>>();
        if (parentElement instanceof IMethodScope) {
            IMethodScope currentMethod = (IMethodScope) parentElement;

            IInterpretedClass classStatic = currentMethod.getCurrentClass();
            if (classStatic != null) {
                children.addAll(classStatic.getImmutableStaticVariables()
                        .entrySet());
                children.addAll(classStatic.getMutableStaticVariables()
                        .entrySet());
            }

            if (!currentMethod.isStatic()) {
                IInterpretedClassInstance classInstance = currentMethod
                        .getCurrentClassInstance();
                if (classInstance != null) {
                    children.addAll(classInstance
                            .getImmutableInstanceVariables().entrySet());
                    children.addAll(classInstance.getMutableInstanceVariables()
                            .entrySet());
                }
            }

            children.addAll(currentMethod.getLocalImmutableVariables()
                    .entrySet());
            children
                    .addAll(currentMethod.getLocalMutableVariables().entrySet());

        } else if (parentElement instanceof Map.Entry) {
            @SuppressWarnings("unchecked")
            IType value = ((Map.Entry<Object, IType>) parentElement).getValue();
            if (value instanceof IArrayType) {
                IType[] array = ((IArrayType) value).getValue();
                Map<String, Integer> indexMap = new HashMap<String, Integer>();
                Map<String, IType> valueMap = new HashMap<String, IType>();
                for (int idx = 0; idx < array.length; idx++) {
                    String index = "[" + idx + "]";
                    indexMap.put(index, new Integer(idx));
                    valueMap.put(index, array[idx]);
                    children.add(new KenyaSimpleEntry<String, IType>(index, array[idx])); // ensure children's size
                }
                
                for (Map.Entry<String, IType> entry: valueMap.entrySet()) {
                    children.set(indexMap.get(entry.getKey()), entry);
                }
                indexMap.clear();
                valueMap.clear();

            } else if (value instanceof IClassInstanceType) {
                IInterpretedClassInstance classInstance = ((IClassInstanceType) value)
                        .getValue();
                children.addAll(classInstance.getInterpretedClass()
                        .getImmutableStaticVariables().entrySet());
                children.addAll(classInstance.getInterpretedClass()
                        .getMutableStaticVariables().entrySet());

                children.addAll(classInstance.getImmutableInstanceVariables()
                        .entrySet());
                children.addAll(classInstance.getMutableInstanceVariables()
                        .entrySet());

            } else if (value instanceof IClassStaticType) {
                IInterpretedClass classStatic = ((IClassStaticType) value)
                        .getValue();
                children.addAll(classStatic.getImmutableStaticVariables()
                        .entrySet());
                children.addAll(classStatic.getMutableStaticVariables()
                        .entrySet());

            } else {
                System.err.println("Unknown Map.Entry type for getChildren: "
                        + parentElement);
            }
        } else {
            System.err
                    .println("Unknown type for getChildren: " + parentElement);
        }
        @SuppressWarnings("unchecked")
        Map.Entry<String, IType>[] childrenArray = children.toArray(new Map.Entry[0]);
        for (int idx = 0; idx < childrenArray.length; idx++) {
            parentMap.put(childrenArray[idx], parentElement);
        }
        return childrenArray;
    }

    public synchronized Object getParent(Object element) {
        if (element == null)
            return null;
        return parentMap.get(element);
    }

    public boolean hasChildren(Object element) {
        if (element == null)
            return false;
        if (element instanceof Map.Entry) {
            @SuppressWarnings("unchecked")
            Map.Entry<Object, IType> entry = (Map.Entry<Object, IType>) element;
            if (entry.getValue() instanceof IPrimitiveType) {
                if (entry.getValue() instanceof IArrayType)
                    if (((IArrayType) entry.getValue()).getLength() != 0) {
                        return true;
                    } else {
                        return false;
                    }
                else
                    return false;
            } else if (entry.getValue() instanceof INullType) {
                return false;
            } else if (entry.getValue() instanceof IGenericType) {
                return false;
            } else if (entry.getValue() instanceof INonPrimitiveType) {
                if (entry.getValue() instanceof IEnumType)
                    return false;
                else if (entry.getValue() instanceof IStringType)
                    return false;
                else
                    return true;
            } else {
                System.err.println("Unknown entry type: " + element.toString());
                return false;
            }
        } else if (element instanceof IMethodScope) {
            return true;
        } else {
            System.err.println("Unknown type: " + element.toString());
            return false;
        }
    }

    public Object[] getElements(Object inputElement) {
        if (inputElement instanceof IStackMachine) {
            return getChildren(((IStackMachine) inputElement).getMethodScope());
        } else if (inputElement instanceof IMethodScope) {
            return getChildren(inputElement);
        } else {
            return EMPTY_ARRAY;
        }
    }

    public synchronized void dispose() {
        parentMap.clear();
    }

    public synchronized void inputChanged(Viewer viewer, Object oldInput,
            Object newInput) {
        parentMap.clear();
    }
    
    /* A straight copy of Java 6 AbstractMap.SimpleEntry.
     * We just do not want to be Java 6 dependent
     * FIXME: once we switch to Java 6, remove KenyaSimpleEntry
     */
    static class KenyaSimpleEntry<K,V>
    implements Map.Entry<K,V>, java.io.Serializable
    {
        private static final long serialVersionUID = -8499721149061103585L;
        
        private final K key;
        private V value;
        
        public KenyaSimpleEntry(K key, V value) {
            this.key = key;
            this.value = value;
        }
    
        public KenyaSimpleEntry(Map.Entry<K,V> e) {
            this.key = e.getKey();
            this.value = e.getValue();
        }
    
        public K getKey() {
            return key;
        }
    
        public V getValue() {
            return value;
        }
    
        public V setValue(V value) {
            V oldValue = this.value;
            this.value = value;
            return oldValue;
        }
    
        @SuppressWarnings("unchecked")
        public boolean equals(Object o) {
            if (!(o instanceof Map.Entry))
                return false;
            Map.Entry<K, V> e = (Map.Entry<K, V>)o;
            
            return eq(key, e.getKey()) && eq(value, e.getValue());
        }
    
        public int hashCode() {
            return ((key == null) ? 0 : key.hashCode()) ^
            ((value == null) ? 0 : value.hashCode());
        }
        
        public String toString() {
            return key + "=" + value;
        }
    
        private static boolean eq(Object o1, Object o2) {
            return (o1 == null ? o2 == null : o1.equals(o2));
        }
    }

}

