/* *******************************************************************************
 *   kenya                                                             *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/
/*
 * Created on 25-Aug-2004
 */
package uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.concurrent.ExecutorService;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Text;

import uk.ac.imperial.doc.kenya.gui.editor.EditingWindow;
import uk.ac.imperial.doc.kenya.gui.editor.utils.StdOutWriter;
import uk.ac.imperial.doc.kenya.mediator.stackMachine.StackMachineFactory;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import uk.ac.imperial.doc.kenya.stackMachine.IPointListener;
import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.types.ArrayTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.StringTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class InterpreterInterpretSelectionListener extends SelectionAdapter {

	private final Button interpret;

	private final EditingWindow window;

	private final StyledText stdOut;

	private final Text stdIn;

	private final Button eof;

	private final Button terminate;

	private final Runnable enableStuff;

	private final Runnable disableStuff;

	private SelectionListener eofListener = null;

	private KeyListener keyListener = null;

	private SelectionListener terminateListener = null;
    
    private final ExecutorService executor;

	public InterpreterInterpretSelectionListener(EditingWindow w) {

		window = w;
		stdOut = w.getInterpreterStdOutText();
		stdIn = w.getInterpreterStdInText();
		interpret = w.getInterpreterInterpretButton();
		eof = w.getInterpreterEOFButton();
		terminate = w.getInterpreterTerminateButton();
        executor = w.getExecutorService();

		enableStuff = new Runnable() {

			public void run() {
				interpret.setEnabled(false);
				stdIn.setEnabled(true);
				stdIn.setFocus();
				stdIn.setText("");
				terminate.setEnabled(true);
				terminate.setText("Terminate");
				eof.setEnabled(true);
				stdOut.setText("");
			}
		};

		disableStuff = new Runnable() {

			public void run() {
				interpret.setEnabled(true);
				stdIn.setEnabled(false);
				terminate.setText("Terminate");
				terminate.setEnabled(false);
				eof.setEnabled(false);
				window.getInterpreterContinueButton().setEnabled(false);
				IStackMachine sm = window.getStackMachine();
				if (sm != null) {
					sm.resume();
					window.setStackMachine(null);
				}
			}
		};
	}

	public synchronized void widgetSelected(SelectionEvent e) {
		interpret.setEnabled(false);

		final IStackMachine sm = (new StackMachineFactory()).createStackMachine(executor);

		final IStackMachineInformationProvider ismip = window
				.getBaseStackMachine();

		if (ismip == null)
			return;

		try {
			PipedInputStream pis = new PipedInputStream();
			PipedOutputStream pos = new PipedOutputStream(pis);

			final PrintStream outStream = new PrintStream(pos, true);
			sm.setOut(outStream);

			BufferedReader outReader = new BufferedReader(
					new InputStreamReader(pis), 16);
			Runnable stdOutReader = new StdOutWriter(outReader, stdOut,
					SWT.COLOR_BLUE);

			PipedInputStream pisErr = new PipedInputStream();
			PipedOutputStream posErr = new PipedOutputStream(pisErr);

			final PrintStream outStreamErr = new PrintStream(posErr, true);
			sm.setErr(outStreamErr);

			BufferedReader outErrReader = new BufferedReader(
					new InputStreamReader(pisErr), 16);
			Runnable stdErrReader = new StdOutWriter(outErrReader, stdOut,
					SWT.COLOR_RED);

			PipedOutputStream qout = new PipedOutputStream();
			PipedInputStream qin = new PipedInputStream(qout);
			final InputStream inStream = new BufferedInputStream(qin);
			sm.setIn(inStream);

			final PrintWriter inWriter = new PrintWriter(
					new OutputStreamWriter(qout), true);

			if (keyListener != null)
				stdIn.removeKeyListener(keyListener);
			keyListener = new KeyAdapter() {

				public void keyReleased(KeyEvent e) {
					if (e.character == SWT.CR) {
						e.doit = false;
						String text = stdIn.getText() + "\n";
						stdIn.setText("");

						stdOut.append(text);
						StyleRange sr = new StyleRange();
						sr.start = stdOut.getCharCount() - text.length();
						sr.length = text.length();
						sr.foreground = stdOut.getDisplay().getSystemColor(
								SWT.COLOR_DARK_GREEN);
						stdOut.setStyleRange(sr);
						stdOut.setCaretOffset(stdOut.getCharCount());
						stdOut.setSelection(stdOut.getCharCount());
						stdOut.showSelection();

						inWriter.write(text);
						inWriter.flush();
					}
				}
			};
			stdIn.addKeyListener(keyListener);

			if (eofListener != null)
				eof.removeSelectionListener(eofListener);

			eofListener = new SelectionAdapter() {

				public void widgetSelected(SelectionEvent e) {
					inWriter.close();
					stdIn.setEnabled(false);
					eof.setEnabled(false);
				}
			};
			eof.addSelectionListener(eofListener);

			if (terminateListener != null)
				terminate.removeSelectionListener(terminateListener);

			terminateListener = new InterpreterTerminateButtonSelectionListener(
					window, inWriter);
			terminate.addSelectionListener(terminateListener);

			Runnable launch = new Runnable() {

				public void run() {
					try {
						window.getTopTabs().getDisplay().syncExec(enableStuff);
						IInterpretedClass launchClass = ismip
								.getEntryPointClass();
						String launchMethod = ismip.getEntryPoint();
						IPointListener bcm = new BlueCarrotMover(window);
						sm.addPositionReachedListener(bcm);

						InterpreterLastPointReachedCatcher ilprc = new InterpreterLastPointReachedCatcher(
								sm);
						sm.addPositionReachedListener(ilprc);

						if (window.getInterpreterStepMode()) {
							sm.setStepMode(true);
						} else {
							sm.setStepMode(false);
						}

						window.setStackMachine(sm);
						try {
							if (ismip.entryPointTakesArguments()) {
								String[] argsString = window
										.getInterpreterArgs();
								IType[] argsIType = new IType[argsString.length];
								for (int idx = 0; idx < argsString.length; idx++) {
									argsIType[idx] = StringTypeFactory
											.createStringType(argsString[idx]);
								}
								sm.push(ArrayTypeFactory
										.createArrayType(argsIType));
							}
							window.safeSetStatus("Debugger starting.");
							sm.invokeMethod(launchClass, launchMethod);
							window.positionBlueCarrot(0);
							window.safeSetStatus("Debugger completed.");
						} catch (Throwable t) {
							sm.shutdown();
							IAtomicClosure lastClosure = ilprc.getLastClosure();
							ISourceCodeLocation loc = (ISourceCodeLocation) ilprc
									.getLastPointData();
							if (loc == null || lastClosure == null) {
								outStreamErr.println(t);
							} else {
								outStreamErr.println(t + " at " + lastClosure
										+ ", line " + loc.getLineNumber()
										+ ", column " + loc.getColumnNumber());
								window.positionBlueCarrot(loc);
							}
						} finally {
							outStream.close();
							outStreamErr.close();
							inWriter.close();

							sm.removePositionReachedListener(bcm);

							sm.shutdown();
							window.setStackMachine(null);

							window.getTopTabs().getDisplay().syncExec(
									disableStuff);
						}
					} catch (RuntimeException e) {
						System.err.println("SM encountered an error:\n");
						e.printStackTrace(System.err);
					}
				}
			};

            executor.execute(stdOutReader);
            executor.execute(stdErrReader);
			executor.execute(launch);
		} catch (IOException e3) {
			e3.printStackTrace();
		}
	}

}

