/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on Aug 10, 2004
 *
 */
package uk.ac.imperial.doc.kenya.gui.editor.utils.interpreter;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.concurrent.ExecutorService;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.StyledTextContent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.TabFolder;

import uk.ac.imperial.doc.kenya.errors.KenyaInternalError;
import uk.ac.imperial.doc.kenya.gui.editor.EditingWindow;
import uk.ac.imperial.doc.kenya.gui.editor.utils.FileLoaderSaver;
import uk.ac.imperial.doc.kenya.mediator.Mediator;
import uk.ac.imperial.doc.kenya.mediator.stackMachine.StackMachineFactory;
import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.stackMachine.ClosureMiscHelper;
import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.scope.ClosureScope;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IClosureScope;
import uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode;

/**
 * @author Matthew Sackman
 * @version 1
 */
public class TabSelectionAdapter extends SelectionAdapter {

    private static final String noTranslation = "Kenya code contains errors, "
            + "can not translate to Java.";

    private static final String noInterpretation = "Kenya code contains errors, "
            + "can not debug.";

    private final EditingWindow window;

    private String oldJavaText = null;

    private String oldInterpreterText = null;

    private String lineDelimiter = null;

    private StringBuffer javaBuffer = new StringBuffer();
    
    private final ExecutorService executor;

    public TabSelectionAdapter(EditingWindow w) {
        window = w;
        executor = w.getExecutorService();
    }

    public void widgetSelected(SelectionEvent e) {
        TabFolder topTabs = window.getTopTabs();
        int index = topTabs.getSelectionIndex();

        if (index == EditingWindow.KENYATABINDEX) {
            if (window.getStackMachine() != null) {
                window.getInterpreterTerminateButton().notifyListeners(
                        SWT.Selection, null);
            }
            if (window.getJavaCode() != null) {
                window.getJavaTerminateButton().notifyListeners(SWT.Selection,
                        null);
            }
            return;
        } else if (index == EditingWindow.JAVATABINDEX) {
            if (window.getStackMachine() != null) {
                window.getInterpreterTerminateButton().notifyListeners(
                        SWT.Selection, null);
            }
        } else if (index == EditingWindow.INTERPRETERTABINDEX) {
            if (window.getJavaCode() != null) {
                window.getJavaTerminateButton().notifyListeners(SWT.Selection,
                        null);
            }
        }

        StyledText kenyaText = window.getKenyaTextWidget();
        StyledText javaText = window.getJavaTextWidget();
        StyledText interpreterText = window.getInterpreterTextWidget();

        if (javaText == null || kenyaText == null) return;

        if (lineDelimiter == null) lineDelimiter = javaText.getLineDelimiter();

        String text = kenyaText.getText();
        if (oldInterpreterText == null || !text.equals(oldInterpreterText)
                || oldJavaText == null || !text.equals(oldJavaText)) {

            try {
                ICheckedCode checkedCode = Mediator
                        .check(new StringReader(text));
                window.setKenyaCheckedCode(checkedCode);

                if (index == EditingWindow.JAVATABINDEX) {
                    if (oldJavaText == null || !text.equals(oldJavaText)) {

                        if (window.isJavaRunning()) { return; }

                        try {
                            Button javaExecuteButton = window
                                    .getJavaExecuteButton();
                            javaExecuteButton.setText("E&xecute Java");
                            if (checkedCode.isErroredCode()) {
                                javaText.setText(noTranslation);
                                window.getJavaLineStyler().parseBlockComments(
                                        noTranslation);
                                window.rebuildJavaNumbers();
                                window.setJavaCode(null);
                                window.setJavaClassName(null);
                                javaExecuteButton.setEnabled(false);
                            } else {
                                window.setJavaCode(checkedCode.translate());
                                javaText.setRedraw(false);
                                StyledTextContent content = javaText
                                        .getContent();
                                content.setText("");
                                BufferedReader javaReader = null;
                                FileLoaderSaver fls = window.getKenyaFile();

                                boolean getPlaceHeld = false;
                                String name = null;
                                IJavaCode code = window.getJavaCode();

                                if (fls == null || fls.getName() == null
                                        || fls.getName().length() == 0) {
                                    if (code.isValidClassName(window
                                            .getJavaDefaultClassName())) {
                                        name = window.getJavaDefaultClassName();
                                    } else {
                                        getPlaceHeld = true;
                                    }
                                } else {
                                    String fileName = fls.getName();
                                    if (fileName.endsWith(".k"))
                                            fileName = fileName.substring(0,
                                                    fileName.length() - 2);
                                    if (fileName.length() > 1) {
                                        char prefix = fileName.charAt(0);
                                        fileName = new String("" + prefix)
                                                .toUpperCase()
                                                + fileName.substring(1);
                                    }
                                    if (code.isValidClassName(fileName)) {
                                        name = fileName;
                                    } else if (code.isValidClassName(window
                                            .getJavaDefaultClassName())) {
                                        name = window.getJavaDefaultClassName();
                                    } else {
                                        getPlaceHeld = true;
                                    }
                                }

                                if (getPlaceHeld) {
                                    window.setJavaClassName(null);
                                    javaReader = new BufferedReader(code
                                            .getPlaceHeldCode());
                                    javaExecuteButton.setEnabled(false);
                                } else {
//                                    name += ""
//                                            + Math.abs(SecureRandomNumber
//                                                    .nextInt());
                                    window.setJavaClassName(name);
                                    javaReader = new BufferedReader(code
                                            .getCode(name));
                                    javaExecuteButton.setEnabled(true);
                                }

                                javaBuffer.setLength(0);
                                String line = javaReader.readLine();
                                while (line != null) {
                                    javaBuffer.append(line);
                                    javaBuffer.append(lineDelimiter);
                                    line = javaReader.readLine();
                                }
                                javaReader.close();
                                content.setText(javaBuffer.toString());
                                window.getJavaLineStyler().parseBlockComments(
                                        javaBuffer.toString());
                                window.rebuildJavaNumbers();
                                javaText.setRedraw(true);
                            }
                        } catch (IOException e1) {
                            e1.printStackTrace();
                        }
                        oldJavaText = text;
                    }
                } else if (index == EditingWindow.INTERPRETERTABINDEX) {

                    if (window.getStackMachine() != null) { return; }

                    if (oldInterpreterText == null
                            || !text.equals(oldInterpreterText)) {
                        window.getInterpreterContinueButton().setEnabled(false);
                        if (checkedCode.isErroredCode()) {
                            interpreterText.setText(noInterpretation);
                            window.rebuildInterpreterNumbers();
                            window.getInterpreterInterpretButton().setEnabled(
                                    false);
                            window.getInterpreterStepModeButton().setEnabled(
                                    false);
                        } else {
                            interpreterText.setRedraw(false);
                            interpreterText.setText(kenyaText.getText());
                            window.rebuildInterpreterNumbers();
                            final IStackMachineInformationProvider ismip = checkedCode
                                    .getBaseStackMachine();
                            window.setBaseStackMachine(ismip);
                            window.getCarrotPainter().clear();
                            interpreterText.setRedraw(true);
                            window.getInterpreterInterpretButton().setEnabled(
                                    true);
                            window.getInterpreterStepModeButton().setEnabled(
                                    true);

                            IStackMachine sm = window.getStackMachine();
                            if (sm != null) {
                                sm.shutdown();
                                window.setStackMachine(null);
                            }
                            sm = (new StackMachineFactory()).createStackMachine(executor);
                        	window.getInterpreterStdOutText().setText("");
                            try {
                                IClosureScope scope = new ClosureScope();
                                ClosureMiscHelper.executeClosureInScope(scope,
                                        ismip.getPreInitClosure(), sm);
                            } catch (Throwable t) {
                                window.getInterpreterStdOutText()
                                        .append("" + t);
                                window.getInterpreterInterpretButton()
                                        .setEnabled(false);
                                window.getInterpreterStepModeButton()
                                        .setEnabled(false);
                            }
                        }
                        oldInterpreterText = text;
                    }
                }
            } catch (IOException e1) {
                e1.printStackTrace();
            } catch (KenyaInternalError e2) {
                System.err.println("SM compilation failed: ");
                e2.printStackTrace(System.err);
            }
        }
    }

}

