/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.mediator;

import java.io.IOException;
import java.io.PrintStream;
import java.io.Reader;
import java.io.StringReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import uk.ac.imperial.doc.kenya.builtIns.BuiltInMethodsLoader;
import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.io.VariablePushbackReader;
import uk.ac.imperial.doc.kenya.minijava.lexer.Lexer;
import uk.ac.imperial.doc.kenya.minijava.lexer.LexerException;
import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.minijava.node.Start;
import uk.ac.imperial.doc.kenya.minijava.parser.Parser;
import uk.ac.imperial.doc.kenya.minijava.parser.ParserException;
import uk.ac.imperial.doc.kenya.passes.DefinitionFinder;
import uk.ac.imperial.doc.kenya.passes.RestrictedStructureChecker;
import uk.ac.imperial.doc.kenya.passes.StructureChecker;
import uk.ac.imperial.doc.kenya.passes.TypeChecker;
import uk.ac.imperial.doc.kenya.passes.TypeFinder;
import uk.ac.imperial.doc.kenya.passes.util.InformationHolder;
import uk.ac.imperial.doc.kenya.passes.util.ValidMainChecker;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode;
import uk.ac.imperial.doc.kenya.styleCheckers.util.Util;
import uk.ac.imperial.doc.kenya.types.IBuiltInMethod;
import uk.ac.imperial.doc.kenya.types.KClassType;
import uk.ac.imperial.doc.kenya.types.KFunction;
import uk.ac.imperial.doc.kenya.types.KType;
import uk.ac.imperial.doc.kenya.types.tables.ClassTable;
import uk.ac.imperial.doc.kenya.types.tables.FunctionTable;
import uk.ac.imperial.doc.kenya.types.tables.SymbolTable;
import uk.ac.imperial.doc.kenya.values.tables.VariableTable;


/**
 * 
 * @author toa02
 *
 */
public class Mediator {

    public static final int DBG_MESSAGES			=   					   1;
    public static final int DBG_PARSE				= DBG_MESSAGES 			<< 1;
    public static final int DBG_TYPEFIND 			= DBG_PARSE 			<< 1;
    public static final int DBG_DEFINITIONFIND 	= DBG_TYPEFIND 			<< 1;
    public static final int DBG_BUILTINLOAD		= DBG_DEFINITIONFIND	<< 1;
    public static final int DBG_TYPECHECK			= DBG_BUILTINLOAD	 	<< 1;
    public static final int DBG_STRUCTURECHECK	= DBG_TYPECHECK			<< 1;
    
    
    /**
     * Gives back an ICheckedCode of the Kenya from the given Input Stream.
     * @param in InputStream from which the Kenya SourceCode will be drawn.
     * @return ICheckedCode instance for the Kenya.
     * @throws IOException if something goes wrong with the stream.
     */
    public static ICheckedCode check(Reader in) throws IOException{
        return check(in, 0 ,null);
        //return check(in,1, System.out );
    }
    
    /**
     * This is one horribly long method,
     * but there isn't really any other way to do it.
     * This gives back an ICheckedCode representing the Kenya from the input stream,
     * with errors as appropriate.
     * @param in InputStream with the Kenya source code on it.
     * @param dbg Debugging flag bits.
     * @param debug A printwriter to print debug information to.
     * @return ICheckedCode representing the Kenya on the InputStream.
     * @throws IOException
     */
    public static synchronized ICheckedCode check(Reader in, int dbg, PrintStream debug) throws IOException{

        /* ------------------------ */
        /* Beginning initialisation */
        /* ------------------------ */
        
        String kenyaSourceCode = Util.drainReader(in);
        
        Start ast;
        List<ISourceCodeError> errors = new LinkedList<ISourceCodeError>();
        
        
        /* ------- */
        /* Parsing */
        /* ------- */
        if( isOpt(dbg,DBG_MESSAGES ) ){
            debug.println(">>>Parsing");
        }
        
        Parser parser;
        try{
        	parser = new Parser(
        	        new Lexer( new VariablePushbackReader(new StringReader(kenyaSourceCode)))
        	);
        	ast = parser.parse();
        	
        }catch(LexerException e){
            errors.add( SourceCodeException.wrapLexerException(e));
            return new InvalidCode(errors, new InformationHolder(new ClassTable(), new FunctionTable(), new VariableTable(), 
                    null, null, new HashMap<Node,KFunction>(), new HashMap<Node,KType>(), null), kenyaSourceCode);
        }catch(ParserException e){
            errors.add( SourceCodeException.wrapParserException(e));
            return new InvalidCode(errors, new InformationHolder(new ClassTable(), new FunctionTable(), new VariableTable(),
                    null, null, new HashMap<Node,KFunction>() ,new HashMap<Node,KType>(), null), kenyaSourceCode);
        }
        	
        	
    	if( isOpt(dbg,DBG_MESSAGES) ){
    	    debug.println("<<<Parsing");
    	}
    	
    	if( isOpt(dbg, DBG_PARSE)){
    	    return new PartiallyCheckedCode(errors,new InformationHolder(new ClassTable(), new FunctionTable(), new VariableTable(),
    	            null, null, new HashMap<Node,KFunction>(),new HashMap<Node,KType>(), ast), kenyaSourceCode);
    	}

    	/* ----------------------- */
    	/* Add the builtin classes */
    	/* ----------------------- */
    	ClassTable classTable = new ClassTable();
    	classTable.put("Boolean", KClassType.getBoolean(), null);
    	classTable.put("Character", KClassType.getCharacter(), null);
    	classTable.put("Integer", KClassType.getInteger(), null);
    	classTable.put("Double", KClassType.getDouble(), null);
    	
    	
    	/* ------------ */
    	/* Type Finding */
    	/* ------------ */
    	if( isOpt(dbg, DBG_MESSAGES)){
    	    debug.println(">>>TypeFinder");
    	}
    	
    	TypeFinder tfa = new TypeFinder(classTable);
    	ast.apply(tfa);
    	
    	errors.addAll(tfa.getErrors());
    	
    	if( isOpt(dbg, DBG_MESSAGES) ){
    	    debug.println("<<<TypeFinder");
    	}
    	
    	if( isOpt(dbg, DBG_TYPEFIND)){
    	    return new PartiallyCheckedCode(errors,new InformationHolder(new ClassTable(), 
    	            new FunctionTable(), new VariableTable(),
    	            null, null, new HashMap<Node, KFunction>(),new HashMap<Node, KType>(), ast), kenyaSourceCode);
    	}
    	
    	/* ------------------ */
    	/* Definition Finding */
    	/* ------------------ */
    	if( isOpt( dbg, DBG_MESSAGES) ){
    	    debug.println(">>>DefinitionFinder");
    	}

    	
    	Map<Node, KType> typeMap = new HashMap<Node, KType>();
    	
    	
    	Set<Node> errorNodes = tfa.getErrorNodes();
    	
    	DefinitionFinder df = new DefinitionFinder(tfa.getClassTable(),
    	        errorNodes, typeMap);
    	ast.apply(df);
        
  	    errors.addAll( df.getErrors() );
    	
    	if( isOpt( dbg, DBG_MESSAGES) ){
    	    debug.println("<<<DefinitionFinder");
    	}
    	
    	if( isOpt( dbg, DBG_DEFINITIONFIND )){
    	    return new PartiallyCheckedCode(errors,new InformationHolder(classTable,df.getFunctions(),new VariableTable(),
    	            null, null, new HashMap<Node, KFunction>(),new HashMap<Node, KType>(), ast), kenyaSourceCode);
    	}
    	
    	/* ------------------------ */
    	/* Merge in Builtin Methods */
    	/* ------------------------ */
    	if( isOpt(dbg, DBG_MESSAGES )){
    	    debug.println(">>>Builtins");
    	}
    	
    	FunctionTable functionTable;
    	{
    	    Set<IBuiltInMethod> builtins = BuiltInMethodsLoader.getBuiltInMethods();

            functionTable = df.getFunctions();
            for (IBuiltInMethod bim: builtins) {
                KFunction kf = new KFunction(bim.getName(), bim
                        .getTemplateParameters(), bim.getReturnType(), bim
                        .getMethodParameters(), null, bim, null);
                try{
                    functionTable.add(kf,true);
                }catch(SourceCodeException sce){
                    errors.add(sce);
                }
            }
    	}
    	
    	if( isOpt(dbg, DBG_MESSAGES )){
    	    debug.println("<<<Builtins");
    	}
    	
    	if( isOpt(dbg, DBG_BUILTINLOAD)){
    	    return new PartiallyCheckedCode(errors, new InformationHolder(df.getClassTable(),functionTable,new VariableTable(),
    	            null, null, new HashMap<Node, KFunction>(),new HashMap<Node, KType>(), ast), kenyaSourceCode);
    	}
    	
    	/* ------------ */
    	/* Type Checker */
    	/* ------------ */
    	
    	
    	if( isOpt(dbg,DBG_MESSAGES)){
    	    debug.println(">>>TypeChecker");
    	}

    	List<ISourceCodeWarning> warnings = new LinkedList<ISourceCodeWarning>();
    	
 //   	ClassTable classTable = df.getClassTable(); // Defined earlier
    	SymbolTable globalsSymbolTable = df.getGlobalSymbolTable();
    	
    	TypeChecker typechecker = new TypeChecker(classTable,globalsSymbolTable,functionTable, errorNodes, typeMap);
    	
    	ast.apply(typechecker);
    	
    	if( typechecker.getErrors().size() != 0){
    	    errors.addAll(typechecker.getErrors());
    	}
    	
    	warnings.addAll(typechecker.getWarnings());
    	
    	if( isOpt(dbg, DBG_MESSAGES)){
    	    debug.println("<<<TypeChecker");
    	}
    	
    	if( isOpt(dbg, DBG_TYPECHECK)){
    	    return new PartiallyCheckedCode(errors, new InformationHolder(classTable,functionTable,new VariableTable(),
    	            null, null, new HashMap<Node, KFunction>(),new HashMap<Node, KType>(), ast), kenyaSourceCode);
    	}
	    
    	
    	/* ---------------------- */
    	/* Variable Value Checker */
    	/* ---------------------- */
    	
    	if( isOpt(dbg,DBG_MESSAGES)){
    	    debug.println(">>>StructureChecker");
    	}
    	
    	
    	
    	Map<Node, KFunction> funcMap = typechecker.getFuncMap();
    	Set<Node> tcErrorNodes = typechecker.getErrorNodes();
    	Set<Node> arrayLengths = typechecker.getArrayLengthRefs();
    	Set<Node> enumChildSet = typechecker.getEnumChildSet();
    	
    	VariableTable variableTable = RestrictedStructureChecker.processConstants(ast, typeMap, tcErrorNodes, arrayLengths, enumChildSet, warnings, errors);
    
 	    StructureChecker sc = new StructureChecker(classTable,globalsSymbolTable,functionTable,
 	          variableTable, tcErrorNodes, typeMap, funcMap, arrayLengths, enumChildSet );
    	
 	    ast.apply(sc);
 	    errors.addAll(sc.getErrors());
 	    warnings.addAll(sc.getInfos());
 	    
    	
    	if(isOpt(dbg,DBG_MESSAGES)){
    	    debug.println("<<<StructureChecker");
    	}
    
 	    
    	if( isOpt(dbg,DBG_STRUCTURECHECK)){
    	    return new PartiallyCheckedCode(errors,warnings, 
    	            new InformationHolder(classTable,functionTable,variableTable,enumChildSet,arrayLengths, funcMap,typeMap, ast), kenyaSourceCode);
    	}
    	
    	/* ------------ */
    	/* Final Checks */
    	/* ------------ */
    	
    	// Numero uno: Do we have a main function:
    	InformationHolder ih = 
    	    new InformationHolder(classTable,functionTable,variableTable,enumChildSet,arrayLengths, funcMap,typeMap, ast);
    	
    	ValidMainChecker vmc = new ValidMainChecker(ih);
    	if( vmc.haveMainWithStringArgs() ){
    	    /* its all good */
    	}else if( vmc.haveMainNoArgs() ){
    	    /* its all good */
    	}else{
    	    errors.add(SourceCodeException.noValidMainFunction());
    	}
    	
    	// Numero dos: Do we have any class/enum's with a reserved class name?
    	Set<String> badClassNames = new HashSet<String>();
    	badClassNames.addAll(   Arrays.asList(new String[]{
    	       "Boolean", "Character", "Integer", "Double", "String", "Object" }));
    	
    	Set<String> funcClasses = new HashSet<String>();	// set of strings
    	{
	    for (KFunction kf: funcMap.values()) {
	            if( kf.isBuiltin() ){
	                funcClasses.addAll(kf.getBuiltinMethod().getReservedClasses());
	            }
	        }
    	}
    	
    	for(int i = 0 ; i < ih.getClasses().length ; i++){
    	    String name = ih.getClasses()[i].getName();
    	    if( badClassNames.contains(name)){
    	        errors.add(SourceCodeException.badClassName(ih.getClasses()[i].getPosition(),name));
    	    }else if( funcClasses.contains(name)){
    	        errors.add(SourceCodeException.badClassBuiltinName(ih.getClasses()[i].getPosition(),name));
    	    }
    	}
    	
    	for(int i = 0 ; i < ih.getEnums().length ; i++){
    	    String name = ih.getEnums()[i].getName();
    	    if( badClassNames.contains(name)){
    	        errors.add(SourceCodeException.badEnumName(ih.getEnums()[i].getPosition(),name));
    	    }else if( funcClasses.contains(name)){
    	        errors.add(SourceCodeException.badEnumBuiltinName(ih.getClasses()[i].getPosition(),name));
    	    }
    	}
    	
    	
    	if( errors.size() == 0 ){
            ICheckedCode code = new ValidCode(warnings,ih,vmc,funcMap,typeMap, kenyaSourceCode);

            return code;
            
    	}else{
    	    return new InvalidCode(errors,warnings,ih, kenyaSourceCode);
    	}
    	

    }
 
    


    /**
     * Utility method to see whether a specific option flag is set on an flags int.
     * @param dbg The int containing the options
     * @param opt The int representing the options that should be set
     * @return true iff every high bit in opt is high in dbg.
     */
    private static boolean isOpt(int dbg, int opt){
        return( (dbg & opt) == opt);
    }
    
}