/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.mediator;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.passes.util.InformationHolder;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable;
import uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode;
import uk.ac.imperial.doc.kenya.styleCheckers.IStyleCheckResult;


/**
 * This class represents code that has been partially checked.
 * @author toa02
 *
 */
class PartiallyCheckedCode implements ICheckedCode {

    private final List<ISourceCodeError> _errors;
    private final List<ISourceCodeWarning> _infos;
    private final InformationHolder _ih;
    private final String _kenyaSourceCode;
    
    PartiallyCheckedCode( List<ISourceCodeError> errors, InformationHolder ih, String kenyaSourceCode ){
        if( errors == null ){
            throw new NullPointerException("errors list cannot be null!");
        }
        
        if( ih == null ){ throw new NullPointerException("Information Holder can't be null"); }
        
        _errors = Collections.unmodifiableList(errors);
        _infos = Collections.unmodifiableList(new LinkedList<ISourceCodeWarning>());
        
        _ih = ih;
        
        this._kenyaSourceCode = kenyaSourceCode;
    }
    
    PartiallyCheckedCode( List<ISourceCodeError> errors, List<ISourceCodeWarning> infos, InformationHolder ih, String kenyaSourceCode ){
        if( errors == null ){throw new NullPointerException("errors can't be null"); }
        if( infos == null ){throw new NullPointerException("infos can't be null"); }
        if( ih == null ){ throw new NullPointerException("Information Holder can't be null"); }
        _errors = Collections.unmodifiableList(errors);
        _infos = Collections.unmodifiableList(infos);
        _ih = ih;
        _kenyaSourceCode = kenyaSourceCode;
    }
    
    /**
     * Although strictly speaking this class may represent valid code,
     * we don't know that. So we err on the side of caution and assume its invalid.
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#isErroredCode()
     */
    public boolean isErroredCode() {
        return true;
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getErrors()
     */
    public List<ISourceCodeError> getErrors() {
        return _errors;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getInfos()
     */
    public List<ISourceCodeWarning> getInfos() {
        return _infos;
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#translate()
     */
    public IJavaCode translate() {
        throw new IllegalStateException("Cannot translate a partially checked Kenya Program.");
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getBaseStackMachine()
     */
    public IStackMachineInformationProvider getBaseStackMachine() {
        throw new IllegalStateException("Build a stack machine to interpret a partially checked Kenya Program.");
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getClasses()
     */
    public IClass[] getClasses() {
        return _ih.getClasses();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getConstants()
     */
    public IVariable[] getConstants() {
        return _ih.getConstants();
    }
    
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getEnums()
     */
    public IClass[] getEnums() {
        return _ih.getEnums();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getFunctions()
     */
    public IFunction[] getFunctions() {
        return _ih.getFunctions();
    }

    public String getKenyaSourceCode() {
        return _kenyaSourceCode;
    }

    @SuppressWarnings("unchecked")
    public List<IStyleCheckResult> getStyleCheckResults() {
        return Collections.EMPTY_LIST;
    }
    
    
}
