/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.mediator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import uk.ac.imperial.doc.kenya.errors.KenyaInternalError;
import uk.ac.imperial.doc.kenya.mediator.stackMachine.StackMachineInformationManager;
import uk.ac.imperial.doc.kenya.mediator.util.JavaCode;
import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.passes.Translator;
import uk.ac.imperial.doc.kenya.passes.util.InformationHolder;
import uk.ac.imperial.doc.kenya.passes.util.ValidMainChecker;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable;
import uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode;
import uk.ac.imperial.doc.kenya.styleCheckers.IStyleCheckResult;
import uk.ac.imperial.doc.kenya.styleCheckers.IStyleChecker;
import uk.ac.imperial.doc.kenya.styleCheckers.StyleCheckerFactory;
import uk.ac.imperial.doc.kenya.types.KFunction;
import uk.ac.imperial.doc.kenya.types.KType;


/**
 * Represents a Kenya Program with Valid Code.
 * @author toa02
 *
 */
class ValidCode implements ICheckedCode {

    private final List<ISourceCodeWarning> _infos;
    private final InformationHolder _ih;
    private final ValidMainChecker _vmc;
    private final Map<Node,KFunction> _functionMap; // Maps nodes to functions used: needed by 
    private final Map<Node,KType> _typeMap;		// Maps nodes to the type of the node
    private final String _kenyaSourceCode;
    private final List<IStyleCheckResult> _styleCheckResults;
    
    public ValidCode( List<ISourceCodeWarning> infos, InformationHolder ih, ValidMainChecker vmc, Map<Node,KFunction> functionMap, Map<Node,KType> typeMap, String kenyaSourceCode){
        if( infos == null ){throw new NullPointerException("Infos can't be null");}
        if( ih == null ){ throw new NullPointerException("Information Holder can't be null");}
        _infos = Collections.unmodifiableList(infos);
        _ih = ih;
        _vmc = vmc;
        _functionMap = functionMap;
        _typeMap = typeMap;
        _kenyaSourceCode = kenyaSourceCode;
        _styleCheckResults = runStyleCheckers();
    }
    
    private List<IStyleCheckResult> runStyleCheckers() {
        List<IStyleCheckResult> res = new ArrayList<IStyleCheckResult>();
        
        Set<IStyleChecker> checkers = StyleCheckerFactory.getInstance().getStyleCheckers();
        for(IStyleChecker checker : checkers) {
            Set<IStyleCheckResult> iscr = checker.apply(this);
            res.addAll(iscr);
        }
        
        return Collections.unmodifiableList(res);
    }
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#isErroredCode()
     */
    public boolean isErroredCode() {
        return false;
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getInfos()
     */
    public List<ISourceCodeWarning> getInfos() {
        return _infos;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getErrors()
     */
    public List<ISourceCodeError> getErrors() {
        throw new IllegalStateException("Valid Kenya has no errors.");
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#translate()
     */
    public IJavaCode translate() {
        /** OK to translate what do I need to do:
         * 1: establish used builtin functions & get their imports
         * 2: translate functions
         * 3: translate classes
         * 4: translate enums
         */
        
        String imports;
        String constants;
        String functions;
        String classesAndEnums;
        Set<String> typeNames;
        Set<String> allBuiltinTypes;

        boolean require15 = false;
        
        {
            SortedSet<String> allImports = new TreeSet<String>();		// Sorted set of Strings
	        for (Node n: _functionMap.keySet()) {
	            KFunction kf = _functionMap.get(n);
	            if( kf.isBuiltin() ){
	                allImports.addAll(kf.getBuiltinMethod().getImports());
	            } else if( kf.getTypeParamMap().entrySet().size() != 0 ){
	                require15 = true;
	            }
	        }
	        
	        StringBuffer sb = new StringBuffer();
	        
	        for (String sImport: allImports) {
	            sb.append("import ");
	            sb.append(sImport);
	            sb.append(";");
	            sb.append(IJavaCode.NEWLINE);
	        }
	        
	        imports = sb.toString();
        }
        
        {
            allBuiltinTypes = new HashSet<String>();		// Sorted set of Strings
	        for (KFunction kf: _functionMap.values()) {
	            if( kf.isBuiltin() ){
	                allBuiltinTypes.addAll(kf.getBuiltinMethod().getReservedClasses());
	            }
	        }
        }
        

        Node node = (_vmc.getMainWithStringArgs() != null) ? _vmc.getMainWithStringArgs().getDeclarationNode()
                											: null;
        Translator translator = new Translator( _functionMap, node, _typeMap, true );

        /* Constants */
        translator.reset(Translator.CONSTANTS);
        _ih.getRootNode().apply(translator);
        constants = translator.getLastCode();
        
        /* Functions */
        translator.reset(Translator.FUNCTIONS);
        _ih.getRootNode().apply(translator);
        functions = translator.getLastCode();
        
        if( _vmc.haveMainWithStringArgs() ){
            // Then everything is fine
        }else if( _vmc.haveMainNoArgs() ){
            functions =
            IJavaCode.NEWLINE + 
            Translator.INDENT + "public static void main(String[] args) {" + IJavaCode.NEWLINE +
            Translator.INDENT + Translator.INDENT + "main();" + IJavaCode.NEWLINE +
            Translator.INDENT + "}" + IJavaCode.NEWLINE + functions;
        }else{
            throw new KenyaInternalError("Failed Precondition");
        }
        
        
        translator.reset(Translator.CLASSES | Translator.ENUMS );
        _ih.getRootNode().apply(translator);
        classesAndEnums = translator.getLastCode();
        
        
        typeNames = new HashSet<String>();
        typeNames.addAll(allBuiltinTypes);
        
        for(int i = 0 ; i < _ih.getClasses().length ; i++ ){
            typeNames.add(_ih.getClasses()[i].getName());
            require15 |= _ih.getClasses()[i].getTemplateParams().length != 0;
        }
        
        for(int i = 0 ; i < _ih.getEnums().length ; i++ ){
            typeNames.add(_ih.getEnums()[i].getName());
            require15 = true;
        }
        
        /*
         * only do this check if we absolutely have to 
         */
        if(!require15){
            require15 = _ih.areClassBasicTypesUsed();
        }
        
        return new JavaCode( imports, constants, functions, classesAndEnums, typeNames, require15 );
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getBaseStackMachine()
     */
    public IStackMachineInformationProvider getBaseStackMachine() {
        IStackMachineInformationManager smi = new StackMachineInformationManager(_ih);
        return smi.build(_vmc);
    }

    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getClasses()
     */
    public IClass[] getClasses() {
        return _ih.getClasses();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getConstants()
     */
    public IVariable[] getConstants() {
        return _ih.getConstants();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getEnums()
     */
    public IClass[] getEnums() {
        return _ih.getEnums();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.styleCheckers.ICheckedCode#getFunctions()
     */
    public IFunction[] getFunctions() {
        return _ih.getFunctions();
    }
    public String getKenyaSourceCode() {
        return _kenyaSourceCode;
    }
    public List<IStyleCheckResult> getStyleCheckResults() {
        return _styleCheckResults;
    }
}
