/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 10-Aug-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.mediator.stackMachine;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.passes.StackMachineBuilder;
import uk.ac.imperial.doc.kenya.passes.util.InformationHolder;
import uk.ac.imperial.doc.kenya.passes.util.ValidMainChecker;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable;
import uk.ac.imperial.doc.kenya.stackMachine.InterpretedBuildableClosure;
import uk.ac.imperial.doc.kenya.stackMachine.InterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.InterpretedEnumeration;
import uk.ac.imperial.doc.kenya.stackMachine.InterpretedMethod;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IBuildableClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedEnumeration;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedMethod;

/**
 * @author toa02
 *
 */
public class StackMachineInformationManager implements IStackMachineInformationManager{

    private final NameGenerator _classGen;
    private final NameGenerator _enumGen;
    private final NameGenerator _funcGen;
    private final NameGenerator _varGen;
    
    final BidirectionalMap<IClass, String> _classMap;
    final BidirectionalMap<IClass, String> _enumMap;	//map of IClass/IFunction <--> stack mach name
    final BidirectionalMap<IFunction, String> _funcMap;
    final BidirectionalMap<IVariable, String> _varMap;
    
    final BidirectionalMap<IClass, IClass> _normClassMap;
    final BidirectionalMap<IClass, IClass> _normEnumMap;
    final BidirectionalMap<IFunction, IFunction> _normFuncMap;
    final BidirectionalMap<IVariable, IVariable> _normVarMap;
    
    final Map<String, IInterpretedClass> _classBodies;
    final Map<String, IInterpretedMethod> _funcBodies;
    final Map<String, IInterpretedEnumeration> _enumBodies;
    
    final Map<AFuncDecDeclaration, IFunction> _funcNodeMap;
    
    final InformationHolder _ih;

    final IInterpretedClass _globalClass;
    
    final IBuildableClosure _preInitClosure;
    
    final List<ISourceCodeLocation> _codePoints;
    
    public StackMachineInformationManager(InformationHolder ih){
        _classGen = new NameGenerator("_cls");
        _enumGen = new NameGenerator("_enu");
        _funcGen= new NameGenerator("_fun");
        _varGen = new NameGenerator("_var");
        
        _classMap = new BidirectionalMap<IClass, String>();
        _enumMap = new BidirectionalMap<IClass, String>();
        _funcMap = new BidirectionalMap<IFunction, String>();
        _varMap = new BidirectionalMap<IVariable, String>();
        
        _normClassMap = new BidirectionalMap<IClass, IClass>();
        _normEnumMap = new BidirectionalMap<IClass, IClass>();
        _normFuncMap = new BidirectionalMap<IFunction, IFunction>();
        _normVarMap = new BidirectionalMap<IVariable, IVariable>();
        
        _classBodies = new HashMap<String, IInterpretedClass>();
        _funcBodies = new HashMap<String, IInterpretedMethod>();
        _enumBodies = new HashMap<String, IInterpretedEnumeration>();
        
        _funcNodeMap = new HashMap<AFuncDecDeclaration, IFunction>();
        
        _globalClass = new InterpretedClass(GLOBALCLASS);
        
        _preInitClosure = new InterpretedBuildableClosure();
        
        _ih = ih;
        
        _codePoints = new ArrayList<ISourceCodeLocation>();
        
        addBasicDetails(ih);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#addBasicDetails(uk.ac.imperial.doc.kenya.passes.util.InformationHolder)
     */
    private synchronized void addBasicDetails(InformationHolder ih) {
        
        /* Do classes: */
        {
            for (IClass c: ih.getClasses()) {
                String name = _classGen.getNewName();
                _classMap.associate(c, name);
                _normClassMap.associate(c, c.normalise());
                _classBodies.put(name, new InterpretedClass(name));
                
                for (IVariable v: c.getDeclarations()) {
                    String smName = _varGen.getNewName();
                    _varMap.associate(v, smName);
                    _normVarMap.associate(v, v.normalise());
                }
            }
        }
        
        /* Do enumerations: */
        {
            for (IClass e: ih.getEnums()) {
                String name = _enumGen.getNewName();
                _enumMap.associate(e,name);
                _normEnumMap.associate(e, e.normalise());
                String[] kids = e.getEnumConstants();
                _enumBodies.put(name, new InterpretedEnumeration(name, kids ));
            }
        }
        
        /* Do functions */
        {
            for (IFunction f: ih.getFunctions()) {
                String name = _funcGen.getNewName();
                _funcMap.associate( f, name );
                _normFuncMap.associate(f, f.normalise());
                _funcNodeMap.put(f.getDeclarationNode(), f );
                boolean hasRet = f.getReturnType() != ih.getVoid();
                IInterpretedMethod method = new InterpretedMethod(name,_globalClass,true,hasRet); 
                _funcBodies.put(name, method);
                _globalClass.addStaticMethod(method);
                // function param variables are created in caseAFuncDecDeclaration in StackMachineBuilder
            }
        }
        
        /* Placeholder the constants */
        {
            for (IVariable var: ih.getConstants()) {
                String name = _varGen.getNewName();
                _varMap.associate( var, name );
                _normVarMap.associate(var, var.normalise());
            }
        }
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#addNewVariable(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable)
     */
    public synchronized String addNewVariable(IVariable var) {
        String name = _varGen.getNewName();
        _varMap.associate(var,name);
        _normVarMap.associate(var,var.normalise());
        return name;
    }    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#build()
     */
    public synchronized IStackMachineInformationProvider build(ValidMainChecker vmc) {
        StackMachineBuilder smb = new StackMachineBuilder(this,_ih, true);
        _ih.getRootNode().apply(smb);
        return new StackMachineInformationProvider(this,vmc);
    }
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#getClassBody(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass)
     */
    public synchronized IInterpretedClass getClassBody(IClass clazz) {
        // From the class get the name, from the name get the body
        String name = _classMap.lookupRHS(clazz);
        return _classBodies.get(name);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#getEnumBody(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass)
     */
    public IInterpretedEnumeration getEnumBody(IClass enumm) {
        String name = _enumMap.lookupRHS( enumm );
        return _enumBodies.get(name);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#getMethodBody(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction)
     */
    public synchronized IInterpretedMethod getMethodBody(IFunction func) {
        String name = _funcMap.lookupRHS(func);
        return _funcBodies.get(name);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#lookupClass(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass)
     */
    public synchronized String lookupClass(IClass clazz) {
        return _classMap.lookupRHS(clazz);
    }
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#lookupEnum(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass)
     */
    public synchronized String lookupEnum(IClass zenum) {
        return _enumMap.lookupRHS(zenum);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#lookupFunction(uk.ac.imperial.doc.kenya.types.IFunction)
     */
    public synchronized String lookupFunction(IFunction func) {
        return _funcMap.lookupRHS(func);
    }
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#lookupVariable(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable)
     */
    public synchronized String lookupVariable(IVariable var) {
        return _varMap.lookupRHS(var);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#getFunctionFromNode(uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration)
     */
    public IFunction getFunctionFromNode(AFuncDecDeclaration node) {
        return _funcNodeMap.get(node);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#getGlobalClassBody()
     */
    public IInterpretedClass getGlobalClassBody() {
        return _globalClass;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#getPreInitClosure()
     */
    public IBuildableClosure getPreInitClosure() {
        return _preInitClosure;
    }
    
	/**
	 * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationManager#addCodePoint(uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeLocation)
	 */
	public void addCodePoint(ISourceCodeLocation iscl) {
	    _codePoints.add(iscl);
	}
    
    
}
