/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 16-Aug-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.mediator.stackMachine;

import java.util.List;

import uk.ac.imperial.doc.kenya.errors.KenyaPreconditionError;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.passes.util.ValidMainChecker;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable;
import uk.ac.imperial.doc.kenya.stackMachine.InterpretedBuildableClosure;
import uk.ac.imperial.doc.kenya.stackMachine.InterpretedMethod;
import uk.ac.imperial.doc.kenya.stackMachine.ops.StackOpsFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.ClassTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IBuildableClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedMethod;

/**
 * @author toa02
 *
 */
class StackMachineInformationProvider implements
        IStackMachineInformationProvider {
   
    private static final String INIT_METHOD_NAME = "<INIT_METHOD>";
    
    private final StackMachineInformationManager _ismim;
    private final ValidMainChecker _vmc;
    private final List<ISourceCodeLocation> _codePoints;
    private final IAtomicClosure _preInitClosure;
    
    
    public StackMachineInformationProvider(StackMachineInformationManager ismim, ValidMainChecker vmc){
        _ismim = ismim;
        _vmc = vmc;
        _codePoints = ismim._codePoints;

        IInterpretedMethod iim = new InterpretedMethod(INIT_METHOD_NAME,_ismim.getGlobalClassBody(),true,false);
        _ismim.getGlobalClassBody().addStaticMethod(iim);
        
        iim.setMethodBody(_ismim._preInitClosure);
        IBuildableClosure ibc = new InterpretedBuildableClosure();
        ibc.addClosure( StackOpsFactory.push( ClassTypeFactory.createClassStaticType(getEntryPointClass())));
        ibc.addClosure( StackOpsFactory.invokeForeignStaticMethod(INIT_METHOD_NAME));
    
        _preInitClosure = ibc;
        
    
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#entryPointTakesArguments()
     */
    public boolean entryPointTakesArguments() {
        return _vmc.haveMainWithStringArgs();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#getEntryPoint()
     */
    public String getEntryPoint() {
        IFunction mainFunc;
        if( _vmc.haveMainWithStringArgs() ){
            mainFunc = _vmc.getMainWithStringArgs();
        }else if( _vmc.haveMainNoArgs() ){
            mainFunc = _vmc.getMainWithNoArgs();
        }else{
            throw KenyaPreconditionError.get();
        }
        return _ismim.lookupFunction(mainFunc);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#getEntryPointClass()
     */
    public IInterpretedClass getEntryPointClass() {
        return _ismim.getGlobalClassBody();
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#lookupClass(java.lang.String)
     */
    public IClass lookupClass(String name) {
        IClass cls = _ismim._classMap.lookupLHS(name);
        
        if( cls == null ){
            return null;
        } else {
            return _ismim._normClassMap.lookupRHS(cls);
        }
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#lookupEnum(java.lang.String)
     */
    public IClass lookupEnum(String name) {
        IClass enu = _ismim._enumMap.lookupLHS(name);
        
        if( enu == null ){
            return null;
        } else {
            return _ismim._normEnumMap.lookupRHS(enu);
        }
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#lookupFunction(java.lang.String)
     */
    public IFunction lookupFunction(String name) {
        IFunction iff = _ismim._funcMap.lookupLHS(name);
        if( iff == null ){
            return null;
        } else {
            return _ismim._normFuncMap.lookupRHS(iff);
        }
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#lookupVariable(java.lang.String)
     */
    public IVariable lookupVariable(String name) {
        IVariable ivar = _ismim._varMap.lookupLHS(name);
        if( ivar == null ){
            return null;
        } else {
            return _ismim._normVarMap.lookupRHS(ivar);
        }
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#getPreInitClosure()
     */
    public IAtomicClosure getPreInitClosure() {
        return _preInitClosure;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider#getDiscreteExecutionPointMarkersFromTheAST()
     */
    public List<ISourceCodeLocation> getDiscreteExecutionPointMarkersFromTheAST() {
        return _codePoints;
    }

}
