/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Aug-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.mediator.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringReader;
import java.util.Set;
import java.util.concurrent.ExecutorService;

import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.Util;

/**
 * @author toa02
 *  
 */
public class JavaCode implements IJavaCode {

    public static final boolean DEBUG_OUTPUT = new Boolean(System.getProperty(
            "KENYA_DEBUG", "false")).booleanValue();
	
    private final String _imports;

    private final String _constants;

    private final String _functions;

    private final String _classesAndEnums;

    private final Set<String> _typeNames; 

    private final Object lock = new Object();

    private Process _lastProcess;

    private boolean recompile = true;

    private String oldClassName = null;

    private String oldFilePath = null;
    
    private final boolean require15;
    
    public JavaCode(String imports, String constants, String functions,
            String classesAndEnums, Set<String> typeNames, boolean require15) {
        _imports = imports;
        _constants = constants;
        _functions = functions;
        _classesAndEnums = classesAndEnums;
        _typeNames = typeNames;
        this.require15 = require15;
    }

    /**
     * @see uk.ac.imperial.doc.kenya.passes.IJavaCode#getCode(java.lang.String)
     */
    public Reader getCode(String className) {
        if (!isValidClassName(className)) { throw new IllegalArgumentException(
                "Bad ClassName: " + className); }
        return getUncheckedCode(className);
    }

    /**
     * @see uk.ac.imperial.doc.kenya.passes.IJavaCode#getPlaceHeldCode()
     */
    public Reader getPlaceHeldCode() {
        return getUncheckedCode(Util.GLOBAL);
    }

    private Reader getUncheckedCode(String className) {
        String outCode = _imports + ((_imports.equals("")) ? "" : NEWLINE)
                + "public class " + className + "{" + NEWLINE + _constants
                + ((_constants.equals("")) ? "" : NEWLINE) + _functions
                + NEWLINE + "}" + NEWLINE + _classesAndEnums;
        return new StringReader(outCode);
    }

    /**
     * @see uk.ac.imperial.doc.kenya.passes.IJavaCode#require15()
     */
    public boolean require15() {
        return require15;
    }

    /**
     * @see uk.ac.imperial.doc.kenya.passes.IJavaCode#killLastProcess()
     */
    public void killLastProcess() {
        Process lastProcess = getLastProcess();
        if (lastProcess != null) {
            lastProcess.destroy();
            boolean finished = false;
            while (!finished) {
                try {
                    lastProcess.waitFor();
                    finished = true;
                } catch (InterruptedException e) {
                    /* */
                }
            }
            try {
                lastProcess.getInputStream().close();
                lastProcess.getErrorStream().close();
                lastProcess.getOutputStream().close();
            } catch (IOException e) {
                System.err.println("IOException when closing lastProcess.");
            }
            if (getLastProcess() == lastProcess) setLastProcess(null);
        }
    }

    /**
     * @see uk.ac.imperial.doc.kenya.passes.IJavaCode#execute(java.lang.String, java.lang.String,
     *      java.io.BufferedReader, java.io.PrintStream, java.io.PrintStream,
     *      java.lang.String[])
     */
    public synchronized boolean execute(String filePath, String className,
            BufferedReader nStdIn, BufferedWriter nStdInWriter,
            BufferedWriter nStdOut, BufferedWriter nStdErr, String[] args, ExecutorService executor) {

        if (oldClassName == null || !oldClassName.equals(className)
                || oldFilePath == null || !oldFilePath.equals(filePath)) {
            setRecompile(true);
            oldClassName = className;
            oldFilePath = filePath;
        }

        if (getLastProcess() != null) { throw new IllegalStateException(
                "A process is already running"); }

        //      PrintWriter stdOutPW = new PrintWriter(nStdOut);
        PrintWriter stdErrPW = new PrintWriter(nStdErr);
        String classpath = System.getProperty("java.class.path", ".");

        File classFile = new File(filePath, className + ".java");
        if (!(classFile.exists() && classFile.canRead() && classFile.isFile())) {
            setRecompile(true);
        }

        if (isRecompile()) {
            if (!writeOutFile(filePath, className, stdErrPW)) {
                if (DEBUG_OUTPUT) {
                    System.err.println("Write out failed.");
                }
                setRecompile(true);
                return false;
            }

            if (!compile(classpath, filePath, className, nStdOut, nStdErr,
                    stdErrPW, executor)) {
                if (DEBUG_OUTPUT) {
                    System.err.println("compile failed.");
                }
                setRecompile(true);
                return false;
            }
        }

        if (!run(filePath, classpath, className, args, nStdOut, nStdErr,
                nStdIn, nStdInWriter, stdErrPW, executor)) {
            if (DEBUG_OUTPUT) {
                System.err.println("execute failed.");
            }

            setRecompile(true);
            return false;
        }

        setRecompile(false);
        return true;
    }

    private boolean compile(String classpath, String filePath,
            String className, BufferedWriter nStdOut, BufferedWriter nStdErr,
            PrintWriter stdErrPW, ExecutorService executor) {
        /*
         * Javac it.
         */
        if (DEBUG_OUTPUT) {
            System.out.println(classpath);
        }
        String[] cmdArray;
        if (require15()) {
            cmdArray = new String[] { "javac", "-source", "1.5", "-classpath",
                    classpath,
                    filePath + File.separatorChar + className + ".java" };
if(DEBUG_OUTPUT){
    System.out.println("java 1.5");
}
        } else {
            cmdArray = new String[] {
                    "javac",
                    "-source",
                    "1.4",
                    "-classpath",
                    classpath,
                    filePath + File.separatorChar + className + ".java" };
if(DEBUG_OUTPUT){
    System.out.println("java 1.4");
}
        }

        try {
            Process p = Runtime.getRuntime().exec(cmdArray);
            setLastProcess(p);

            InputStreamReader stdOutReader = new InputStreamReader(p
                    .getInputStream());
            InputStreamReader stdErrReader = new InputStreamReader(p
                    .getErrorStream());

            StreamHandler stdOutH = new StreamHandler(stdOutReader, nStdOut);
            StreamHandler stdErrH = new StreamHandler(stdErrReader, nStdErr);

            StreamHandlerWatcher shw = new StreamHandlerWatcher();
            shw.addStreamHandler(stdOutH);
            shw.addStreamHandler(stdErrH);
            executor.execute(shw);
            executor.execute(stdOutH);
            executor.execute(stdErrH);

            synchronized (shw) {
                while (!shw.isFinished()) {
                    shw.wait();
                }
            }

            int exitValue = p.waitFor();
            setLastProcess(null);

            if (DEBUG_OUTPUT) {
                System.out.println("Javac finished");
            }

            /**
             * Javac exited badly:
             */
            if (exitValue != 0) { return false; }

        } catch (IOException e) {
            stdErrPW.println(e.getMessage());
            return false;
        } catch (InterruptedException e) {
            stdErrPW.println(e.getMessage());
            return false;
        }

        return true;

    }

    private boolean run(String filePath, String classpath, String className,
            String[] args, BufferedWriter nStdOut, BufferedWriter nStdErr,
            BufferedReader nStdIn, BufferedWriter nStdInWriter,
            PrintWriter stdErrPW, ExecutorService executor) {
        /*
         * And now to java it.
         */
        {
            String[] cmdArrayPart = {
                    "java",
                    "-ea",
                    "-cp",
                    filePath + File.separatorChar + File.pathSeparatorChar
                            + classpath, className };
            String[] cmdArray = new String[cmdArrayPart.length + args.length];

            System.arraycopy(cmdArrayPart, 0, cmdArray, 0, cmdArrayPart.length);
            System.arraycopy(args, 0, cmdArray, cmdArrayPart.length,
                    args.length);

            try {
                Process p = Runtime.getRuntime().exec(cmdArray);
                setLastProcess(p);
                if (DEBUG_OUTPUT) {
                    System.out.println("Process started.");
                }

                InputStreamReader stdOutReader = new InputStreamReader(p
                        .getInputStream());
                InputStreamReader stdErrReader = new InputStreamReader(p
                        .getErrorStream());
                final OutputStreamWriter stdInWriter = new OutputStreamWriter(p
                        .getOutputStream());

                StreamHandler stdOutH = new StreamHandler(stdOutReader, nStdOut);
                StreamHandler stdErrH = new StreamHandler(stdErrReader, nStdErr);
                final StreamHandler stdInH = new StreamHandler(nStdIn,
                        stdInWriter);

                StreamHandlerWatcher shw = new StreamHandlerWatcher();
                shw.addStreamHandler(stdOutH);
                shw.addStreamHandler(stdErrH);
                shw.addStreamHandler(stdInH);
                executor.execute(shw);
                executor.execute(stdOutH);
                executor.execute(stdErrH);
                executor.execute(stdInH);

                executor.execute(new Runnable() {

                    /**
                     * @see uk.ac.imperial.doc.kenya.stackMachine.misc.Runnable#execute()
                     */
                    public void run() {
                        try {
                            synchronized (stdInH) {
                                while (!stdInH.isFinished()) {
                                    stdInH.wait();
                                }
                            }
                            if (DEBUG_OUTPUT) {
                                System.out.println("stream handler A finished");
                            }
                            stdInWriter.close();
                            if (DEBUG_OUTPUT) {
                                System.out
                                        .println("stream handler AB finished");
                            }
                        } catch (InterruptedException ie) {
                            ie.printStackTrace();
                        } catch (IOException ioe) {
                            ioe.printStackTrace();
                        }
                    }
                });

                int exitValue = p.waitFor();

                setLastProcess(null);
                nStdInWriter.flush();
                nStdInWriter.close();
                //nStdIn.flush();
                //nStdIn.close();

                synchronized (shw) {
                    while (!shw.isFinished()) {
                        shw.wait();
                    }
                }

                if (DEBUG_OUTPUT) {
                    System.err.println("Java Finished");
                }

                if (exitValue != 0) { return false; }

            } catch (IOException e) {
                stdErrPW.println(e.getMessage());
                return false;
            } catch (InterruptedException e) {
                return false;
            }

        }
        return true;
    }

    private boolean writeOutFile(String filePath, String className,
            PrintWriter stdErrPW) {
        /*
         * Write out the .java file
         */
        PrintWriter fw;
        BufferedReader br;

        try {
            fw = new PrintWriter(new FileWriter(new File(filePath, className
                    + ".java")));
            br = new BufferedReader(getCode(className));

            String ln;
            while ((ln = br.readLine()) != null) {
                fw.println(ln);
            }

        } catch (IOException e) {
            stdErrPW.println(e.getMessage());
            return false;
        }

        fw.close();
        try {
            br.close();
        } catch (IOException e) {
            stdErrPW.println(e.getMessage());
            return false;
        }

        return true;
    }

    private static final String KEYWORDS = ".Boolean.Double.Character.Integer.String.Object."
            + "assert.enum."
            + "abstract.default.if.private.this."
            + "boolean.do.implements.protected.throw."
            + "break.double.import.public.throws."
            + "byte.else.instanceof.return.transient."
            + "case.extends.int.short.try."
            + "catch.final.interface.static.void."
            + "char.finally.long.strictfp.volatile."
            + "class.float.native.super.while."
            + "const.for.new.switch."
            + "continue.goto.package.synchronized.";

    /**
     * Originally Sourced from
     * http://forum.java.sun.com/thread.jsp?thread=433717&forum=31&message=1943005
     * Updated a bit by me for thsi context
     */
    public boolean isValidClassName(String name) {
        if (name == null) { return false; }

        if (_typeNames.contains(name)) { return false; }

        char[] nameChars = name.toCharArray();
        int nameLength = nameChars.length;

        if (nameLength == 0) return false;

        if (!Character.isJavaIdentifierStart(nameChars[0])) return false;

        boolean identifierStarted = true;
        int identifierStart = 0;

        for (int i = 1; i < nameLength; i++) {
            char currentChar = nameChars[i];

            if (currentChar == '.') {
                return false;
            } else if (identifierStarted) {
                if (!Character.isJavaIdentifierPart(currentChar)) { return false; }
            } else {
                if (!Character.isJavaIdentifierStart(currentChar)) { return false; }
                identifierStart = i;
                identifierStarted = true;
            }
        }

        String maybeKeyword = '.' + name.substring(identifierStart) + '.';

        return (KEYWORDS.indexOf(maybeKeyword) == -1);
    }

    private Process getLastProcess() {
        synchronized (lock) {
            return _lastProcess;
        }
    }

    private void setLastProcess(Process process) {
        synchronized (lock) {
            if (process != null && _lastProcess != null) { throw new IllegalStateException(
                    "A process is already running"); }

            _lastProcess = process;
        }
    }

    private boolean isRecompile() {
        synchronized (lock) {
            return recompile;
        }
    }

    private void setRecompile(boolean recompile) {
        synchronized (lock) {
            this.recompile = recompile;
        }
    }
}