/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along wit	h this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 01-Jul-2004 by 
 * toa02
 *
 */
package uk.ac.imperial.doc.kenya.passes;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import uk.ac.imperial.doc.kenya.errors.ExceptionMapperWrapper;
import uk.ac.imperial.doc.kenya.errors.KenyaInternalError;
import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.minijava.analysis.DepthFirstAdapter;
import uk.ac.imperial.doc.kenya.minijava.node.AArrayDecInnerDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.ABasicTypeType;
import uk.ac.imperial.doc.kenya.minijava.node.AClassDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AClassInnerDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AClassTypeReferenceType;
import uk.ac.imperial.doc.kenya.minijava.node.AConstDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AEnumDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AInitialiser;
import uk.ac.imperial.doc.kenya.minijava.node.AReferenceTypeType;
import uk.ac.imperial.doc.kenya.minijava.node.AVarDecInnerDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AVoidBasicType;
import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.minijava.node.PClassInnerDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.PInnerDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.PType;
import uk.ac.imperial.doc.kenya.types.KBasicType;
import uk.ac.imperial.doc.kenya.types.KClassType;
import uk.ac.imperial.doc.kenya.types.KParamType;
import uk.ac.imperial.doc.kenya.types.KType;
import uk.ac.imperial.doc.kenya.types.KVariable;
import uk.ac.imperial.doc.kenya.types.tables.ClassTable;
import uk.ac.imperial.doc.kenya.types.tables.FunctionTable;
import uk.ac.imperial.doc.kenya.types.tables.SymbolTable;


/**
 * DefinitionFinder: Locates the definitions of:
 * a) Global constants
 * b) In-code Methods
 * c) Built-in methods.
 * d) Adds variables to the Class definitions.
 * It also type-checks to ensure that these things are all valid.
 * @author toa02
 *
 */
public class DefinitionFinder extends DepthFirstAdapter {

    private final ClassTable _ct;/* The Class Table */
    private final SymbolTable _st; /* The symbol table */
    private final FunctionTable _ft;	/* The function table */
	
	private final List<SourceCodeException> _errors;
	private final Set<Node> _errNodes;
	private final Map<Node, KType> _typeMap;
	
	public DefinitionFinder(ClassTable ct, Set<Node> errNodes, Map<Node, KType> typeMap){
		_ct = ct;
		_st = new SymbolTable();
		_ft = new FunctionTable();
		_errors = new LinkedList<SourceCodeException>();
		_errNodes = errNodes;
		_typeMap = typeMap;
	}
	
	public FunctionTable getFunctions(){
	    return _ft;
	}
	
	public SymbolTable getGlobalSymbolTable(){
	    return new SymbolTable(_st.collapse());
	}
	
	public ClassTable getClassTable(){
	    return _ct;
	}
	
	public Set<Node> getErrorNodes(){
	    return _errNodes;
	}
	
	public List<SourceCodeException> getErrors(){
	    return _errors;
	}
	

	/**
	 * Just generates a function for the type-information.
	 * Actual type checking is done in TypeChecker.
	 * @see uk.ac.imperial.doc.kenya.minijava.analysis.Analysis#caseAFuncDecDeclaration(uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration)
	 */
	public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
	  if( _errNodes.contains(node)){ return; }
	    
	    /*
		 * Must check that this function does not alreay exist with a different
		 * 	return type. (see below).
		 */
	    try{
		    _ft.add( Util.makeFunction(node, _ct),false); 
	    }catch( SourceCodeException e){
	        _errors.add( e);
	        _errNodes.add(node);
	    }
	    
	}

	
	/**
	 * A Constant Declaration.
	 * Check that the constant hasn't already been declared.
	 * Need to check that all variables in the rhs are already declared also.
	 * @see uk.ac.imperial.doc.kenya.minijava.analysis.Analysis#caseAConstDecDeclaration(uk.ac.imperial.doc.kenya.minijava.node.AConstDecDeclaration)
	 */
	public void caseAConstDecDeclaration(AConstDecDeclaration node){
	    if( _errNodes.contains(node)){ return; }
	    
	    try{
		    PType type = node.getType();
		    if( type instanceof ABasicTypeType){
		        KType kt = Util.getBasicType((ABasicTypeType) type);
		        if( kt.equals(KBasicType.getVoid())){
		            int ln = ((AVoidBasicType)((ABasicTypeType)type).getBasicType()).getVoid().getLine();
			        int pos = ((AVoidBasicType)((ABasicTypeType)type).getBasicType()).getVoid().getPos();
			        int len = ((AVoidBasicType)((ABasicTypeType)type).getBasicType()).getVoid().getText().trim().length();
			        SourceCodeException.throwVoidConstant(ln, pos, len);
		        }
		        
		        /* So we have a type that isn't void. Yipee! */
		        String ident =  node.getIdentifier().getText().trim();
		        KVariable kv = new KVariable(ident,kt,true,node);
		        
		        if(_st.containsVariable(ident)){
		            int ln = node.getIdentifier().getLine();
		            int pos = node.getIdentifier().getPos();
		            int len = node.getIdentifier().getText().trim().length();
		            KVariable kvClash = _st.lookupVariable(ident);
		            
		            int[][] lnkPos = { Util.getFirstIdent(kvClash.getNode()) }; 
		            SourceCodeException.throwDuplicateVariable(ln, pos, len, lnkPos, ident, Util.GLOBAL);
		        }
		        
		        List<SourceCodeException> errors = RestrictedTypeChecker.apply(_ct, _st, new HashMap<String,KParamType>(),
		                kt, _errNodes, (AInitialiser)node.getInitialiser(), _typeMap);
		        
		        if( errors.size() != 0 ){
		            for (SourceCodeException sce: errors) {
		                _errors.add( ExceptionMapperWrapper.wrapConstantException(sce));
		            }

		            _errNodes.add(node);
		            return;
		        }
		        
		        
		        _st.push(ident, kv );
		        _typeMap.put(node,kt);
		        
		    }else{
		        AReferenceTypeType artt = (AReferenceTypeType) type;
		        AClassTypeReferenceType actrt = (AClassTypeReferenceType) artt.getReferenceType();
		        int ln = actrt.getIdentifier().getLine();
		        int pos = actrt.getIdentifier().getPos();
		        int len = actrt.getIdentifier().getText().trim().length();
		        SourceCodeException.throwClassConstantError(ln, pos, len);
		    }
	    }catch(SourceCodeException sce){
            _errNodes.add(node);
            _errors.add( ExceptionMapperWrapper.wrapConstantException(sce));
	    }
	}

	public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
		/* Don't let this recurse down, there's no point. */
	}
	
	
	public void caseAClassDecDeclaration(AClassDecDeclaration node) {
	 if( _errNodes.contains(node)){ return; }
	    
	 try{
	    KClassType kct = (KClassType) _ct.getType(node.getIdentifier().getText().trim());

	    @SuppressWarnings("unchecked")
	    List<PClassInnerDeclaration> innerDecs = node.getClassInnerDeclaration();

	    SymbolTable classST = new SymbolTable();
	    
	    for (Node n: innerDecs) {

	        PInnerDeclaration pid = ((AClassInnerDeclaration)n).getInnerDeclaration();
	        KType type;
	        //KType baseType;
	        String ident;
	        //List typeConst;
	        int ln;
	        int pos;
	        int len;
		    try{	
		        
		        if( pid instanceof AVarDecInnerDeclaration ){
		            AVarDecInnerDeclaration avdid = (AVarDecInnerDeclaration) pid;
		            type = Util.getType(
		                    avdid.getType(),  
		                    _ct,
		                    kct.getTypeParams(),
		                    0,
		                    "class " + node.getIdentifier().getText().trim()
		                  );
		            ident = avdid.getIdentifier().getText().trim();
		            //typeConst = Util.getTypeParamList(avdid.getType(), _ct, kct.getTypeParams(), "class " + node.getIdentifier().getText().trim());
		            
		            ln = avdid.getIdentifier().getLine();
		            pos = avdid.getIdentifier().getPos();
		            len = ident.length();
		            //baseType = type;
		            
		            if( avdid.getInitialiser() != null ){
		                List<SourceCodeException> errors = RestrictedTypeChecker.apply(_ct, classST, kct.getTypeParams(), type, _errNodes, (AInitialiser) avdid.getInitialiser(), _typeMap);
		                if( errors.size() != 0 ){
		                    _errors.addAll(errors);
				            _errNodes.add(node);
		                    // return;
				            continue;
		                }
		            }
		            
		        }else if( pid instanceof AArrayDecInnerDeclaration){
		            AArrayDecInnerDeclaration aadid = (AArrayDecInnerDeclaration) pid;
		            int numBrackets = aadid.getBracketPair().size();
		            type = Util.getType(aadid.getType(),
		                    _ct,
		                    kct.getTypeParams(),
		                    numBrackets,
		                    "class " + node.getIdentifier().getText().trim());
		            ident = aadid.getIdentifier().getText().trim();
		            //typeConst = Util.getTypeParamList(aadid.getType(), _ct, kct.getTypeParams(), "class " + node.getIdentifier().getText().trim());
	
		            ln = aadid.getIdentifier().getLine();
		            pos = aadid.getIdentifier().getPos();
		            len = ident.length();
		            //baseType = ((KArrayType)type).getBaseType();
		            
		            List<SourceCodeException> errors = null;
		            
		            if( aadid.getArrayInitialiser() == null ){
		                //do nothing
		            }else {
		                errors = RestrictedTypeChecker.apply(_ct, classST, kct.getTypeParams(), type, _errNodes, aadid.getArrayInitialiser(), _typeMap);
		            }
		            
		            if( errors != null && errors.size() != 0 ){
		                _errors.addAll(errors);
			            _errNodes.add(node);
		                // return;
			            continue;
		            }
		            
		        }else{
		            /* This should be impossible */
		            throw new KenyaInternalError("Got an unknown node type!" + pid);
		        }
		        
		        if( type == KBasicType.getVoid() ){
		            int[] vpos = Util.getFirstIdent(pid);
		            _errNodes.add(node);
		            SourceCodeException.throwVariableDeclaration_Void(vpos[0], vpos[1], vpos[2]);
		        }
		        
		        
		        if( classST.containsVariable(ident)){
		            KVariable conf = classST.lookupVariable(ident);
		            int[][] lnkPos = { Util.getFirstIdent(conf.getNode()) };
		            _errNodes.add(node);
	                SourceCodeException.throwDuplicateVariable(ln, pos, len, lnkPos, ident, kct.toString());
		        }
	        
	        }catch(SourceCodeException sce){
                _errNodes.add(node);
                _errors.add(sce);
                continue;
            }

	            
	        /* So here, the rhs is legal. And we have:
	         * typeConst = the SomeClass< HERE, Pair<..>, BLAH> part
	         * baseType = the underlying type of the object, be it ArrayType or just normal.
	         * type = the actual type of the object.
	         */
	        KVariable kv;
            kv = new KVariable(ident, type, false, pid);
	        classST.push(ident,kv);	        
	    
	    }
	    
	    
	    kct.setChildren( new SymbolTable(classST.collapse()));
	   
	    
	  }catch( SourceCodeException sce){
	      throw new KenyaInternalError("Should not be possible to get here");
	  }
	    
	}
}
