/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 29-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.passes;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import uk.ac.imperial.doc.kenya.errors.ExceptionMapperWrapper;
import uk.ac.imperial.doc.kenya.errors.KenyaPreconditionError;
import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.minijava.analysis.DepthFirstAdapter;
import uk.ac.imperial.doc.kenya.minijava.node.ABasicTypeType;
import uk.ac.imperial.doc.kenya.minijava.node.AClassDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AConstDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AEnumDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.minijava.node.Start;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.types.KBasicType;
import uk.ac.imperial.doc.kenya.types.KFunction;
import uk.ac.imperial.doc.kenya.types.KType;
import uk.ac.imperial.doc.kenya.types.KVariable;
import uk.ac.imperial.doc.kenya.types.tables.ClassTable;
import uk.ac.imperial.doc.kenya.types.tables.FunctionTable;
import uk.ac.imperial.doc.kenya.types.tables.SymbolTable;
import uk.ac.imperial.doc.kenya.values.IKNumericValue;
import uk.ac.imperial.doc.kenya.values.IKValue;
import uk.ac.imperial.doc.kenya.values.tables.VariableTable;


/**
 * @author toa02
 *
 */
public final class RestrictedStructureChecker extends DepthFirstAdapter{
    
    private VariableTable _varTab;
    private SymbolTable _symTab;
    private Map<Node, KType> _typeMap;

    private List<ISourceCodeError> _errors;
    private List<ISourceCodeWarning> _infos;
    private Set<Node> _errNodes;
    private Set<Node> _arrayLengths;
    private Set<Node> _enumChildSet;
    
    private RestrictedStructureChecker(Map<Node, KType> typeMap, Set<Node> errNodes, Set<Node> arrayLengths, Set<Node> enumChildSet){
        _varTab = new VariableTable();
        _symTab = new SymbolTable();
        _typeMap = typeMap;
        _errNodes = errNodes;
        _symTab.newLocalScope();
        _infos = new LinkedList<ISourceCodeWarning>();
        _errors = new LinkedList<ISourceCodeError>();
        _arrayLengths = arrayLengths;
        _enumChildSet = enumChildSet;
    }
    
    public static final VariableTable processConstants(Start ast, Map<Node, KType> typeMap, Set<Node> errNodes, Set<Node> arrayLengths,
            Set<Node> enumChildSet, List<ISourceCodeWarning> infos, List<ISourceCodeError> errors){
        RestrictedStructureChecker rsc = new RestrictedStructureChecker(typeMap, errNodes, arrayLengths, enumChildSet);
        ast.apply(rsc);
        infos.addAll( rsc._infos);
        errors.addAll( rsc._errors );
        
        return rsc.getVariableTable();
    }

    public VariableTable getVariableTable(){
        return _varTab;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.minijava.analysis.DepthFirstAdapter#caseAConstDecDeclaration(uk.ac.imperial.doc.kenya.minijava.node.AConstDecDeclaration)
     */
    public void caseAConstDecDeclaration(AConstDecDeclaration node) {
        if( _errNodes.contains(node) ){ return; }
        
        inAConstDecDeclaration(node);
        StructureChecker sc = new StructureChecker(new ClassTable(), _symTab,
                new FunctionTable(),_varTab, new HashSet<Node>(), _typeMap, new HashMap<Node, KFunction>(),_arrayLengths, _enumChildSet );
        try{
            node.getInitialiser().apply(sc);
            IKValue ikv = sc.getTopOfValueStack();
            
            
            KBasicType type = Util.getBasicType( (ABasicTypeType) node.getType() );
            String ident = node.getIdentifier().getText().trim();
            
            if( type == KBasicType.getChar() ){
	            if( ikv instanceof IKNumericValue ){
	                IKNumericValue iknv = (IKNumericValue) ikv;
	                
	                if (iknv.inCharRange() ){
	                    // yes its fine.
	                }else{
	                    // no its not!
	                    int[] pos = Util.getFirstIdent(node.getInitialiser());
	                    SourceCodeException.throwPossibleLossOfPrecisionCharInt(pos[0],pos[1],pos[2]);
	                }
	            }else if( Util.isSingleVariableExpression(node.getInitialiser() )){
	                // then ok
	            }else{
	                int[] pos = Util.getFirstIdent(node.getInitialiser());
	                SourceCodeException.throwPossibleLossOfPrecisionCharInt(pos[0],pos[1],pos[2]);
	            }
            }
            
            KVariable kv = new KVariable(ident,type,true,node);
            
            if(!_symTab.pushLocal(ident, kv)){ throw KenyaPreconditionError.get(); }
            if(!_varTab.addConstant(kv, ikv)){ throw KenyaPreconditionError.get(); }
            
        }catch(SourceCodeException sce){
            _errors.add(ExceptionMapperWrapper.wrapConstantException(sce));
            _errNodes.add(node);
        }
        _infos.addAll(sc.getInfos());
        
        outAConstDecDeclaration(node);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.minijava.analysis.DepthFirstAdapter#caseAFuncDecDeclaration(uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration)
     */
    public void caseAFuncDecDeclaration(AFuncDecDeclaration node) {
        // Don't enter the function
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.minijava.analysis.DepthFirstAdapter#caseAClassDecDeclaration(uk.ac.imperial.doc.kenya.minijava.node.AClassDecDeclaration)
     */
    public void caseAClassDecDeclaration(AClassDecDeclaration node) {
        // Don't enter the class declaration
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.minijava.analysis.DepthFirstAdapter#caseAEnumDecDeclaration(uk.ac.imperial.doc.kenya.minijava.node.AEnumDecDeclaration)
     */
    public void caseAEnumDecDeclaration(AEnumDecDeclaration node) {
        // Don't enter the enum declaration
    }
}
