/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 14-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.passes;

import java.util.List;
import java.util.Map;
import java.util.Set;

import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.minijava.node.AInitialiser;
import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.minijava.node.PArrayInitialiser;
import uk.ac.imperial.doc.kenya.types.KBasicType;
import uk.ac.imperial.doc.kenya.types.KParamType;
import uk.ac.imperial.doc.kenya.types.KType;
import uk.ac.imperial.doc.kenya.types.KTypeSourceCodeException;
import uk.ac.imperial.doc.kenya.types.tables.ClassTable;
import uk.ac.imperial.doc.kenya.types.tables.SymbolTable;

/**
 * This is used to type-check constant and class member declarations.
 * You cannot initialise a class type inside a class declaration, and
 * constants must be of basic type.
 * However you can (just about) initialise array types of class type
 * to array types of class types. 
 * It is essentially a static wrapper round TypeChecker.
 * @author toa02
 *
 */
final class RestrictedTypeChecker {


    /**
     * The general "do the pass" method. Wrapped by 3 typesafe versions below,
     * so you can only start this on an initialiser node.
     * @param ct ClassTable to use
     * @param st SymbolTable to use
     * @param paramMap Map of Strings->KParamTypes that are in play
     * @param retType Return type expected.
     * @param start Node to start on.
     * @return List of the errors encountered, or an empty list if there where none.
     */
    private static List<SourceCodeException> apply(ClassTable ct, SymbolTable st, Map<String,KParamType> paramMap,
            KType retType, Node start, Set<Node> errNodes, Map<Node, KType> typeMap, int ln, int pos, int len){

        TypeChecker tc = new TypeChecker(ct,st,paramMap,errNodes,typeMap,retType);
        
        if( retType == KBasicType.getChar()){
            tc.setWantChar();
        }
        
        start.apply(tc);
        KType actRetType = tc.getTopOfTypeStack();
        List<SourceCodeException> errors = tc.getErrors();
        
        if(!KParamType.isAssignmentCompatible(retType, actRetType)){
            try{
                KTypeSourceCodeException.throwIncompatibleTypes_Assignment(ln, pos, len, retType, actRetType);
            }catch(SourceCodeException sce){
                errors.add(sce);
            }
        }
        

        
        return errors;
        
    }
    
    static List<SourceCodeException> apply(ClassTable ct, SymbolTable st, Map<String,KParamType> paramMap,
            KType retType, Set<Node> errorNodes, AInitialiser start, Map<Node, KType> typeMap){
        int ln = start.getAssign().getLine();
        int pos = start.getAssign().getPos();
        int len = start.getAssign().getText().trim().length();
        return apply(ct,st,paramMap,retType,start, errorNodes, typeMap, ln, pos, len);
    }
 
    static List<SourceCodeException> apply(ClassTable ct, SymbolTable st, Map<String,KParamType> paramMap,
            KType retType, Set<Node> errorNodes, PArrayInitialiser start, Map<Node, KType> typeMap){
    	int[] pos = Util.getFirstIdent(start);
    	return apply(ct,st,paramMap,retType,start, errorNodes, typeMap,  pos[0],pos[1],pos[2]);
    }
    
}
