/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 04-Aug-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.passes.util;

import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeLocation;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable;
import uk.ac.imperial.doc.kenya.types.KParamTypeMapper;

/**
 * @author toa02
 *
 */
public class AClassInstance implements IClass {

    public static final int A_CLASS = 0;
    public static final int AN_ENUM = 1;
    public static final int AN_ARRAY = 2;
    public static final int A_BASIC = 3;
    public static final int A_PARAM = 4;
    private static final int A_PARAM_NORMAL = 5;
    
    private String _name;
    private IVariable[] _decls;
    private String[] _consts;
    private ISourceCodeLocation _position;
    private IClass[] _templateParams;
    
    private final int _classType;

    
    
    
    
    public AClassInstance(String name, IClass[] templateParams, ISourceCodeLocation position, int type ){
        _name = name;
        _decls = new IVariable[0];
        _consts = new String[0];
        _position = position;
        _templateParams = templateParams;
        _classType = type;
        //_derivedFrom = null;
    }
    
    public AClassInstance(String name, IClass[] templateParams, IClass derivedFrom, int type ){
        this(name,templateParams, derivedFrom.getPosition(), type);
        setDeclarations(derivedFrom.getDeclarations());
        setEnumConsts(derivedFrom.getEnumConstants());
        //_derivedFrom = derivedFrom;
    }
    
    private AClassInstance(String name, IClass[] templateParams, ISourceCodeLocation position, IVariable[] vars, String[] consts, int type){
        _name = name;
        _templateParams = templateParams;
        _position = position;
        _decls = vars;
        _consts = consts;
        _classType = type;
    }
    
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#normalise()
     */
    public IClass normalise() {
        IVariable[] newVars = new IVariable[_decls.length];
        for(int i = 0 ; i < newVars.length ; i++ ){
            newVars[i] = _decls[i].normalise();
        }
        
        IClass[] templateParams = new IClass[_templateParams.length];
        for(int i = 0 ; i < templateParams.length ; i++ ){
            templateParams[i] = _templateParams[i].normalise();
        }
        
        
        
        
        if( _classType == A_PARAM){
            String name = KParamTypeMapper.get().lookupName(_name);
            return new AClassInstance(name,templateParams,_position,newVars,_consts,A_PARAM_NORMAL);
        } else if( _classType == AN_ARRAY){
            String arrayPart = _name.substring(_name.indexOf('['));
			String namePart = _name.substring(0,_name.indexOf('['));
			
			if( namePart.startsWith(KParamTypeMapper.PRE)){
			    namePart = KParamTypeMapper.get().lookupName(namePart);
			}
			
			return new AClassInstance(namePart + arrayPart,templateParams,_position,newVars,_consts,AN_ARRAY);
        } else {
            return new AClassInstance(_name,templateParams,_position,newVars,_consts,_classType);
        }
        
    }
    
    public void setDeclarations(IVariable[] decls){
        _decls = decls;
    }
    
    public void setEnumConsts(String[] consts){
        _consts = consts;
    }

    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#isArray()
     */
    public boolean isArray() {
        return _classType == AN_ARRAY;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#isClass()
     */
    public boolean isClass() {
        return _classType == A_CLASS;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#isEnum()
     */
    public boolean isEnum() {
        return _classType == AN_ENUM;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#isBasic()
     */
    public boolean isBasic() {
        return _classType == A_BASIC;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#isParam()
     */
    public boolean isParam() {
        return _classType == A_PARAM || (_classType == A_PARAM_NORMAL);
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#getDeclarations()
     */
    public IVariable[] getDeclarations() {
        return _decls;
    }
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#getEnumConstants()
     */
    public String[] getEnumConstants() {
        return _consts;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#getName()
     */
    public String getName() {
        return _name;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#getPosition()
     */
    public ISourceCodeLocation getPosition() {
        return _position;
    }
    
    /**
     * @see uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass#getTemplateParams()
     */
    public IClass[] getTemplateParams() {
        return _templateParams;
    }
    
    /**
     * Just to make my life easier while using the debugger:
     * @see java.lang.Object#toString()
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        
        if( isEnum() ){
            /* Its an enum */
            sb.append("enum ");
        }
        
        if( _classType == A_PARAM ){
            sb.append( KParamTypeMapper.get().lookupName(_name));
        } else {
            sb.append(_name);
        }
            
        
        if( _templateParams.length > 0 ){
            sb.append("<");
            
            for(int i = 0 ; i < _templateParams.length ; i++ ){
                sb.append( _templateParams[i].toString() );
                if( i < _templateParams.length -1){
                    sb.append(",");
                }
            }
            
            sb.append(">");
        }

        return sb.toString();
    
    }
    
    

    
    public boolean equals(Object other){
        if( other instanceof IClass ){
            return ((IClass)other).getName().equals(_name);
        }
        
        return false;
    }

    public int hashCode(){
        return _name.hashCode();
    }
}

