/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 06-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine;

import java.util.Map;
import java.util.TreeMap;

import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * This class represents a method. When you want to add a method to a class, you
 * create an instance of this class, fill in the details, provide a method body
 * (Closure) and then make sure you remember to add this class instance to the
 * InterpretedClass.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class InterpretedMethod implements IInterpretedMethod {

    private String name = null;

    private IInterpretedClass myClass = null;

    private IAtomicClosure methodBody = null;

    private boolean returnType = false;

    private boolean isStatic = false;

    private final Map<String,IType> mutableVariables = new TreeMap<String,IType>();

    private final Map<String,IType> immutableVariables = new TreeMap<String,IType>();

    private final Object lock = new Object();

    /**
     * Creates a new InterpretedMethod instance. Despite the fact that you have
     * to provide the class in which the method resides, you still need to add
     * this 'method' instance to the class itself.
     * 
     * @param name
     *            The name of the method.
     * @param myClass
     *            The class in which this method resides.
     * @param isStatic
     *            Whether this method is a static method.
     * @param hasReturnType
     *            Whether this method has a return type.
     */
    public InterpretedMethod(String name, IInterpretedClass myClass,
            boolean isStatic, boolean hasReturnType) {
        this.name = name;
        this.myClass = myClass;
        this.isStatic = isStatic;
        this.returnType = hasReturnType;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#setName(java.lang.String)
     */
    public synchronized void setName(String name) {
        if (this.name != null)
                throw new RuntimeException(
                        "Attempt made to redefine InterpretedClass's name: "
                                + this.name);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#getName()
     */
    public synchronized String getName() {
        return name;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#setReturnType(boolean)
     */
    public synchronized void setReturnType(boolean hasReturnType) {
        returnType = hasReturnType;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#hasReturnType()
     */
    public synchronized boolean hasReturnType() {
        return returnType;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#setMethodBody(org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethodBody)
     */
    public void setMethodBody(IAtomicClosure body) {
        synchronized (lock) {
            methodBody = body;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#getMethodBody()
     */
    public IAtomicClosure getMethodBody() {
        synchronized (lock) {
            return methodBody;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#invoke(org.wellquite.kenya.stackMachine.StackMachine)
     */
    public void invoke(IStackMachine sm) {
        Map<String,IType> mutableMethodVariablesCopy = null;
        synchronized (this) {
            IAtomicClosure methodBody = getMethodBody();
            if (methodBody == null)
                    throw new RuntimeException(
                            "Attempt made to invoke on a null method body in method "
                                    + this);
            mutableMethodVariablesCopy = new TreeMap<String,IType>(mutableVariables);
        }

        ClosureMiscHelper.executeClosureInScope(sm.getClosureScope(),
                methodBody, sm);

        synchronized (this) {
            mutableVariables.clear();
            mutableVariables.putAll(mutableMethodVariablesCopy);
            mutableMethodVariablesCopy.clear();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#addMutableMethodVariable(java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void addMutableMethodVariable(String name,
            IType variable) {
        mutableVariables.put(name, variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#addImmutableMethodVariable(java.lang.String,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public synchronized void addImmutableMethodVariable(String name,
            IType variable) {
        immutableVariables.put(name, variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#isStatic()
     */
    public synchronized boolean isStatic() {
        return isStatic;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#setStatic(boolean)
     */
    public synchronized void setStatic(boolean isStatic) {
        this.isStatic = isStatic;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#getInterpretedClass()
     */
    public IInterpretedClass getInterpretedClass() {
        return myClass;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    public synchronized String toString() {
        return "InterpretedMethod '" + name + "' in class " + myClass
                + " (static = " + isStatic + ")";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#getMutableMethodVariables()
     */
    public synchronized Map<String,IType> getMutableMethodVariables() {
        return mutableVariables;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IInterpretedMethod#getImmutableMethodVariables()
     */
    public synchronized Map<String,IType> getImmutableMethodVariables() {
        return immutableVariables;
    }
}