/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import uk.ac.imperial.doc.kenya.stackMachine.scope.ClosureScope;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IClosureScope;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IMethodScope;
import uk.ac.imperial.doc.kenya.stackMachine.scope.MethodScope;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClass;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedClassInstance;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IInterpretedMethod;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * The mighty stack machine itself. The methods in here are generally called as
 * needed by the Closures returned by the StackOpsFactory methods. You
 * shouldn't, when programming for the stack machine need to call anything in
 * here.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class StackMachine implements IStackMachine {

    private final ExecutorService executor;
    
	public StackMachine(ExecutorService executor) {
        this.executor = executor;
	}

	private final Object lock = new Object();

	private PrintStream out = System.out;

	private PrintStream err = System.err;

	private InputStream in = System.in;

	private final Stack<IType> theStack = new Stack<IType>();

	private final Stack<Integer> methodBoundaries = new Stack<Integer>();

	private final Stack<Integer> argPeekCounts = new Stack<Integer>();

	private int currentBoundary = 0;

	private int currentArgPeekCount = 0;

	private final List<IPointListener> positionReachedListeners = new ArrayList<IPointListener>();

	private boolean stackMachineHalt = false;

	private boolean stackMachineContinue = true;

	private boolean stepMode = false;

	private IMethodScope methodScope = null;

	private IClosureScope closureScope = null;

	/**
	 * Returns the current scope. This is the same as doing Scope.getScope() but
	 * you should use this method inside closures (avoid dependancy on Scope).
	 * 
	 * @return The current scope.
	 */
	public synchronized IMethodScope getMethodScope() {
		if (methodScope == null) {
			methodScope = new MethodScope();
		}
		return methodScope;
	}

	public synchronized void setMethodScope(IMethodScope scope) {
		methodScope = scope;
	}

	public synchronized IClosureScope getClosureScope() {
		if (closureScope == null) {
			closureScope = new ClosureScope();
		}
		return closureScope;
	}

	public synchronized void setClosureScope(IClosureScope scope) {
		closureScope = scope;
	}

	/**
	 * Push the supplied object onto the top of the stack.
	 * 
	 * @param object
	 */
	public synchronized void push(IType object) {
		theStack.push(object);
	}

	/**
	 * Pop and return the item on the top of the stack. If you are attempting to
	 * pop an arguement to your method, you will not pop it, only peek it. This
	 * is handled automatically.
	 * <p>
	 * Note that you can only peek at your arguements once and that each peek
	 * will return an item successively further down the stack. Therefore, if
	 * your method takes 3 arguements then the first thing your method should do
	 * is 3 storeNewVariable(name, pop()) calls - this will result in the three
	 * top most items on the stack being stored in your method's variable
	 * namespace.
	 * 
	 * @return The item on the top of the stack (or further down if peeking, as
	 *         described).
	 */
	public synchronized IType pop() {
		if (theStack.size() <= currentBoundary) {
			//System.out.println("Reading method args (peeking)");
			currentArgPeekCount++;
			return (IType) theStack.get(currentBoundary - currentArgPeekCount);
		}
		return (IType) theStack.pop();
	}

	/**
	 * Print out the string representation of the item at the top of the stack
	 * on the stackmachine's output stream.. This method does not affect the
	 * stack.
	 *  
	 */
	public synchronized void printStackPeek() {
		getOut().print(theStack.peek().toString());
	}

	/**
	 * Println out the string representation of the item at the top of the stack
	 * on the stackmachine's output stream.. This method does not affect the
	 * stack.
	 *  
	 */
	public synchronized void printlnStackPeek() {
		getOut().println(theStack.peek().toString());
	}

	/**
	 * Print the supplied value on the stackmachine's output stream. This method
	 * does not touch the stack at all.
	 * 
	 * @param value
	 */
	public synchronized void print(String value) {
		getOut().print(value);
	}

	/**
	 * Println the supplied value on the stackmachine's output stream. This
	 * method does not touch the stack at all.
	 * 
	 * @param value
	 */
	public synchronized void println(String value) {
		getOut().println(value);
	}

	/**
	 * Println a newline. This method does not touch the stack at all.
	 */
	public synchronized void println() {
		getOut().println();
	}

	/**
	 * Invoke the method in this class corresponding to the supplied method
	 * name. If the current method is a static method then only static methods
	 * can be called. If the current method is a class instance method (ie
	 * non-static) then both static and non-static methods can be called.
	 * 
	 * @param method
	 *            The name of the method to invoke.
	 */
	public void invokeMethod(String method) {
		if (getMethodScope() == null)
			unableToFindMethod(method);
		else {
			IMethodScope scope = getMethodScope();
			IInterpretedMethod currentMethod = scope.getCurrentMethod();
			if (currentMethod.isStatic()) {
				IInterpretedClass currentClass = currentMethod
						.getInterpretedClass();
				invokeMethod(currentClass, method);
			} else {
				IInterpretedClassInstance currentClassInstance = scope
						.getCurrentClassInstance();
				invokeMethod(currentClassInstance, method);
			}
		}
	}

	/**
	 * Invoke the method corresponding to the supplied method name in the
	 * supplied class. The method must be a static method.
	 * 
	 * @param targetClass
	 *            The class in which to search for the method.
	 * @param methodName
	 *            The name of the method to search for.
	 */
	public void invokeMethod(IInterpretedClass targetClass, String methodName) {
		IInterpretedMethod method = targetClass.getMethod(methodName);

		IMethodScope scope = new MethodScope(method, targetClass);
		synchronized (this) {
			getMethodScope().switchToNewScope(scope, this);

			methodBoundaries.push(currentBoundary);
			currentBoundary = theStack.size();

			argPeekCounts.push(currentArgPeekCount);
			currentArgPeekCount = 0;
		}
		method.invoke(this);
		synchronized (this) {

			IType returnResult = null;
			if (method.hasReturnType())
				returnResult = (IType) theStack.peek();

			while (theStack.size() > currentBoundary)
				theStack.pop();

			if (method.hasReturnType())
				theStack.push(returnResult);

			currentBoundary = methodBoundaries.pop();

			currentArgPeekCount = argPeekCounts.pop();

			scope.switchToPreviousScope(this);
		}
	}

	/**
	 * Invoke the method corresponding to the supplied method name in the
	 * supplied class instance. If the method is not found in the class
	 * instance's instance methods then then class instance's static methods are
	 * searched.
	 * 
	 * @param targetClass
	 *            The class instance in which to search for the method.
	 * @param methodName
	 *            The name of the method to search for.
	 */
	public void invokeMethod(IInterpretedClassInstance targetClass,
			String methodName) {
		IInterpretedMethod method = targetClass.getMethod(methodName);
		IMethodScope scope = new MethodScope(method, targetClass);
		synchronized (this) {
			getMethodScope().switchToNewScope(scope, this);

			methodBoundaries.push(currentBoundary);
			currentBoundary = theStack.size();

			argPeekCounts.push(currentArgPeekCount);
			currentArgPeekCount = 0;
		}
		method.invoke(this);
		synchronized (this) {
			IType returnResult = null;
			if (method.hasReturnType())
				returnResult = theStack.peek();

			while (theStack.size() > currentBoundary)
				theStack.pop();

			if (method.hasReturnType()) {
				theStack.push(returnResult);
			}

			currentBoundary = methodBoundaries.pop();

			currentArgPeekCount = argPeekCounts.pop();

			scope.switchToPreviousScope(this);
		}
	}

	private void unableToFindMethod(String method) {
		throw new RuntimeException("Unable to find method '" + method + "'");
	}

	public void addPositionReachedListener(IPointListener listener) {
		synchronized (lock) {
			positionReachedListeners.add(listener);
		}
	}

	public void removePositionReachedListener(IPointListener listener) {
		synchronized (lock) {
			positionReachedListeners.remove(listener);
		}
	}

	/**
	 * Called to indicate that a specific position has been reached. The
	 * positionReachedListeners are fired with the supplied parameter. If the
	 * stackMachine is in step mode then the stack machine is halted until
	 * instructed to continue.
	 * 
	 * @param data
	 */
	public void positionReached(final Object data) {
		Future<?> future = executor.submit(new Runnable() {

			public void run() {
				List<IPointListener> listeners = getPositionReachedListeners();
				for (int idx = 0; idx < listeners.size(); idx++) {
					((IPointListener) listeners.get(idx)).pointReached(data);
				}
			}
		});
        try {
            future.get();
        } catch (InterruptedException e1) {
            e1.printStackTrace();
        } catch (ExecutionException e1) {
            e1.printStackTrace();
        }

		if (getStepMode()) {
			synchronized (lock) {
				setStackMachineHalt(true);
				setStackMachineContinue(false);

				while (!getStackMachineContinue()) {
					try {
						lock.wait();
					} catch (InterruptedException e) {
					}
				}

				setStackMachineHalt(false);
			}
		}
	}

	private List<IPointListener> getPositionReachedListeners() {
		synchronized (lock) {
			return new ArrayList<IPointListener>(positionReachedListeners);
		}
	}

	private void setStackMachineHalt(boolean value) {
		synchronized (lock) {
			stackMachineHalt = value;
		}
	}

	private void setStackMachineContinue(boolean value) {
		synchronized (lock) {
			stackMachineContinue = value;
		}
	}

	private boolean getStackMachineContinue() {
		synchronized (lock) {
			return stackMachineContinue;
		}
	}

	private boolean getStackMachineHalt() {
		synchronized (lock) {
			return stackMachineHalt;
		}
	}

	/**
	 * If called when the stack machine is not halted then this method has no
	 * effect. Otherwise, this method turns step mode off and instructs the
	 * halted stack machine to continue execution.
	 *  
	 */
	public void resume() {
		resume(false);
	}

	private void resume(boolean step) {
		synchronized (lock) {
			if (getStackMachineHalt()) {
				stepMode = step;
				setStackMachineContinue(true);
				lock.notifyAll();
			}
		}
	}

	/**
	 * If called when the stack machine is not halted then this method has no
	 * effect. Otherwise, this method turns step mode on and instructs the
	 * halted stack machine to continue execution.
	 *  
	 */
	public void step() {
		resume(true);
	}

	/**
	 * Set the stackMachine's step mode to the supplied value. If the value is
	 * true then the stack machine will halt whenever a set position is reached.
	 * Otherwise the stack machine will run to the next break point or until
	 * completion if no break points are set.
	 * 
	 * @param value
	 */
	public void setStepMode(boolean value) {
		synchronized (lock) {
			stepMode = value;
		}
	}

	/**
	 * Indicates whether this stackMachine is in step mode or not.
	 * 
	 * @return true iff the stackMachine is in step mode. False otherwise.
	 */
	public boolean getStepMode() {
		synchronized (lock) {
			return stepMode;
		}
	}

	public InputStream getIn() {
		synchronized (lock) {
			return in;
		}
	}

	public void setIn(InputStream in) {
		synchronized (lock) {
			this.in = in;
		}
	}

	public PrintStream getErr() {
		synchronized (lock) {
			return err;
		}
	}

	public void setErr(PrintStream err) {
		synchronized (lock) {
			this.err = err;
		}
	}

	public PrintStream getOut() {
		synchronized (lock) {
			return out;
		}
	}

	public void setOut(PrintStream out) {
		synchronized (lock) {
			this.out = out;
		}
	}

	public void shutdown() {
		synchronized (lock) {
			setStepMode(false);
			resume();
			try {
				in.close();
				out.close();
			} catch (IOException e) {
				System.err.println("IO error when shutting down SM.");
				e.printStackTrace(System.err);
			}
		}
	}
}