/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on Jul 8, 2004
 *
 */
package uk.ac.imperial.doc.kenya.stackMachine.ops;

import java.util.ArrayList;
import java.util.List;

import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.types.AbstractAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.ArrayTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.PrimitiveTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IArrayType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.INumericType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * This class returns closures that, when executed, manipulate IArrayType
 * instances on the top of the stack. Unless otherwise stated, all methods
 * remove from the stack any arguements they require.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class ArrayOpsFactory {

    private ArrayOpsFactory() {
    }

    private static final IAtomicClosure get = new AbstractAtomicClosure("getArrayElement") {

        public void execute(IStackMachine sm) {
            int index = ((INumericType) sm.pop()).getValue().intValue();
            IArrayType array = (IArrayType) sm.pop();
            sm.push(array.get(index));
        }
    };

    private static final IAtomicClosure set = new AbstractAtomicClosure("setArrayElement") {

        public void execute(IStackMachine sm) {
            int index = ((INumericType) sm.pop()).getValue().intValue();
            IArrayType array = (IArrayType) sm.pop();
            IType value = sm.pop();
            array.set(index, value);
        }
    };

    private static final IAtomicClosure getLength = new AbstractAtomicClosure("getLengthArray") {

        public void execute(IStackMachine sm) {
            IArrayType array = (IArrayType) sm.pop();
            IType len = PrimitiveTypeFactory.createPrimitiveType(array
                    .getLength());
            sm.push(len);
        }
    };

    private static final IAtomicClosure buildArrayWithValues = new AbstractAtomicClosure("buildArray") {

        public void execute(IStackMachine sm) {
            int length = ((INumericType) sm.pop()).getValue().intValue();
            IType[] ary = new IType[length];
            for (int idx = length - 1; idx >= 0; idx--)
                ary[idx] = sm.pop();
            sm.push(ArrayTypeFactory.createArrayType(ary));
        }
    };

    private static final IAtomicClosure buildMultidimensionalArray = new AbstractAtomicClosure("buildMultidimensionalArray") {

        public void execute(IStackMachine sm) {
            IType value = sm.pop();
            int dimensions = ((INumericType) sm.pop()).getValue().intValue();
            if (dimensions < 1)
                    throw new IllegalArgumentException(
                            "Can not create an array with " + dimensions
                                    + " dimensions.");
            int[] sizes = new int[dimensions];
            for (int idx = dimensions - 1; idx >= 0; idx--)
                sizes[idx] = ((INumericType) sm.pop()).getValue().intValue();

            int count = 1;
            for (int idx = 0; idx < dimensions; idx++)
                count *= sizes[idx];

            List<IArrayType> holdingPen = new ArrayList<IArrayType>();
            for (int dimIdx = dimensions - 1; dimIdx >= 0; dimIdx--) {
                count /= sizes[dimIdx];
                for (int idx = 0; idx < count; idx++) {
                    IArrayType ary = ArrayTypeFactory
                            .createArrayType(new IType[sizes[dimIdx]]);
                    if (dimIdx == dimensions - 1) {
                        for (int childrenCount = 0; childrenCount < sizes[dimIdx]; childrenCount++) {
                            ary.set(childrenCount, value);
                        }
                    } else {
                        for (int childrenCount = 0; childrenCount < sizes[dimIdx]; childrenCount++) {
                            ary
                                    .set(childrenCount, (IType) holdingPen
                                            .remove(0));
                        }
                    }
                    holdingPen.add(ary);
                }
            }
            sm.push((IType) holdingPen.remove(0));
            if (!holdingPen.isEmpty())
                    throw new IllegalStateException(
                            "Error occured when creating "
                                    + dimensions
                                    + "-dimensional array. Holding Pen left with contents: "
                                    + holdingPen);
        }
    };

    /**
     * The returned closure, when executed, gets the indicated element out of
     * the array and places that element on the top of the stack. If arg0 is the
     * item on the top of the stack and arg1 is the next item then after
     * execution the top of the stack will (in effect) contain the item
     * arg1[arg0]. arg1 must be an IArrayType instance and arg0 must be an
     * IntType.
     * 
     * @return
     */
    public static IAtomicClosure get() {
        return get;
    }

    /**
     * The returned closure, when executed, sets the indicated element in the
     * indicated array to the indicated value. If arg0 is the item on the top of
     * the stack, arg1 is the next item and arg2 is the next item on the stack
     * then the closure effectively results in arg2[arg1] = arg0. arg0 must be
     * the same type as the other elements in the array (specified at array
     * creation). arg1 must be an IntType instance and arg2 must be an
     * IArrayType instance.
     * 
     * @return
     */
    public static IAtomicClosure set() {
        return set;
    }

    /**
     * The returned closure, when executed, takes the indicated array off the
     * top of the stack and puts in its place an Int Primitive type holding the
     * value of the length of the given array.
     * 
     * @return
     */
    public static IAtomicClosure getLength() {
        return getLength;
    }

    /**
     * The returned closure, when executed, builds an array. The argument on the
     * top of the stack is the length of the array to be built. The values of
     * the array are the next elements on the stack such that the nth element in
     * an n element array is at the top whilst element 0 is at the bottom. The
     * constructed array is left on the stack.
     * 
     * @return
     */
    public static IAtomicClosure buildArrayWithValues() {
        return buildArrayWithValues;
    }

    /**
     * The returned closure, when executed, builds a multidimensional array. The
     * item on the top of the stack is the value that all elements of the array
     * will have. The next item on the stack is the number of dimensions the
     * array has. For that number, x, the next x items on the stack describe the
     * size of each dimension. The greatest dimension is the item closest to the
     * top. For example, if you wanted to construct int [][][] x = new int
     * [5][6][2] then you would push 5, push 6, push 2 and then push 3 (there
     * are 3 dimensions here) and then an IntType with value zero.
     * 
     * @return
     */
    public static IAtomicClosure buildMultidimensionalArray() {
        return buildMultidimensionalArray;
    }
}
