/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 07-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine.ops;


import uk.ac.imperial.doc.kenya.stackMachine.ClosureMiscHelper;
import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.scope.IClosureScope;
import uk.ac.imperial.doc.kenya.stackMachine.types.AbstractHaltableClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IBooleanType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IHaltableClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * This class provides static methods which return closures which, when
 * executed, provide control flow primitive functionality. Unless otherwise
 * stated, all methods remove from the stack any arguements they require.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class ControlFlowOpsFactory {

    private ControlFlowOpsFactory() {
    }

    /**
     * The returned closure implementes an if construct when executed. None of
     * the closures should rely on the contents of the stack. They should fetch
     * any variables they need from memory and evaluate those. The
     * conditionClosure must always leave an IBooleanType instance on the top of
     * the stack. If this instance has a true value and the trueClosure is not
     * null then the trueClosure is executed. If the value is false and the
     * falseClosure is not null then the falseClosure is executed. The
     * conditionClosure can not be null.
     * 
     * @param conditionClosure
     *            The closure representing the if condition.
     * @param trueClosure
     *            The closure executed if the conditionClosure leaves a true
     *            value IBooleanType on the stack.
     * @param falseClosure
     *            The closure executed if the conditionClosure leaves a false
     *            value IBooleanType on the stack.
     * @return
     */
    public static IHaltableClosure ifClosure(
            final IAtomicClosure conditionClosure,
            final IAtomicClosure trueClosure, final IAtomicClosure falseClosure) {
        return new AbstractHaltableClosure() {

            public synchronized void execute(IStackMachine sm) {
                IClosureScope scope = getClosureScope();
                if (scope == null)
                    throw new IllegalStateException(
                            "Can't execute closure without a scope.");

                ClosureMiscHelper.executeClosureInScope(scope,
                        conditionClosure, sm);
                if (scope.getReturn()) {
                    return;
                }
                boolean result = ((IBooleanType) sm.pop()).getValue()
                        .booleanValue();
                if (scope.getReturn()) {
                    return;
                }
                if (result && trueClosure != null) {
                    if (scope.getReturn()) {
                        return;
                    }
                    ClosureMiscHelper.executeClosureInScope(scope, trueClosure,
                            sm);
                } else if (!result && falseClosure != null) {
                    if (scope.getReturn()) {
                        return;
                    }
                    ClosureMiscHelper.executeClosureInScope(scope,
                            falseClosure, sm);
                }
            }

            public String toString() {
                return "IfClosure:\n{\n\tcondition: <" + conditionClosure
                        + ">\n\ttrueClosure: <" + trueClosure
                        + ">\n\tfalseClosure: <" + falseClosure + ">\n}\n";
            }
        };
    }

    /**
     * The returned closure represents a while statement when executed. The
     * conditionClosure is executed. That must leave an IBooleanType on the top
     * of the stack. If the IBooleanType has a true value then the bodyClosure
     * is excuted, then the conditionClosure is executed again. Ad infinitum.
     * The bodyClosure can be null, the conditionClosure can not be null.
     * 
     * @param conditionClosure
     *            The closure representing the while condition.
     * @param bodyClosure
     *            The closure to be executed every time the while condition is
     *            evaluated and results in a true value.
     * @return
     */
    public static IHaltableClosure whileClosure(
            final IAtomicClosure conditionClosure,
            final IAtomicClosure bodyClosure) {
        return new AbstractHaltableClosure() {

            public synchronized void execute(IStackMachine sm) {
                IClosureScope scope = getClosureScope();
                if (scope == null)
                    throw new IllegalStateException(
                            "Can't execute closure without a scope.");
                ClosureMiscHelper.executeClosureInScope(scope,
                        conditionClosure, sm);
                if (scope.getReturn()) {
                    return;
                }
                boolean result = ((IBooleanType) sm.pop()).getValue()
                        .booleanValue();
                if (scope.getReturn()) {
                    return;
                }
                while (result) {
                    if (bodyClosure != null)
                        ClosureMiscHelper.executeClosureInScope(scope,
                                bodyClosure, sm);
                    if (scope.getReturn()) {
                        return;
                    }
                    ClosureMiscHelper.executeClosureInScope(scope,
                            conditionClosure, sm);
                    if (scope.getReturn()) {
                        return;
                    }
                    result = ((IBooleanType) sm.pop()).getValue()
                            .booleanValue();
                    if (scope.getReturn()) {
                        return;
                    }
                }
            }

            public String toString() {
                return "whileClosure:\n{\n\tcondition: <" + conditionClosure
                        + ">\n\tbody: <" + bodyClosure + ">\n}\n";
            }
        };
    }

    /**
     * The returned closure represents a switch statement when executed. The
     * array supplied is iterated through and the behaviour is as dictated by
     * the JLS (I think).
     * 
     * @param cond
     *            The closure representing the condition at the root of this
     *            switch. This should leave on the top of the stack the value of
     *            the switch condition ( this will be removed and used to test
     *            against each of the cases )
     * @param cases
     *            The closures representing the conditions for each part of this
     *            switch statment. The defaul case will not be evaluated. These
     *            closures *cannot* have side-effects [ as per switch
     *            definition, they must essentially be constants ]. They must
     *            leave on the top of the stack what should be equality-compared
     *            against all the other cases. Behaviour is undefined if there
     *            are multiple cases with the same constant value ( this is not
     *            detected )
     * @param statements
     *            IAtomicClosures represnting the statements for each of the
     *            cases.
     * @param defaultCase
     *            index of the cases / statements arrays that refer to the
     *            default case, or -1 if it is not to be used.
     * @return
     */
    public static IHaltableClosure switchClosure(final IAtomicClosure cond,
            final IAtomicClosure[] cases, final IAtomicClosure[] statements,
            final int defaultCase) {

        return new AbstractHaltableClosure() {

            public synchronized void execute(IStackMachine sm) {
                IClosureScope scope = getClosureScope();
                if (scope == null)
                    throw new IllegalStateException(
                            "Can't execute closure without a scope.");

                ClosureMiscHelper.executeClosureInScope(scope, cond, sm);
                if (scope.getReturn()) {
                    return;
                }
                IType result = sm.pop();
                if (scope.getReturn()) {
                    return;
                }
                int startCase = -1;

                for (int i = 0; i < cases.length; i++) {
                    if (i == defaultCase)
                        continue;

                    ClosureMiscHelper
                            .executeClosureInScope(scope, cases[i], sm);

                    if (scope.getReturn()) {
                        return;
                    }
                    sm.push(result);
                    if (scope.getReturn()) {
                        return;
                    }
                    ClosureMiscHelper.executeClosureInScope(scope,
                            LogicalOpsFactory.equal(), sm);
                    if (scope.getReturn()) {
                        return;
                    }

                    boolean res = ((IBooleanType) sm.pop()).getValue()
                            .booleanValue();
                    if (scope.getReturn()) {
                        return;
                    }

                    if (res) {
                        startCase = i;
                        break;
                    }
                }

                if (startCase == -1) { /* no matching case */
                    startCase = defaultCase;
                }

                for (int i = startCase; i != -1 && i < statements.length; i++) {
                    ClosureMiscHelper.executeClosureInScope(scope,
                            statements[i], sm);
                    if (scope.getReturn()) {
                        return;
                    }
                }
            }

            public String toString() {
                return "switch: \n{\n\tcases: <" + cases + ">\n\tstatements: <"
                        + statements + ">\n}\n";
            }
        };
    }
}