/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 07-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine.ops;


import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;
import uk.ac.imperial.doc.kenya.stackMachine.types.AbstractAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.DoubleType;
import uk.ac.imperial.doc.kenya.stackMachine.types.IntType;
import uk.ac.imperial.doc.kenya.stackMachine.types.PrimitiveTypeFactory;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IAtomicClosure;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.ICharType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.INumericType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * This class contains static methods which return Closures which do numericy
 * things. Unless otherwise stated, all methods remove from the stack any
 * arguments they require.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class NumericOpsFactory {

    private static final int INTRESULT = 0;

    private static final int DOUBLERESULT = 1;

    private static final int CHARRESULT = 2;

    private NumericOpsFactory() {
    }

    static private abstract class NumericOps extends AbstractAtomicClosure {        
        
        public NumericOps(String name) {
            super(name);            
        }
        
        public void execute(IStackMachine sm) {
            IType arg0 = sm.pop();
            IType arg1 = sm.pop();
            int resultType = INTRESULT;
            int resultI = 0;
            double resultD = 0;

            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    resultI += ((INumericType) arg0).getValue().intValue();
                    resultD += ((INumericType) arg0).getValue().intValue();
                } else if (arg0 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD += ((INumericType) arg0).getValue().doubleValue();
                }
            } else if (arg0 instanceof ICharType) {
                resultType = CHARRESULT;
                resultI += ((ICharType) arg0).getValue().charValue();
                resultD += ((ICharType) arg0).getValue().charValue();
            } else {
                throw new IllegalArgumentException(
                        "Can not perform addition on arg0 from stack: " + arg0);
            }

            if (arg1 instanceof INumericType) {
                if (arg1 instanceof IntType) {
                    resultI = operation(resultI, ((INumericType) arg1).getValue().intValue());
                    resultD = operation(resultD, ((INumericType) arg1).getValue().intValue());
                } else if (arg1 instanceof DoubleType) {
                    resultType = DOUBLERESULT;
                    resultD = operation(resultD, ((INumericType) arg1).getValue().doubleValue());
                }
            } else if (arg1 instanceof ICharType) {
                resultI = operation(resultI, ((ICharType) arg1).getValue().charValue());
                resultD = operation(resultD, ((ICharType) arg1).getValue().charValue());
            } else {
                throw new IllegalArgumentException(
                        "Can not perform " + this._name + " on arg1 from stack: " + arg1);
            }

            if (resultType == INTRESULT || (resultType == CHARRESULT && !Character.isDefined(resultI))) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultI));
            } else if (resultType == DOUBLERESULT) {
                sm.push(PrimitiveTypeFactory.createPrimitiveType(resultD));
            } else {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType((char) resultI));
            }
        }
        
        // hp105: do we have to distinguish chars from integers?
        public abstract int operation(int first, int second);
        public abstract int operation(int first, char second);
        public abstract double operation(double first, int second);
        public abstract double operation(double first, double second);
        public abstract double operation(double first, char second);
        
    }
    
    private static final IAtomicClosure add = new NumericOpsFactory.NumericOps("add") {

        public int operation(int first, int second) { return first + second; }
        public int operation(int first, char second) { return first + second; }
        public double operation(double first, int second) { return first + second; }
        public double operation(double first, double second) { return first + second; }
        public double operation(double first, char second) { return first + second; }
        
    };
    
    private static final IAtomicClosure subtract  = new NumericOpsFactory.NumericOps("subtract ") {

        public int operation(int first, int second) { return first - second; }
        public int operation(int first, char second) { return first - second; }
        public double operation(double first, int second) { return first - second; }
        public double operation(double first, double second) { return first - second; }
        public double operation(double first, char second) { return first - second; }
        
    };
    
    private static final IAtomicClosure multiply = new NumericOpsFactory.NumericOps("multiply") {

        public int operation(int first, int second) { return first * second; }
        public int operation(int first, char second) { return first * second; }
        public double operation(double first, int second) { return first * second; }
        public double operation(double first, double second) { return first * second; }
        public double operation(double first, char second) { return first * second; }
        
    };
    
    private static final IAtomicClosure divide = new NumericOpsFactory.NumericOps("divide") {

        public int operation(int first, int second) { return first / second; }
        public int operation(int first, char second) { return first / second; }
        public double operation(double first, int second) { return first / second; }
        public double operation(double first, double second) { return first / second; }
        public double operation(double first, char second) { return first / second; }
        
    };

    private static final IAtomicClosure modulus = new NumericOpsFactory.NumericOps("modulus") {

        public int operation(int first, int second) { return first % second; }
        public int operation(int first, char second) { return first % second; }
        public double operation(double first, int second) { return first % second; }
        public double operation(double first, double second) { return first % second; }
        public double operation(double first, char second) { return first % second; }
        
    };
    
    static private class IncDecOps extends AbstractAtomicClosure {
        private int value;        
        
        public IncDecOps(String name, int value) {
            super(name);
            this.value = value;
        }
        
        public void execute(IStackMachine sm) {
            IType arg0 = sm.pop();
            if (arg0 instanceof INumericType) {
                if (arg0 instanceof IntType) {
                    sm.push(PrimitiveTypeFactory
                            .createPrimitiveType(((INumericType) arg0)
                                    .getValue().intValue() + value));
                } else {
                    sm.push(PrimitiveTypeFactory
                            .createPrimitiveType(((INumericType) arg0)
                                    .getValue().doubleValue() + value));
                }
            } else if (arg0 instanceof ICharType) {
                sm.push(PrimitiveTypeFactory
                        .createPrimitiveType(((ICharType) arg0)
                                .getValue().charValue() + value));
            } else {
                throw new IllegalArgumentException(
                        "Unable to perform " + this._name + " operation on arg: " + arg0);
            }
        }
    }

    private static final IAtomicClosure inc = new NumericOpsFactory.IncDecOps("inc", 1);
    private static final IAtomicClosure dec = new NumericOpsFactory.IncDecOps("dec", -1);
    /**
     * The returned closure adds together the top two items on the stack and
     * place the result on the top of the stack.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguments are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure add() {
        return add;
    }

    /**
     * The returned closure subtracts the top two items on the stack and place
     * the result on the top of the stack. If the top item is arg0 and the next
     * item is arg1 then the result is arg0 - arg1.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguments are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure subtract() {
        return subtract;
    }

    /**
     * The returned closure multiplies together the top two items on the stack
     * and place the result on the top of the stack.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguments are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure multiply() {
        return multiply;
    }

    /**
     * The returned closure divides the top two items on the stack and place the
     * result on the top of the stack. If the top item is arg0 and the next item
     * is arg1 then the result is arg0 / arg1.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguments are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure divide() {
        return divide;
    }

    /**
     * The returned closure finds the remainder after division of the top two
     * items on the stack and place the result on the top of the stack. If the
     * top item is arg0 and the next item is arg1 then the result is arg0 %
     * arg1.
     * <p>
     * The top two items must both be INumericType instances. The resulting type
     * will be DoubleType if either of the arguments are DoubleType. Otherwise
     * the result will be IntType.
     * 
     * @return
     */
    public static IAtomicClosure modulus() {
        return modulus;
    }

    /**
     * The returned closure increments the top item of the stack. The top item
     * must be an INumbericType instance. The resulting type will be the same
     * type as the argument.
     * 
     * @return
     */
    public static IAtomicClosure inc() {
        return inc;
    }

    /**
     * The returned closure decrements the top item of the stack. The top item
     * must be an INumbericType instance. The resulting type will be the same
     * type as the argument.
     * 
     * @return
     */
    public static IAtomicClosure dec() {
        return dec;
    }
}