/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine.types;

import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IArrayType;
import uk.ac.imperial.doc.kenya.stackMachine.types.interfaces.IType;

/**
 * Basic implementation of the IArrayType interface. Calling setValue when a
 * value is already set will throw an error.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public class ArrayType implements IArrayType {

    
private static final String NULLNAME = "null";
    private IType[] value = null;

    private boolean assigned = false;

    public ArrayType() {
    }

    public ArrayType(IType[] array) {
        setValue(array);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IArrayType#setValue(org.wellquite.kenya.stackMachine.types.interfaces.IType[])
     */
    public final synchronized void setValue(IType[] value) {
        if (assigned && value != null) illegalAssignmentError(this);

        this.value = new IType[value.length];
        System.arraycopy(value, 0, this.value, 0, value.length);
        assigned = true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IArrayType#getValue()
     */
    public final synchronized IType[] getValue() {
        return value;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    public synchronized String toString() {
        return "ArrayType: " + value;
    }

    private static void illegalAssignmentError(ArrayType variable) {
        throw new RuntimeException("Attempt made to change value of type "
                + variable);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IType#isAssigned()
     */
    public synchronized boolean valueSet() {
        return assigned;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IType#valueToString()
     */
    public String valueToString() {
        if (value == null) {
            return NULLNAME;
        } else {
            StringBuffer out = new StringBuffer();
            for (int idx = 0; idx < value.length; idx++) {
                out.append(", ");
                if (value[idx] == null)
                    out.append(NULLNAME);
                else
                    out.append(value[idx].valueToString());
            }
            String result = "[Array of length " + value.length + ": ";
            if (out.length() > 0) result += out.toString().substring(2);
            result += "]";
            return result;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IArrayType#getLength()
     */
    public int getLength() {
        return getValue() == null ? 0 : getValue().length;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IArrayType#get(int)
     */
    public IType get(int index) {
        if (getValue() == null) return null;
        if (index >= getValue().length || index < 0)
                throw new IndexOutOfBoundsException(
                        "Attempt to access element " + index
                                + " in array of length " + getLength());
        return getValue()[index];
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.wellquite.kenya.stackMachine.types.interfaces.IArrayType#set(int,
     *      org.wellquite.kenya.stackMachine.types.interfaces.IType)
     */
    public void set(int index, IType newValue) {
        if (getValue() == null) return;
        if (index >= getValue().length || index < 0)
            throw new IndexOutOfBoundsException("Attempt to access element "
                    + index + " in array of length " + getLength());
        else
            getValue()[index] = newValue;
    }

    public boolean equals(Object obj) {
        if (obj instanceof IArrayType)
            return ((IArrayType) obj).getValue().equals(getValue());
        else
            return false;
    }

    public int hashCode() {
        if (getValue() == null)
            return NULLNAME.hashCode();
        else
            return getValue().hashCode();
    }
}