/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine.types.interfaces;

import java.util.Map;

/**
 * This interface descibes classes in the stack machine.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public interface IInterpretedClass {

    /**
     * Add a variable name and value to the class, declaring the variable as
     * being static and mutable.
     * 
     * @param name
     * @param variable
     */
    void addMutableStaticVariable(String name, IType variable);

    /**
     * Add a variable name and value to the class, declaring the variable as
     * being static and immutable.
     * 
     * @param name
     * @param variable
     */
    void addImmutableStaticVariable(String name, IType variable);

    /**
     * Gets a Map containing the class's mutable static variables. The Map
     * returned is a map of String to IType values. It is the actual Map used by
     * the class - so don't break it!
     * 
     * @return
     */
    Map<String,IType> getMutableStaticVariables();

    /**
     * Gets a Map containing the class's immutable static variables. The Map
     * returned is a map of String to IType values. It is the actual Map used by
     * the class - so don't break it!
     * 
     * @return
     */
    Map<String,IType> getImmutableStaticVariables();

    /**
     * Add a variable name and value to the class, declaring the variable as
     * being non-static (class instance scope) and mutable.
     * 
     * @param name
     * @param variable
     */
    void addMutableInstanceVariable(String name, IType variable);

    /**
     * Add a variable name and value to the class, declaring the variable as
     * being non-static (class instance scope) and immutable.
     * 
     * @param name
     * @param variable
     */
    void addImmutableInstanceVariable(String name, IType variable);

    /**
     * Add a method to the class. The method is declared as being static. The
     * method's isStatic() method must return true.
     * 
     * @param method
     */
    void addStaticMethod(IInterpretedMethod method);

    /**
     * Add a method to the class. The method is declared as being non-static
     * (class instance scope). The method's isStatic() method must return false.
     * 
     * @param method
     */
    void addInstanceMethod(IInterpretedMethod method);

    /**
     * Gets the method registered in this class under the following name (ie, by
     * using the methods' own getName() method). This will only search the
     * class's static methods.
     * 
     * @param name
     * @return
     */
    IInterpretedMethod getMethod(String name);

    /**
     * Creates a new instance of the class, providing the instance with all the
     * wholesome goodness (details on variables, methods etc) it needs.
     * 
     * @return
     */
    IInterpretedClassInstance createInstance();

    /**
     * Sets the class's name.
     * 
     * @param name
     */
    void setName(String name);

    /**
     * Gets the class's name.
     * 
     * @return
     */
    String getName();
}