/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004
 */
package uk.ac.imperial.doc.kenya.stackMachine.types.interfaces;

import java.util.Map;

import uk.ac.imperial.doc.kenya.stackMachine.IStackMachine;

/**
 * This interface describes methods as they exist within the stack machine.
 * Methods can not be put on the stack. Classes and Class instances can be put
 * on the stack.
 * 
 * @author Matthew Sackman (ms02)
 * @version 1
 */
public interface IInterpretedMethod {

    /**
     * Sets the name of the method. A method's name, one set can not be changed.
     * 
     * @param name
     */
    void setName(String name);

    /**
     * Gets the name of the method.
     * 
     * @return
     */
    String getName();

    /**
     * Sets whether or not the method has a return type. If set true, then when
     * the method has finished executing, the item on the top of the stack is
     * presented to the previously executing method as the return value.
     * 
     * @param hasReturnType
     */
    void setReturnType(boolean hasReturnType);

    /**
     * Indicates whether the method has a return type.
     * 
     * @return
     */
    boolean hasReturnType();

    /**
     * Sets the body of the method. This closure will be executed when the
     * method is invoked. Make sure you read the details of pop() in
     * StackMachine to understand how to obtain parameters that are passed to
     * methods.
     * 
     * @param body
     */
    void setMethodBody(IAtomicClosure body);

    /**
     * Gets the closure representing this method's body.
     * 
     * @return
     */
    IAtomicClosure getMethodBody();

    /**
     * Invokes this method. If no method body is set then an error is thrown.
     * The method body closure is executed with the supplied StackMachine
     * instance.
     * 
     * @param sm
     */
    void invoke(IStackMachine sm);

    /**
     * Add a variable name and value to the method, declaring the variable as
     * being mutable. This can be used to provide the method with variables with
     * known names and predefined values if the value is known at runtime,
     * rather than using a Factory to create the value and then storing that in
     * the method's scope.
     * 
     * @param name
     * @param variable
     */
    void addMutableMethodVariable(String name, IType variable);

    /**
     * Add a variable name and value to the method, declaring the variable as
     * being immutable. This can be used to provide the method with variables
     * with known names and predefined values if the value is known at runtime,
     * rather than using a Factory to create the value and then storing that in
     * the method's scope.
     * 
     * @param name
     * @param variable
     */
    void addImmutableMethodVariable(String name, IType variable);

    /**
     * Gets a Map containing the method's mutable variables. The Map returned is
     * a map of String to IType values. It is the actual Map used by the class -
     * so don't break it! Note that this Map is different from the Map
     * manipulated by IScope's storeNewVariable() method.
     * 
     * @return
     */
    Map<String,IType> getMutableMethodVariables();

    /**
     * Gets a Map containing the method's immutable variables. The Map returned
     * is a map of String to IType values. It is the actual Map used by the
     * class - so don't break it! Note that this Map is different from the Map
     * manipulated by IScope's storeNewVariable() method.
     * 
     * @return
     */
    Map<String,IType> getImmutableMethodVariables();

    /**
     * Indicates whether this method is static or not.
     * 
     * @return
     */
    boolean isStatic();

    /**
     * Set whether this method is static or not. Note that it's a stupidly bad
     * idea to change this after the method has been added to a class.
     * 
     * @param isStatic
     */
    void setStatic(boolean isStatic);

    /**
     * Gets the class containing this method. Note that a method only knows the
     * class from which it came, not the class instance. Use the
     * getCurrentClassInstance() method in IScope if you need to find which
     * class instance the current method is executing on.
     * 
     * @return
     */
    IInterpretedClass getInterpretedClass();

}