/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 19-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.styleCheckers;

import java.util.List;

import uk.ac.imperial.doc.kenya.passes.IJavaCode;
import uk.ac.imperial.doc.kenya.passes.IStackMachineInformationProvider;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IClass;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IFunction;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeError;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.ISourceCodeWarning;
import uk.ac.imperial.doc.kenya.sourceCodeInformation.interfaces.IVariable;

/**
 * This holds a representation of a Kenya Program and provides
 * utility methods to translate it into Java, and 
 * @author toa02
 */
public interface ICheckedCode {

    /**
     * @return The full unaltered String of the Kenya program
     * that has been checked
     */
    public String getKenyaSourceCode();
    
    /**
     * Lets you determine if this ICheckedCode instance represents
     * a valid Kenya Program.
     * 
     * @return true iff getErrors() will not throw an
     * IllegalStateException and (translate and getBaseStackMachine)
     * will. Returns false iff getErrors() will throw an IllegalStateException
     * and (translate and getBaseStackMachine) won't.
     */
    public boolean isErroredCode();
    
    /**
     * This method will return a java.util.List of all errors
     * that occured during processing of the Kenya.
     * If there where no errors (i.e. this ICheckedCode
     * represents a VALID program) this function WILL return
     * an IllegalStateException, use isErroredCode to determine
     * whether it is valid to call this.
     * 
     * @return List of kenya.sourceCodeInformation.ISourceCodeError
     * @throws IllegalStateException if used on code that has no errors.
     */
    public List<ISourceCodeError> getErrors();
    
    /**
     * This method will return a java.util.List of all informations
     * that occured during the processing of the Kenya.
     * Regardless of whether there where errors or not, this function
     * WILL return a java.util.List, that may contain no elements.
     * 
     * @return List of kenya.sourceCodeInformation.ISourceCodeWarning
     */
    public List<ISourceCodeWarning> getInfos();
    
    /**
     * This method will always return a list of all style check
     * results for this source code.  May be empty. 
     * @return
     */
    public List<IStyleCheckResult> getStyleCheckResults();
    
    /**
     * If this ICheckedCode instance represents code that is
     * valid, this method returns an IJavaCode instance of the
     * java code of this Kenya Code.
     * This method may throw an IllegalStateException if the 
     * underlying Kenya code had errors in it as this function
     * is not valid in that case.
     * 
     * @throws IllegalStateException if used on code that had errors
     */
    public IJavaCode translate();
    
    
    /**
     * If this ICheckedCode instance represents code that is
     * "valid" this will return a new IKenyaStackMachine ready for
     * interpreting upon.
     * This method may throw an IllegalStateException if the
     * underlying Kenya code had errors in it as this function
     * is not valid in that case. 
     * 
     * @throws IllegalStateException if used on code that
     * had errors.
     */
    public IStackMachineInformationProvider getBaseStackMachine();
    
    
    /**
     * This gives back a representation of all the classes that
     * have been defined within this code ( regardless of whether
     * there are errors or not ).
     *
     * @return An always non-null array of IClass representing the known
     * classes in this code.
     */
    public IClass[] getClasses();
    
    /**
     * This gives back a representation of all the enumerations that
     * have been defined within this code ( regardless of whether
     * there are errors or not ).
     *
     * @return An always non-null array of IEnum representing the known
     * enums in this code.
     */
    public IClass[] getEnums();
    
    /**
     * This gives back a representation of the constants that
     * have been defined within this code ( regardless of whether this
     * is errored code or not).
     * 
     * @return An always non-null array of IConstant representing the known
     * constants in this code.
     */
    public IVariable[] getConstants();
    
    /**
     * This gives back a representation of the known functions that
     * have been defined within this code ( regardless of whether this code
     * is errored or not ).
     * 
     * @return An always non-null array of IFunctions representing the known 
     * functions in this code.
     */
    public IFunction[] getFunctions();
    
}

