/*
 * Created on Sep 5, 2006
 */
package uk.ac.imperial.doc.kenya.styleCheckers.util;

import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Stack;

import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.minijava.node.Token;

public class Util {

    public static List<Method> getNodeAccessorMethods(Class< ? extends Node> c) {
        List<Method> methodList = new ArrayList<Method>();

        for (Method m : c.getMethods()) {
            if (m.getName().startsWith("get")
                    && Node.class.isAssignableFrom(m.getReturnType())
                    && m.getParameterTypes().length == 0) {
                methodList.add(m);
            }
        }
        return methodList;
    }

    /**
     * Takes the node instance, and populates all its Token fields
     * with new tokens that are also added to the generalNodes set
     * @param node
     * @param generalNodes
     */
    public static void populateTokens(Node node, Set<Node> generalNodes) {
        try {
            List<Method> methods = getNodeAccessorMethods(node.getClass());
            
            for(Method m : methods) {
                if(!Token.class.isAssignableFrom(m.getReturnType())) {
                    continue;
                }
                String setterName = m.getName().replaceFirst("^get","set");
                Method setter = node.getClass().getMethod(setterName,m.getReturnType());
                
                Token newTokenInstance = (Token) m.getReturnType().newInstance();
                setter.invoke(node,newTokenInstance);
                generalNodes.add(newTokenInstance);
            }
        } catch(Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Fills in a chain of nodes from the start to the end, working on the
     * assumption that there is only one route to get from the start to the
     * end type of nodes, and that no intermediate nodes take more than one child.
     * @param startNode
     * @param endNode
     */
    public static void fillInSimpleChain(Node startNode,
            Node endNode) {
        
        Stack<Class<? extends Node>> route = findRoute(startNode.getClass(), endNode.getClass());
        
        if(route == null) {
            throw new IllegalArgumentException("There is no route from " + startNode.getClass() + " to " + endNode.getClass());
        }
        

        try {
            /* Build a list of instances */
            
            // We already have the first item
            route.pop();
            
            List<Node> instances = new ArrayList<Node>();
            instances.add(startNode);
            
            while(route.size() > 1) {
                Class<? extends Node> c = route.pop();
                instances.add(c.newInstance());
            }
            
            // We already have the last item        
            instances.add(endNode);
    
            /* Now glue the instances together */
            Method setter = null;
            Node previous = null;
            for(Node n : instances) {
                if(setter != null) {
                    setter.invoke(previous, n);
                }
                setter = getSingleSetterFor(n.getClass());
                previous = n;
            }
        } catch (Exception e ) {
            throw new RuntimeException(e);
        }
    }

    private static Stack<Class<? extends Node>> findRoute(
            Class<? extends Node> startClass, Class<? extends Node> endClass) {

        Method m = getSingleSetterFor(startClass);
        
        /* Failure case */
        if(m == null) {
            /* no single setter here, therefore, no route */
            return null;
        }
        
        /* Success case */
        @SuppressWarnings("unchecked")
        Class<Node> nextClass = (Class<Node>)m.getParameterTypes()[0];
        
        if(nextClass.isAssignableFrom(endClass)) {
            Stack<Class<? extends Node>> ret = new Stack<Class<? extends Node>>();
            ret.push(endClass);
            ret.push(startClass);
            return ret;
        }
        
        /* Recursive case */
        Set<Class<? extends Node>> possibleSubclasses = NodeDatabase.getInstance().getSubclasses().get(nextClass);
        
        /* No possible subclasses */
        if(possibleSubclasses == null) {
            return null;
        }
        
        Set<Class<? extends Node>> concreteSubclasses = new HashSet<Class<? extends Node>>(possibleSubclasses);
        concreteSubclasses.retainAll(NodeDatabase.getInstance().getConcreteClasses());
        
        for(Class<? extends Node> c : concreteSubclasses) {
            
            Stack<Class<? extends Node>> nextRoute = findRoute(c,endClass);
            if(nextRoute != null) {
                nextRoute.push(startClass);
                return nextRoute;
            }
        }
  
        /* Total failure */
        return null;
        
    }

    private static Method getSingleSetterFor(Class<? extends Node> class1) {

        Method singleSetter = null;
        
        for(Method m : class1.getMethods()) {
            if(m.getName().startsWith("set")) {
                if(singleSetter == null) {
                    singleSetter = m;
                } else {
                    return null;
                }
            }
        }
        
        return singleSetter;
    }

    public static CodeSpan getStartAndEndOfNode(Node n) {
        int[] start = uk.ac.imperial.doc.kenya.passes.Util.getFirstIdent(n);
        int[] end = uk.ac.imperial.doc.kenya.passes.Util.getLastIdent(n);
        
        return new CodeSpan(start[0], start[1], end[0], end[1]);
    }
    
    
    public static String drainReader(Reader in) {
        try {
            StringBuilder sb = new StringBuilder();
            
            char[] buf = new char[512];
            int read = -1;
            while((read = in.read(buf)) != -1) {
                sb.append(buf,0,read);
            }
            
            return sb.toString();
        } catch (IOException e) {
            return "";
        }
    }
    
    
}
