/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 01-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.types;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import uk.ac.imperial.doc.kenya.errors.KenyaInternalError;
import uk.ac.imperial.doc.kenya.minijava.node.AFuncDecDeclaration;
import uk.ac.imperial.doc.kenya.types.tables.SymbolTable;


/**
 * @author toa02
 * The representation of a function.
 */
public class KFunction{

	private final String _name; 		/* The name of this function */
	private final KType _retType; 	/* Return type of this function */
	private final Map<String,KParamType> _params; 		/* The String->KParamType Template parameters to this KFunction */
	private final List<KType> _args;	 		/* List of KTypes, the varaible's types. */
	private final List<KVariable> _vars; 		/* List of KVariables, i.e. the variable idents -
												could be null if a builtin method. */
	private final IBuiltInMethod _bim;		/* Backing method if it is builtin */ 
	private final AFuncDecDeclaration _node; 	/* Node declared on */
	/** 
	 * @param name
	 * @param retType
	 * @param params the type parameters. (see above)
	 * @param args list of KTypes,
	 * @param vars list of KVaribles (pre: same types as the KTypes or null if builtin)
	 */
	public KFunction( String name, Map<String,KParamType> params,  KType retType, List<KType> args, List<KVariable> vars, IBuiltInMethod method, AFuncDecDeclaration node){
		_name = name;
		_retType = retType;
		_node = node;
		_bim = method;
		
		if( params == null ){
		    _params = new HashMap<String,KParamType>();
		}else{
		    _params = params;
		}
		
		if( args == null ){
			_args = new LinkedList<KType>();
			_vars = new LinkedList<KVariable>();
		}else{
			_args = args;
			_vars = vars;
		}
	}
	
	public boolean equals(Object o){
		if( o instanceof KFunction){
			KFunction kf = (KFunction) o;
			return (
						kf._name.equals(_name) &&
						kf._args.equals(_args)
					);
		}
		
		return false;
	}
	
	public int hashCode(){
		return (_name.hashCode() + _args.hashCode());
	}
	
	/**
	 * Get the return type of this function.
	 * @return KType representing the return type of this function.
	 */
	public KType getReturnType(){
	    return _retType;
	}
	

	public IBuiltInMethod getBuiltinMethod(){
	    return _bim;
	}
	
	/**
	 * Does this KFunction represent a builtin function?
	 * This is determined by whether the _vars list is null.
	 * @return true iff this is the representation of a builtin function.
	 */
	public boolean isBuiltin(){
	    return( _bim != null);
	}
	
	/**
	 * Gives back the name of this function. This CANNOT be used as a unique identifier.
	 * @return The String name
	 */
	public String getName(){
	    return _name;
	}
	
	/**
	 * Gives back the args list of this function.
	 * @return A List of KTypes representing the argument (types) to this function.
	 */
	public List<KType> getArgs(){
	    return _args;
	}
	
	/**
	 * Gives back the vars list of this function.
	 * @return List of KVaraibles
	 */
	public List<KVariable> getVars(){
	    return _vars;
	}

	/**
	 * Gives back a SymbolTable representing the "symbols" defined by the arguments to this KFunction.
	 * @return
	 */
	public SymbolTable getSymbolTable(){
	    if( _vars == null ){
	        throw new KenyaInternalError("Attempting to get SymbolTable for KFunction with null arguments");
	    }
	    
	    SymbolTable st = new SymbolTable();
	    for (KVariable kv: _vars)
	        st.push(kv.getName(),kv);
	    
	    return st;
	}
	
	/**
	 * Gives back the String Name->KParamType map of Template Type parameters that
	 * are present in this function.
	 * @return Map of String -> KParamTypes
	 */
	public Map<String,KParamType> getTypeParamMap(){
	    return _params;
	}
	
	/**
	 * Gives back the Function Declaration Node on the sablecc ast tree.
	 * @return AFuncDecDeclaration node that refers to this node on the tree.
	 */
	public AFuncDecDeclaration getNode(){
	    return _node;
	}

	/**
	 * Returns a String represntation of this function, along with its arguments and
	 * parameterized types (if it has them).
	 * @see java.lang.Object#toString()
	 */
	public String toString(){
	    StringBuffer sb = new StringBuffer();
	    Iterator<String> it = _params.keySet().iterator();
	    
	    sb.append("<");
	    while(it.hasNext()){
	        sb.append(it.next());
	        
	        if(it.hasNext()){
	            sb.append(", ");
	        }
	    }
	    sb.append("> ");
	    
	    sb.append(_retType.getName());
	    
	    sb.append(" ");
	    sb.append(_name);
	    sb.append("(");
	    
	    if( _vars != null){
	        Iterator<KVariable> it2 = _vars.iterator();
	    
	        while(it2.hasNext()){
	            
	            sb.append( it2.next().toString());
	            
	            if(it2.hasNext()){
	                sb.append(", ");
	            }
	        }
	    }else{
	        Iterator<KType> it3 = _args.iterator();
	        
	        while(it3.hasNext()){
	            
	            sb.append(it3.next().getName());

	            if(it3.hasNext()){
	                sb.append(", ");
	            }
	        }
	        
	    }
	    
	    sb.append(")");
	    if( _vars == null){
	        sb.append("*");
	    }

	    return sb.toString();
	}
}
