/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 30-Jun-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.types;

import java.util.Iterator;
import java.util.List;
import java.util.Map;

import uk.ac.imperial.doc.kenya.errors.KenyaInternalError;
import uk.ac.imperial.doc.kenya.minijava.node.Node;
import uk.ac.imperial.doc.kenya.types.tables.TypeTable;


/**
 * @author toa02
 *
 */
public class KParamType extends KType {

	private final String _name;
	private Node _node;
	
	public KParamType(String name, Node declNode){
		_name = KParamTypeMapper.get().mapName(name);
		_node = declNode;
	}
	
	public KParamType(String name){
		_name = KParamTypeMapper.get().mapName(name);
		_node = null;
	}
	
	public KParamType(String name, boolean flag){
	    _name = name;
	    _node = null;
	}
	
	public Node getNode(){
	    if( _node == null ){
	        throw new KenyaInternalError("Getting a null node.");
	    }
	    return _node;
	}
		
	/* A parameterized type is exactly equal to only itself 
	 * @see kenya.types.KType#exactlyMatches(kenya.types.KType)
	 */
	public boolean exactlyMatches(KType kt) {
		if( kt instanceof  KParamType ){
			KParamType kpt = (KParamType) kt;
			return ( kpt._name.equals(_name));
		}
		
		return false;
	}
	
	public KType bind( TypeTable tt){
	    if( tt.containsKey(this)){
	    	return tt.lookup(this);
	    }
	    
	    return this;
	}

	/**
	 * @see uk.ac.imperial.doc.kenya.types.KType#getName()
	 */
	public String getName() {
		return _name;
	}
	
	
	public boolean isTypeMatch( KType kt ) {
		return true;
	}
	
	
	/**
	 * @see uk.ac.imperial.doc.kenya.types.KType#compareAndBind(uk.ac.imperial.doc.kenya.types.KType, uk.ac.imperial.doc.kenya.types.tables.TypeTable)
	 * Quite a few random subtlties her to do with the behaviour of paraterised functions
	 * that call parameterised functions. All very messy.
	 */
	public int compareAndBind(KType target, TypeTable tt) {

	    if( tt.containsKey(this)){
	        KType kt = tt.lookup(this);

	        /*
	        if( Util.isAssignmentCompatible(target, kt) ){
                tt.force(this, target);
            }else if( Util.isAssignmentCompatible(kt,target)){
                tt.force(this,kt);
            }
	        */

	        //* * Int double char conversions 
	        if( target == KBasicType.getDouble() && isAssignmentCompatible(target, kt)){
	                tt.force(this,KBasicType.getDouble());
	        }else if( target == KBasicType.getInt() && isAssignmentCompatible(target, kt)){
	                tt.force(this,KBasicType.getInt());
	        }
	        
	        // In both directions
	        if( kt == KBasicType.getDouble() && isAssignmentCompatible(kt,target)){
	                tt.force(this,KBasicType.getDouble());
	        }else if( kt == KBasicType.getInt() && isAssignmentCompatible(kt, target)){
	                tt.force(this,KBasicType.getInt());
	        }

	        
	        if( kt instanceof KBoundClassType ){
	            KBoundClassType kbct = (KBoundClassType) kt;
	            if( kbct.getBase() == KClassType.getDouble() && isAssignmentCompatible(KBasicType.getDouble(), target) ){
	                tt.force( this, kbct );
	            }else if( kbct.getBase() == KClassType.getInteger() && isAssignmentCompatible(KBasicType.getInt(), target) ){
	                tt.force( this, kbct );
	            }
	        }
	        
	        if( target instanceof KBoundClassType ){
	            KBoundClassType kbct = (KBoundClassType) target;
	            if( kbct.getBase() == KClassType.getDouble() && isAssignmentCompatible(KBasicType.getDouble(),kt) ){
	                tt.force( this, kbct );
	            }else if( kbct.getBase() == KClassType.getInteger() && isAssignmentCompatible(KBasicType.getInt(),kt) ){
	                tt.force( this, kbct );
	            }
	        }
	        
	        if( kt instanceof KParamType){
	            if(((KParamType) kt).exactlyMatches(target)){
	                return 0;
	            }else{
	                return -1;
	            }
	        }
	        
	        int res = kt.compareAndBind(target, tt);
	        if( res == - 1 ){ return -1; }
	        
	        if( kt instanceof KArrayType){
	            KArrayType klt = (KArrayType) kt;
	            return 1 + klt.getDepth() + res;
	        }else{
	            return 1+res;    
	        }

	    } else if( target == KNullType.get() ){
	    	return 1; /* 1 point bind? */
	    }else{
	        tt.put(this, target);
	        return 1;
	    }
	}
	
	public String toString(){
		return KParamTypeMapper.get().lookupName(_name);
	}

	
	/**
     * @see uk.ac.imperial.doc.kenya.types.KType#isBound(java.util.Map)
     */
    public boolean isBound(Map<String,KType> paramMap) {
        return( paramMap.containsKey(KParamTypeMapper.get().lookupName(getName())));
    }
	
    /**
     * @see uk.ac.imperial.doc.kenya.types.KType#populateParamMap(java.util.Map)
     */
    public void populateParamMap(Map<String, KType> paramMap) {
        paramMap.put(KParamTypeMapper.get().lookupName(_name),this);
    }
    
	 /**
     * Checks if two types are assignment compatible.
     * This assumes that a is being assigned to the type of b. ie. a = b ;
     * So basically they must be identical, unless a is double and b is an int/char.
     * @param a The Left hand side of this assignment.
     * @param b The Right hand side of this assignment.
     */
    public static boolean isAssignmentCompatible(KType a, KType b){
        // Low hanging fruit.
        // cannot assign a void to anything
        if( a == KBasicType.getVoid() ) {
            return false;
        }
        
        if( a == b) { return true; }
        
        if( a instanceof KArrayType ){
            if( b instanceof KArrayType ){
                KArrayType kat = (KArrayType) a;
                KArrayType kbt = (KArrayType) b;

                KType childA = kat.getChildType();
                KType childB = kbt.getChildType();
                
                if( childA instanceof KBasicType && childB instanceof KBasicType ){
                    return childA == childB;
                }else if( childA instanceof KBasicType && childB instanceof KBoundClassType ){
                    return false;
                }else if( childA instanceof KBoundClassType && childB instanceof KBasicType){
                    return false;
                }else if( childA instanceof KBoundClassType && childB instanceof KBoundClassType){
                    KBoundClassType kbA = (KBoundClassType) childA;
                    KBoundClassType kbB = (KBoundClassType) childB;
                    
                    if( kbA.getBase() != kbB.getBase()){
                        return false;
                    }
                }
                
                return isAssignmentCompatible(childA,childB);
                
            }else if( b instanceof KNullType){
                return true;
            }else{
                return false;
            }
        
        }else if( a instanceof KBasicType ){
        	if( a == KBasicType.getString() ) {
        		return b == KBasicType.getString() || b == KNullType.get();
        	} else if( b instanceof KBasicType ){
	            return (
	                	( a == KBasicType.getDouble() && 
	                	        ( b == KBasicType.getInt() || b == KBasicType.getChar() )
	                	) ||
	                	( a == KBasicType.getInt() && b == KBasicType.getChar() )
	            );
            }else if( b instanceof KBoundClassType ){
                KBoundClassType kbB = (KBoundClassType) b;
                return isAssignmentCompatible(a, kbB.getBase());
                
            }else if( b instanceof KClassType ){
                return (
	                	( a == KBasicType.getDouble() && 
	                	        ( b == KClassType.getInteger() || b == KClassType.getCharacter() )
	                	) ||
	                	( a == KBasicType.getInt() && b == KClassType.getCharacter() ) ||
	                	a == KBasicType.getBoolean() && b == KClassType.getBoolean() ||
	                	a == KBasicType.getChar() && b == KClassType.getCharacter() ||
	                	a == KBasicType.getInt() && b == KClassType.getInteger() ||
	                	a == KBasicType.getDouble() && b == KClassType.getDouble()
	                );
            }else{
                return false;
            }
            
            
        }else if( a instanceof KBoundClassType ){
            KBoundClassType kbA = (KBoundClassType) a;
            if( b instanceof KBoundClassType){
                
                KBoundClassType kbB = (KBoundClassType) b;
                
                if( ! isAssignmentCompatible(kbA.getBase(), kbB.getBase())){
                    return false;
                }
                
                boolean isOk = true;
                
                List<KParamType> params = kbA.getBase().getTypeParamList();
                TypeTable ttA = kbA.getBindings();
                TypeTable ttB = kbB.getBindings();
                
                Iterator<KParamType> it = params.iterator();
                while(it.hasNext()){
                    KParamType kpt = (KParamType) it.next();
                    KType cA = ttA.lookup(kpt);
                    KType cB = ttB.lookup(kpt);
                    isOk &= isAssignmentCompatible(cA, cB);
                }
                return isOk;
            }else if( b instanceof KBasicType ){
                return isAssignmentCompatible(kbA.getBase(), b);
            }else if( b instanceof KNullType){
                return true;
            }else{
                return false;
            }
        }else if( a instanceof KClassType ){
            if( b instanceof KClassType ){
             /*	Evidently this isn't allowed
                if(
             	( a == KClassType.getDouble() && 
            	        ( b == KClassType.getInteger() || b == KClassType.getCharacter() )
            	) ||
            	( a == KClassType.getInteger() && b == KClassType.getCharacter() ) 
            	){
             	    return true;
             	}else 
             	*/
             	if( a.getName().equals(b.getName())){
             	    return true;
             	}else{
             	    return false;
             	}
            }else if( b instanceof KBasicType ){
                if(
                /*	Evidently this isn't allowed
                     	( a == KClassType.getDouble() && 
                    	        ( b == KBasicType.getInt() || b == KBasicType.getChar() )
                    	) ||
                    	( a == KClassType.getInteger() && b == KBasicType.getChar() ) ||
                 */
                    	a == KClassType.getBoolean() && b == KBasicType.getBoolean() ||
	                	a == KClassType.getCharacter() && b == KBasicType.getChar() ||
	                	a == KClassType.getInteger() && b == KBasicType.getInt() ||
	                	a == KClassType.getDouble() && b == KBasicType.getDouble()){
                     	    return true;
                }else{
                    return false;
                }
            }else{
                return false;
            }
            
        }else if( a instanceof KEnumType ){
            if( b instanceof KEnumType ){
                return b.getName().equals(a.getName());
            } else if( b instanceof KNullType ){
                return true;
            } else {
                return false;
            }
            
        }else if( a instanceof KParamType ){
            if( b instanceof KNullType ){
                return true;
            }else if( b instanceof KParamType){
               return b.getName().equals(a.getName());
            }else{
                return false;
            }
            
        }else if( a instanceof KNullType ){
        	return false;
        }else{
            throw new KenyaInternalError("Got an unknown type " + a);
        }
        
/*        
        if( a instanceof KArrayType && b instanceof KArrayType){
            KArrayType kat = (KArrayType) a;
            KArrayType kbt = (KArrayType) b;
            
            KType childA = kat.getChildType();
            KType childB = kbt.getChildType();
            if( childA instanceof KBasicType && childB instanceof KBasicType ){
                //A double[] cannot be assigned to an int[] etc. Hence this extra condition.
                return childA == childB;
            }else{
                return isAssignmentCompatible(childA, childB);
            }
        }else if( a instanceof KArrayType && b instanceof KNullType ){
            return true;
        }else if( a instanceof KArrayType ){ 
            return false; 			
        }else if( a instanceof KBasicType && b instanceof KBasicType){
            // If they were both the same basic type, they would have passed the LHF above.:)
            return (
                    	( a == KBasicType.getDouble() && 
                    	        ( b == KBasicType.getInt() || b == KBasicType.getChar() )
                    	) ||
                    	( a == KBasicType.getInt() && b == KBasicType.getChar() )
                    );
        }else if( a == KBasicType.getString() && b instanceof KEnumType){
            return true;
        }else if( a instanceof KBasicType ){
            return false;	// They arn't both of basic type, since failed one above.
        }else if( a instanceof KBoundClassType ){
            if( b instanceof KBoundClassType){
                KBoundClassType kbA = (KBoundClassType) a;
                KBoundClassType kbB = (KBoundClassType) b;
                
                if(! kbA.getBase().getName().equals(kbB.getBase().getName()) ){ return false; }
                
                boolean isOk = true;
                
                List params = kbA.getBase().getTypeParamList();
                TypeTable ttA = kbA.getBindings();
                TypeTable iottB = kbB.getBindings();
                
                Iterator it = params.iterator();
                while(it.hasNext()){
                    KParamType kpt = (KParamType) it.next();
                    KType cA = ttA.lookup(kpt);
                    KType cB = ttB.lookup(kpt);
                    isOk &= isAssignmentCompatible(cA, cB);
                }
                
                return isOk;
            }else if( b instanceof KNullType){
                return true;
            }else if( b instanceof KBasicType && b != KBasicType.getString()){
                return isAssignmentCompatible(a, ((KBasicType)b).bind(new TypeTable()));
            }else{
                return false;
            }
        }else if( a instanceof KClassType && b instanceof KClassType){
            return (a.getName().equals(b.getName()) );

        }else if( a instanceof KEnumType ){
            return ( b instanceof KEnumType && b.getName().equals(a.getName()));
        }else if( a instanceof KParamType ){
            // Since a parameterised type could represent a basic type,
            // you can't assign them to null.
            return ((b instanceof KParamType) && b.getName().equals(a.getName())));
        }else if( a instanceof KNullType){
            return false; // Erm. null = ? ...shouldn't really happen. null is a state.
        }else{
            throw new KenyaInternalError(a + " is not a legal KType!");
        }
*/
    }
}
