/* *******************************************************************************
 *   Kenya                                                                       *
 *   Copyright (C) 2004 Tristan Allwood,                                         *
 *                 2004 Matthew Sackman                                          *
 *                                                                               *
 *   This program is free software; you can redistribute it and/or               *
 *   modify it under the terms of the GNU General Public License                 *
 *   as published by the Free Software Foundation; either version 2              *
 *   of the License, or (at your option) any later version.                      *
 *                                                                               *
 *   This program is distributed in the hope that it will be useful,             *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
 *   GNU General Public License for more details.                                *
 *                                                                               *
 *   You should have received a copy of the GNU General Public License           *
 *   along with this program; if not, write to the Free Software                 *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *
 *                                                                               *
 *   The authors can be contacted by email at toa02@doc.ic.ac.uk                 *
 *                                             ms02@doc.ic.ac.uk                 *
 *                                                                               *
 *********************************************************************************/

/*
 * Created on 05-Jul-2004 by toa02
 *
 */
package uk.ac.imperial.doc.kenya.types.tables;

import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Set;

import uk.ac.imperial.doc.kenya.errors.SourceCodeException;
import uk.ac.imperial.doc.kenya.minijava.node.AClassDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.AEnumDecDeclaration;
import uk.ac.imperial.doc.kenya.minijava.node.PDeclaration;
import uk.ac.imperial.doc.kenya.types.KBasicType;
import uk.ac.imperial.doc.kenya.types.KClassType;
import uk.ac.imperial.doc.kenya.types.KEnumType;
import uk.ac.imperial.doc.kenya.types.KType;
import uk.ac.imperial.doc.kenya.types.KTypeSourceCodeException;


/**
 * Holds mappings for ClassNames/Enums (String) to KClassTypes/KEnumType and also
 * to their nodes on the AST. Also holds all the basic types with mappings from
 * their string names.
 * @author toa02
 *
 */
public class ClassTable{

    private HashMap<String,KClassType> _classType = new HashMap<String,KClassType>(); 
    private HashMap<String,AClassDecDeclaration> _classNode = new HashMap<String,AClassDecDeclaration>();

    private HashMap<String,KEnumType> _enumType = new HashMap<String,KEnumType>();
    private HashMap<String,AEnumDecDeclaration> _enumNode = new HashMap<String,AEnumDecDeclaration>();
    
    private HashMap<String,KBasicType> _basicType = new HashMap<String,KBasicType>();
    
    
    public ClassTable(){
		/* add basic types to the _types set */
		_basicType.put("int", KBasicType.getInt());
		_basicType.put("double", KBasicType.getDouble());
		_basicType.put("char", KBasicType.getChar());
		_basicType.put("String", KBasicType.getString());
		_basicType.put("boolean", KBasicType.getString());
		_basicType.put("void", KBasicType.getVoid());
    }
    
    private int[][] getLnkPos(AClassDecDeclaration node)
    {
        int ln = node.getIdentifier().getLine();
        int pos = node.getIdentifier().getPos();
        int len = node.getIdentifier().getText().trim().length();
        int[][] lnkPos = { { ln, pos, len }};
        return lnkPos;
    }
    
    private int[][] getLnkPos(AEnumDecDeclaration node)
    {
        int ln = node.getIdentifier().getLine();
        int pos = node.getIdentifier().getPos();
        int len = node.getIdentifier().getText().trim().length();
        int[][] lnkPos = { { ln, pos, len }};
        return lnkPos;
    }
    
    /**
     * Associates a name with a ClassType and its node. (does the error checking as well :) )
     * @param name The name of this class
     * @param kt The classType node
     * @param node The node on the ast.
     * @throws SourceCodeExcepion if the name is already in either the class or the enum map.
     */    
    public void put(String name, KClassType kt, AClassDecDeclaration node ) throws SourceCodeException{
        
        if(_basicType.containsKey(name)){
        	int[][] lnkPosNode = getLnkPos(node);
            KTypeSourceCodeException.throwClassConflictBasicType(lnkPosNode[0][0], lnkPosNode[0][1], lnkPosNode[0][2], kt);
        }
        
        // Check the error cases where this class type has already been defined.
        if (_classType.containsKey(name) ){
            AClassDecDeclaration otherNode = (AClassDecDeclaration) _classNode.get(name);
            if( otherNode != null ){ 
	        	int[][] lnkPosNode = getLnkPos(node);
	            int[][] lnkPosOtherNode = getLnkPos(otherNode);
	            KTypeSourceCodeException.throwDuplicateClassName(lnkPosNode[0][0], lnkPosNode[0][1], lnkPosNode[0][2], lnkPosOtherNode, kt);
            }
        }

        // Check the error cases where this class type has already been defined.        
        if (_enumType.containsKey(name) ){
            AEnumDecDeclaration otherNode = (AEnumDecDeclaration) _enumNode.get(name);
        	int[][] lnkPosNode = getLnkPos(node);
            int[][] lnkPosOtherNode = getLnkPos(otherNode);
            KTypeSourceCodeException.throwDuplicateClassEnumName(lnkPosNode[0][0], lnkPosNode[0][1], lnkPosNode[0][2], lnkPosOtherNode, kt);
        }
        
        _classType.put(name,kt);
        _classNode.put(name,node);
    }
        
    /**
     * Associates a name with an EnumType and its node.
     * @param name The name of this Enum
     * @param et The KEnumType node
     * @param node The node on te ast.
     * @throws SourceCodeException if the name is in either the class or enum map. 
     */
    public void put(String name, KEnumType et, AEnumDecDeclaration node) throws SourceCodeException{
       
        if(_basicType.containsKey(name)){
        	int[][] lnkPosNode = getLnkPos(node);
            KTypeSourceCodeException.throwEnumConflictBasicType(lnkPosNode[0][0], lnkPosNode[0][1], lnkPosNode[0][2], et);
        }
        
        // Check the error cases where this class type has already been defined.
        if (_classType.containsKey(name) ){
            AClassDecDeclaration otherNode = (AClassDecDeclaration) _classNode.get(name);
            int[][] lnkPosNode = getLnkPos(node);
            int[][] lnkPosOtherNode = getLnkPos(otherNode);
            KTypeSourceCodeException.throwDuplicateEnumClassName(lnkPosNode[0][0], lnkPosNode[0][1], lnkPosNode[0][2], lnkPosOtherNode, et);
        }

        // Check the error cases where this class type has already been defined.        
        if (_enumType.containsKey(name) ){
            AEnumDecDeclaration otherNode = (AEnumDecDeclaration) _enumNode.get(name);
            int[][] lnkPosNode = getLnkPos(node);
            int[][] lnkPosOtherNode = getLnkPos(otherNode);
            KTypeSourceCodeException.throwDuplicateEnumName(lnkPosNode[0][0], lnkPosNode[0][1], lnkPosNode[0][2], lnkPosOtherNode, et);
        }
        
        _enumType.put(name,et);
        _enumNode.put(name,node);
    }
    
    /**
     * Returns true if there is a mapping for this class name in one of the tables.
     * @param name The name to see if it exists.
     * @return True iff there is a class with that name in either the Enum or Class table.
     */
    public boolean containsMapping(String name){
        return( _basicType.containsKey(name) || _classType.containsKey(name) || _enumType.containsKey(name) );
    }

    /**
     * Checks if the given String maps to a Basic Type.
     * @param name The name to check.
     * @return true iff the given name maps to a Basic Type.
     */
    public boolean isBasicType(String name){
        return( _basicType.containsKey(name));
    }
    
    /**
     * Checks if the given String maps to an Enum Type.
     * @param name The name to check.
     * @return true iff the given name maps to a Enum Type.
     */
    public boolean isEnum(String name){
        return (_enumType.containsKey(name));
    }
    
    /**
     * Checks if the given String maps to a Class Type.
     * @param name The name to check.
     * @return true iff the given name maps to a ClassType.
     */
    public boolean isClass(String name){
        return (_classType.containsKey(name));
    }
    
    /**
     * Returns the KType of the given class.
     * @param name The name to look up.
     * @return Its KType.
     * @throws NoSuchElementException if the class name is not in the table.
     */
    public KType getType(String name) throws NoSuchElementException{
        // Invaraint to the method, only one of classType or enumType contain this mapping.
        
        if( _basicType.containsKey(name)){
            return (KType) _basicType.get(name);
        }else if( _classType.containsKey(name)){
            return (KType) _classType.get(name);
        }else if( _enumType.containsKey(name)){
            return (KType) _enumType.get(name);
        }
        
        throw new NoSuchElementException("Name is not a type that exists.");
    }
    
    public PDeclaration getNode(String name) throws NoSuchElementException{
//      Invaraint to the method, only one of classType or enumType contain this mapping.
        
        if( _classNode.containsKey(name)){
            return (PDeclaration) _classNode.get(name);
        }else if( _enumNode.containsKey(name)){
            return (PDeclaration) _enumNode.get(name);
        }
        
        throw new NoSuchElementException(name + " is not a type that exists with a node.");
    }
    
    /**
     * @return Iterator to the string names of the class types iterator.
     */
    public Set<String> getClasses(){
        return _classType.keySet();
    }
    
    /**
     * @return Iterator to the String names of the enum types iterator.
     */
    public Set<String> getEnumTypes(){
        return _enumType.keySet();   
    }
    
    /**
     * Returns a (formatted) string representation of the entire class.
     * @see java.lang.Object#toString()
     */
    public String toString(){
        StringBuffer sb = new StringBuffer();
        

        sb.append("ClassTypes:");
//        sb.append(NewLineClass.NEWLINE);
        Iterator<String> it = _classType.keySet().iterator();
        while(it.hasNext()){
            String name = (String) it.next();
            KClassType kct = (KClassType) _classType.get(name);
            sb.append("\t");
            sb.append(kct);
            if( it.hasNext() ){
                sb.append(";");
                sb.append(NewLineClass.NEWLINE);
            }
        }
        
        sb.append(NewLineClass.NEWLINE);
        sb.append("EnumTypes:");
        sb.append(NewLineClass.NEWLINE);
        it = _enumType.keySet().iterator();
        while(it.hasNext()){
            String name = (String) it.next();
            KEnumType kct = (KEnumType) _enumType.get(name);
            sb.append("\t");
            sb.append(kct);
            if( it.hasNext() ){
                sb.append(";");
                sb.append(NewLineClass.NEWLINE);
            }
        }
        
        sb.append(NewLineClass.NEWLINE);
        sb.append("BasicTypes:");
        sb.append(NewLineClass.NEWLINE);
        it = _basicType.keySet().iterator();
        while(it.hasNext()){
            String name = (String) it.next();
            KBasicType kct = (KBasicType) _basicType.get(name);
            sb.append("\t");
            sb.append(kct);
            if( it.hasNext() ){
                sb.append(";");
                sb.append(NewLineClass.NEWLINE);
            }
        }
        
        return sb.toString();
    }
}
